/* Agent Tcl
   Bob Gray
   9 February 1995

   agentd.cc

   This is the top-level program for the agent server.  On startup it forks
   the process "socketd" which monitors the agent port.  Then its task is to
   maintain the data structures that keep track of agents executing on its
   machine.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "platDelete.h"
#include "platInterrupt.h"
#include "platSystem.h"
#include "platTimeval.h"
#include "suppFileTable.h"
#include "suppStrings.h"
#include "servConf.h"		// ServerData
#include "servRandom.h"		// RandomFile
#include "agent_set.h"
#include "agentd.h"
#include "agentId.h"
#include "genDaemon.h"
#include "genEncrypt.h"
#include "genError.h"
#include "genFile.h"
#include "genPipe.h"
#include "suppHash.h"
#include "genManager.h"
#include "mesgTcpip.h"
#include "hotInterpreter.h"
#include "interp.h"
#include "parse.h"
#include "piped.h"
#include "noise.h"
#include "random.h"
#include "randpool.h"
#include "socketd.h"
#include "truefalse.h"

    /* external signal handlers */

extern void sighup_handler (int, SignalHandlers::ClientData);
extern void sigchld_handler (int, SignalHandlers::ClientData);
extern void coredump_handler (int, SignalHandlers::ClientData);

    /* defaults are to (1) run the server as a daemonm, (2) verify the */
    /* hostname that was specified after the -host command-line flag,  */
    /* and (3) not truncate the log                                    */

const BOOLEAN DEFAULT_DAEMON	      = e_TRUE;
const BOOLEAN DEFAULT_VERIFY_HOSTNAME = e_TRUE;
const BOOLEAN DEFAULT_TRUNCATE_LOG    = e_FALSE;

    /* size of the pass phrase buffer */

const int PASS_BUFFER_SIZE 	= 2048;

    /* delay when getting an IP address */

const struct timeval IP_DELAY 	= {20, 0};

    /* server and agent information */

ServerData      *g_serverData    = NULL;
AgentSet        *agentSet        = NULL;
CommandLine     *g_commandLine   = NULL;
HotInterpreters *hotInterpreters = NULL;

/* verify_server_ip

   Purpose: Verify that the machine specified with the -host parameter
            is the machine on which the server is running

     Input: verifyHostname = e_TRUE if we are actually going to verify the
                             hostname; e_FALSE if we are going to blindly
                             accept the hostname that was specified after the
                             -host command-line option.

    Output: The procedure returns -1 on error or if the specified machine
            is not the machine on which the server is running.  Otherwise
            the procedure returns 0.
*/

int verify_server_ip (BOOLEAN verifyHostname)
{
    char *hostname;
    struct timeval stop;

	/* get the IP number of machine specified with the -host parameter */
 
    stop = TimevalUtil::addTimevals (SystemUtil::getCurrentWall(), IP_DELAY);

    if (tcpip_getIP (g_serverData -> host, stop) != e_TCPIP_OK) {

	char *argv[] = {
	    "agentd: ERROR: unable to get IP address of local host \"", 
	    g_serverData -> host.getName().value(),
	    "\"",
	    ((char *) NULL)
	};

	g_serverData -> errorLog.error_app_cont (argv);
	return -1;
    }

    if (!verifyHostname) {
	return 0;
    }

	/* get the name of the the actual machine */

    if ((hostname = tcpip_getActualHostname ()) == NULL) {

	char *argv[] = {
	    "agentd: ERROR: unable to get actual name of local host",
	    ((char *) NULL)
	};

	g_serverData -> errorLog.error_sys_cont (argv);
	return -1;
    }
  
	/* get the IP address of the actual machine */

    MachineId actualMachine (hostname, UNKNOWN_IP);
    stop = TimevalUtil::addTimevals (SystemUtil::getCurrentWall(), IP_DELAY);

    if (tcpip_getIP (actualMachine, stop) != e_TCPIP_OK) {

	char *argv[] = {
	    "agentd: ERROR: unable to get IP address of local host \"", 
	    hostname,
	    "\"",
	    ((char *) NULL)
	};

	g_serverData -> errorLog.error_app_cont (argv);
	return -1;
    }

	/* compare the IP addresses */

    if (actualMachine.getIp() != g_serverData -> host.getIp()) {

	char *argv[] = {
	    "machine after \"-host\" (",
	    g_serverData -> host.getName().value(),
	    ") is *not* the machine on which the server is actually running (",
	    hostname,
	    ")",
	    ((char *) NULL)
	};

	g_serverData -> errorLog.error_app_cont (argv);
	return -1;
    }

    return 0;
}

/* open_timestamp_log
 
   Purpose: Open the timestamp log file

     Input: None

    Output: The procedure returns -1 if it is unable to open the timestamp
	    log file.  Otherwise the procedure returns 0 and updates the 
 	    file descriptor information inside g_serverData.  Note that the
	    procedure will do nothing (but still return 0) if the configuration
	    file does not request a timestamp log.
*/

int open_timestamp_log (void)
{
	// nothing to do if no timestamp log is required

    if (g_serverData->logTimestamps == e_FALSE) {
	return (0);
    }

	// if a timestmap log is required, see if we have a log filename

    if (g_serverData->timestampFilename == (char *) NULL) {

	char *argv[] = {
	    "agentd: ERROR: must specify filename for timestamp log in configuration file",
	    ((char *) NULL)
	};

	g_serverData -> errorLog.error_app_cont (argv);
	return (-1);
    }

	// open the log

    int fd; 
    char *filename = g_serverData->timestampFilename.value();

    if ((fd = open (filename, O_CREAT | O_TRUNC | O_WRONLY, 0644)) < 0) {

        char *argv[] = {
	    "agentd: INFO: unable to open timestamp log \"",
	    filename,
	    "\"",
	    ((char *) NULL)
	};

	g_serverData -> errorLog.error_app_cont (argv);
	return (-1);
    }

	// remember the file descriptor

    g_serverData->timestampFd = fd;
    return (0);
}

/* construct_server_keyname

   Purpose: Construct the keyname for the server

     Input: None

    Output: The procedure returns -1 if it is unable to construct the
	    keyname.  Otherwise the procedure returns 0 and updates
	    the server information with the appropriate keyname.
*/

int construct_server_keyname (void)
{
    char *keyname;

	/* construct the keyname */

    if ((keyname = makeFullServerKeyname (g_serverData -> host)) == NULL) {

	char *argv[] = {
	    "agentd: ERROR: unable to construct server keyname",
	    ((char *) NULL)
	};

	g_serverData -> errorLog.error_app_cont (argv);
	return -1; 
    } 

    char *argv[] = {
	"agentd: INFO: server keyname is \"",
	keyname,
	"\"",
	((char *) NULL)
    };

    g_serverData -> errorLog.error_app_cont (argv);

	/* record */

    SEC_IDENTITY id (keyname, g_serverData -> passPhrase.value(), e_TRUE, e_FALSE);
    g_serverData -> security.setOwner (id);
    g_serverData -> security.setMachine (id);
    g_serverData -> security.setOwnerSigned (e_TRUE);

	/* cleanup */

    DELETE_ARRAY_OBJECT(keyname);
    return 0;
}

/* printUsage

   Purpose: Print usage instructions to standard output

     Input: name = name that was used to start the program

    Output: The procedure outputs usage instructions to standard output.
*/

void printUsage (char *name)
{
    int i;
    int len      = strlen(name);
    char *spaces = new char [len + 1];

	/* initialize the spacing string */

    for (i = 0; i < len; i++) {
	spaces[i] = ' ';
    }

    spaces[i] = '\0';

	/* print usage instructions */

    printf ("\n");
    printf ("USAGE:\n\n");
    printf ("%s -host <hostname>\n", name);
    printf ("%s -conf <filename>\n", spaces);
    printf ("%s [-nodaemon]\n", spaces);
    printf ("%s [-noverify]\n", spaces);
    printf ("%s [-trunc]\n", spaces);

	/* cleanup */

    DELETE_ARRAY_OBJECT(spaces);
}

/* CommandLine::getArguments

   Purpose: Get the command line parameters

     Input: argc = number of command line parameters
		   (int)

            argv = the command line arguments
		   (char **)

    Output: The procedure returns -1 if the command line parameters are
            specified incorrectly.  Otherwise the procedure fills in the
	    CommandLine structure and returns 0.
*/

CommandLine::CommandLine (void):
    configurationFile ((char *) NULL),
    hostname ((char *) NULL),
    daemon (DEFAULT_DAEMON),
    verifyHostname (DEFAULT_VERIFY_HOSTNAME),
    truncateLog (DEFAULT_TRUNCATE_LOG)
{
    // empty
}

CommandLine::CommandLine (const CommandLine &commandLine):
    configurationFile ((char *) NULL),
    hostname ((char *) NULL),
    daemon (commandLine.daemon),
    verifyHostname (commandLine.verifyHostname),
    truncateLog (commandLine.truncateLog)
{
    if (commandLine.configurationFile != (char *) NULL) {
	configurationFile = strcpyWithAlloc (commandLine.configurationFile);
    }

    if (commandLine.hostname != (char *) NULL) {
	hostname = strcpyWithAlloc (commandLine.hostname);
    }
}

CommandLine& CommandLine::operator= (const CommandLine &commandLine)
{
    if (this != &commandLine) {

        daemon         = commandLine.daemon;
        verifyHostname = commandLine.verifyHostname;
        truncateLog    = commandLine.truncateLog;

	DELETE_ARRAY_OBJECT(configurationFile);
	DELETE_ARRAY_OBJECT(hostname);

	if (commandLine.configurationFile != (char *) NULL) {
	    configurationFile = strcpyWithAlloc (commandLine.configurationFile);
	} else {
	    configurationFile = (char *) NULL;
	}

        if (commandLine.hostname != (char *) NULL) {
	    hostname = strcpyWithAlloc (commandLine.hostname);
        } else {
	    hostname = (char *) NULL;
	}
    }

    return (*this);
}

CommandLine::~CommandLine ()
{
    DELETE_ARRAY_OBJECT(configurationFile);
    DELETE_ARRAY_OBJECT(hostname);
}

	// get the arguments specified on the program command line

CommandLine::CommandLineError CommandLine::getArguments (int argc, char **argv)
{
    int confFound    = FALSE;     // configuration file specified 
    int hostFound    = FALSE;     // host specified
    unsigned numArgs = argc - 1;  // number of arguments       
    unsigned curArg  = 1;         // current argument         

	// get the arguments from the command line

    while (curArg <= numArgs) {

	if (!strcmp(argv[curArg], "-host")) {

	    if (++curArg > numArgs) {
		return (e_CommandMissingParameter);
	    }

#ifdef FIX_NOW
	    DynamicString name (argv[curArg]);
	    name.convertToLowercase();
	    g_serverData -> host.setName (name);
#endif

	    hostname = strcpyWithAlloc (argv[curArg]);
	    hostFound = TRUE;

	} else if (!strcmp(argv[curArg], "-conf")) {

	    if (++curArg > numArgs) {
		return (e_CommandMissingParameter);
	    }

	    configurationFile = strcpyWithAlloc (argv[curArg]);
	    confFound = TRUE;

	} else if (!strcmp(argv[curArg], "-nodaemon")) {
	    daemon = e_FALSE;
	} else if (!strcmp(argv[curArg], "-daemon")) {
	    daemon = e_TRUE;
	} else if (!strcmp(argv[curArg], "-noverify")) {
	    verifyHostname = e_FALSE;
	} else if (!strcmp(argv[curArg], "-verify")) {
	    verifyHostname = e_TRUE;
	} else if (!strcmp(argv[curArg], "-notrunc")) {
	    truncateLog = e_FALSE;
	} else if (!strcmp(argv[curArg], "-trunc")) {
	    truncateLog = e_TRUE;
	} else {
	    return (e_CommandInvalidOption);
	}

	curArg++;
    }

    return (confFound && hostFound ? e_CommandOk : e_CommandMissingOption);
}

/* switchUser

   Purpose: Reset user priveleges

     Input: None

    Output: The procedure changes the user priveleges -- i.e. the procedure
	    changes the user id of the process.
   
     NOTES: This procedure was taken from the Apache Web server.
*/

static uid_t getUserId (char *name) 
{
    struct passwd *ent;

    if(name[0] == '#') {

        return(atoi(&name[1]));
    }

    if(!(ent = getpwnam(name))) {

	char *argv[] = {
	    "ERROR: agentd: bad user name \"",
	    name,
	    "\"",
	    ((char *) NULL)
	};

	g_serverData -> errorLog.error_sys_quit (argv);
	abort_with_message ("should never get here");   // should never get here
    } 

    return (ent->pw_uid);
}

static void switchUser (void)
{
    if(!geteuid()) {    // only switch if running as root

	uid_t uid = getUserId (g_serverData -> userName.value());

	if (setuid(uid) < 0) {

	    char *argv[] = {
		"ERROR: agentd: unable to change uid",
		((char *) NULL)
	    };

	    g_serverData -> errorLog.error_sys_quit (argv);
	    abort_with_message ("should never get here");   // should never get here
	}

	char *argv[] = {
	    "INFO: agentd: user has been switched to \"",
	    g_serverData -> userName.value(),
	    "\"",
	    ((char *) NULL)
	};

	g_serverData -> errorLog.error_app_cont (argv);
    }
}

/* switchGroup

   Purpose: Reset group priveleges

     Input: None

    Output: The procedure resets the group priveleges -- i.e. the procedure
	    changes the group id of the process.

     NOTES: We use the username as set in the config files rather than
	    a lookup on uid -- the same uid may have multiple passwd entries 
	    with different sets of groups for each.

	    This code is taken from the Apache web server.
*/
 
static gid_t getGroupId (char *name) 
{
    struct group *ent;

    if(name[0] == '#') {
        return(atoi(&name[1]));
    }

    if(!(ent = getgrnam(name))) {

	char *argv[] = {
	    "ERROR: agentd: bad group name \"",
	    name,
	    "\"",
	    ((char *) NULL)
	};

	g_serverData -> errorLog.error_sys_quit (argv);
	abort_with_message ("should never get here");   // should never get here
    }
 
    return (ent -> gr_gid);
}

static void switchGroup (void)
{
    if(!geteuid()) {    // can only do this if we are root

	char *name;
 	char *userName = g_serverData -> userName.value();
	char *groupName = g_serverData -> groupName.value();

	    /* get the group id */

	gid_t group_id = getGroupId (groupName);

	    /* get username if passed as a uid */
    
	if (userName[0] == '#') {

	    struct passwd* ent;
	    uid_t uid = atoi(&userName[1]);

	    if ((ent = getpwuid(uid)) == NULL) {

		char *argv[] = {
		    "ERROR: agentd: couldn't determine username from uid\"",
		    userName,
		    "\"",
	    	    ((char *) NULL)
		};

		g_serverData -> errorLog.error_sys_quit (argv);
		abort_with_message ("should never get here");   // should never get here
	    }

	    name = ent -> pw_name;

	} else {

	    name = userName; 
	}

#ifndef __EMX__ 

	    /* OS/2 dosen't support groups. */
	    /* Reset `groups' attributes. */
    
	if (initgroups(name, group_id) == -1) {

	    char *argv[] = {
		"ERROR: agentd: unable to set groups",
	    	 ((char *) NULL)
	    };
	 
	    g_serverData -> errorLog.error_sys_quit (argv); 
	    abort_with_message ("should never get here");   // should never get here
	}

	if (setgid(group_id) == -1) {

	    char *argv[] = {
		"ERROR: agentd: unable to set group id",
	    	 ((char *) NULL)
	    };

	    g_serverData -> errorLog.error_sys_quit (argv); 
	    abort_with_message ("should never get here");   // should never get here
	}
#endif 

	char *argv[] = {
	    "INFO: agentd: group has been switched to \"",
	    groupName,
	    "\"",
	    ((char *) NULL)
	};

	g_serverData -> errorLog.error_app_cont (argv);
    }
}

/* redirectOutput

   Purpose: Redirect stdin, stdout and stdout to /dev/null.

     Input: None

    Output: The procedure halts the server and logs an error message if
            it is unable to redirect stdin, stdout and stdout.  Otherwise
	    the procedure redirects stdin, stdout and stdout and then
	    returns.
*/

static void redirectOutput (void)
{
    if (FileUtility::fileRedirect() != FileUtility::e_OK) {
	g_serverData -> errorLog.error_sys_quit 
	    ("agentd: unable to redirect stdin, stdout and stdout");
    }
}

/* check_environment

    Purpose: Make sure that the necessary environment variables are defined

      Input: None
  
     Output: The procedure returns -1 if the necessary environment variables
	     are not defined.  Otherwise the procedure returns 0.  Currently
	     the procedure only checks for the HOME environment variable which
	     should contain the full pathname of the server account's home 
	     directory.
*/

static int check_environment (void)
{
    char *value;

	/* check for the HOME environment variable */

    if ((value = getenv ("HOME")) == NULL) {

	char *argv[] = {
	     "agentd: ERROR: environment variable \"HOME\" must be set to the full pathname of the agent account's home directory",
	     ((char *) NULL)
	};

	g_serverData -> errorLog.error_app_cont (argv);
	return -1;
    }

	/* success */

    char *argv[] = {
	"agentd: INFO: environment variable \"HOME\" is set to \"",
	value,
	"\"",
	((char *) NULL) 
    };

    g_serverData -> errorLog.error_app_cont (argv);
    return 0;
}

/* writePid

   Purpose: Write out the process id (of the main server process) to the
	    process id file

     Input: None

    Output: The function returns -1 on error.  Otherwise the function returns 0
	    and writes out the process id.
*/

int writePid (void)
{
    int fd;
    char temp[16];

	/* open the file */

    if ((fd = open (g_serverData -> pidFilename.value(), O_CREAT | O_TRUNC | O_WRONLY, 0644)) < 0) {
	
	char *argv[] = {
	    "agentd: ERROR: unable to open the pid file \"",
	    g_serverData -> pidFilename.value(),
	    "\"",
	    ((char *) NULL)
	};

	g_serverData -> errorLog.error_app_cont (argv);
	return (-1);
    }

	/* write out the process id */

    sprintf (temp, "%d\n", g_serverData -> agentdPid);
    write (fd, temp, strlen(temp));

	/* done */

    close (fd);
    return (0);
}

int main (int argc, char **argv)
{
	// get the command-line arguments

    g_commandLine = new CommandLine;

    if (g_commandLine->getArguments (argc, argv) != CommandLine::e_CommandOk) {
	printUsage (argv[0]);
	exit (1);
    }

	// create the server configuration class

    g_serverData = new ServerData (g_commandLine->hostname);

	// read in only the User, Group and LogFile fields from the configuration
	// file, since undex LinuxThreads, we need to switch the user/group and 
	// become a daemon process before we do any other server initialization 
	// (i.e., before we start using threads and mutexes).  If we do other
	// initialization first, the LinuxThreads control thread will be
	// created as root, and will remain root even after we switch the main 
	// thread to the desired User/Group.  Having only the control thread 
	// owned by root leads to LOCKUP, since the application threads can NOT
	// send the necessary signals to the control thread.

#ifdef FIX_LATER
	// Perhaps a better way to do this would be to parse and read in the
	// entire configuration file, but keep the configuration fields/values
	// in a temporary table, rather than doing the server initialization
	// right away.  Put another away, we should probably separate the
	// routines that read in the configuration file and do basic sanity
	// checks from the routines that actually initialize the agent server.
#endif


    if (g_serverData->load (*g_commandLine, ServerData::e_PRELOAD) < 0) {
	exit (1);
    }

	// turn the server into a daemon

    if (g_commandLine->daemon) {
	make_daemon ();
	redirectOutput ();
    }

	// change the ownership of the log file, and switch the user and group 	
	// id's if we are running as root.  We change the ownership of the log
	// file since we want to log messages while still running as root  
	// (which means that we end up opening it as root), but it has to be
	// owned by the user id to which we are going to switch.

    uid_t uid = getUserId (g_serverData->userName.value());
    gid_t gid = getGroupId (g_serverData->groupName.value());
    chown (g_serverData->logFilename.value(), uid, gid);
    switchGroup();
    switchUser();

	// set the process group id to the process id -- only do this 
	// if we are daemon process; otherwise we'll just stay in our 
	// current process group so that we respond to Ctrl-C. 

    if (g_commandLine -> daemon) {
	setpgid (0, 0);
    }

	// clear the file creation mask (otherwise the Unix socket might 
	// be created with the wrong permissions)                    

    umask (0);

	// initialize the signal subsystem

    SignalHandlers::initializeSignals();

	// turn on the recording of all open file descriptors

    FileTable::turnFileRecordingOn();
    FileTable::add (0, FileTable::e_STANDARD_IN);
    FileTable::add (1, FileTable::e_STANDARD_OUT);
    FileTable::add (2, FileTable::e_STANDARD_ERROR);

	// catch all signals that might cause a core dump

    const int c_endFlag = 0xFFFF;

    static int coredumpSignals[] = {
	SIGSEGV,
#ifdef SIGBUS
	SIGBUS,
#endif
#ifdef SIGABORT
	SIGABORT,
#endif
#ifdef SIGABRT
	SIGABRT,
#endif
	c_endFlag
    };

    for (int i = 0; coredumpSignals[i] != c_endFlag; ++i) { 

	if (SignalHandlers::install_signal_intr (coredumpSignals[i], (SignalHandlers::funcType) coredump_handler, 0) < 0) { 

	    char *messageArgv[] = {
	 	"agentd: ERROR: unable to install handler for ",
		(char *) SignalHandlers::SignoToString (coredumpSignals[i]),
		(char *) NULL
	    };

	    g_serverData -> errorLog.error_sys_quit (messageArgv);
	}
    }

	// create the agent tables and hot interpreter data structures

    agentSet        = new AgentSet();
    hotInterpreters = new HotInterpreters();

	/* load the configuration file */

    if (g_serverData -> load (*g_commandLine, ServerData::e_INITIAL_LOAD) < 0) {
	exit (1);
    }

	/* get the IP address of the current machine */

    if (verify_server_ip (g_commandLine -> verifyHostname) < 0) {
	exit (1);
    }

	/* construct the server keyname */

    if (construct_server_keyname() < 0) {
	exit (1);
    }

	/* open the timestamp log file */

    if (open_timestamp_log() < 0) {
	exit (1);
    }
 
	/* resolve the IP address of the AllowedMachine's and TrustedMachine's */

    g_serverData -> resolveIp (ServerData::e_INITIAL_LOAD);

	/* make sure that the HOME environment variable is defined */

    if (check_environment () < 0) {
	exit (1);
    }
  
	/* write the main server process id out to the file */

    g_serverData -> agentdPid = getpid ();

    if (writePid() < 0) {
	exit (1);
    }

	/* ignore SIGPIPE so that we do not die on pipe breakdown, and */
	/* catch SIGTERM, SIGQUIT and SIGINT so that we can cleanup at */
	/* server death and SIGHUP so that we can reload the server    */
	/* configuration                                               */

    if (SignalHandlers::install_signal_intr (SIGPIPE, (SignalHandlers::funcType) SIG_IGN, 0) < 0) { 
	g_serverData -> errorLog.error_sys_quit 
	    ("agentd: unable to ignore SIGPIPE");
    }

    if (SignalHandlers::install_signal_intr (SIGTERM, sigterm_handler, 0) < 0) {
	g_serverData -> errorLog.error_sys_quit 
	    ("agentd: unable to install SIGTERM handler");
    }

    if (SignalHandlers::install_signal_intr (SIGQUIT, sigterm_handler, 0) < 0) {
	g_serverData -> errorLog.error_sys_quit 
	    ("agentd: unable to install SIGQUIT handler");
    }

    if (SignalHandlers::install_signal_intr (SIGINT, sigterm_handler, 0) < 0) {
	g_serverData -> errorLog.error_sys_quit 
	    ("agentd: unable to install SIGTERM handler");
    }

    if (SignalHandlers::install_signal_intr (SIGHUP, sighup_handler, 0) < 0) {
	g_serverData -> errorLog.error_sys_quit 
	    ("agentd: unable to install SIGHUP handler");
    }

	/* open the TCP/IP socket descriptor that brings in remote requests */

    int chosenPort;

    if (tcpip_setup (g_serverData->socketPort, g_serverData->sockfd, chosenPort) != e_TCPIP_OK) {

	char temp[16];
	sprintf (temp, "%u", g_serverData -> socketPort);

	char *argv[] = {
	    "agentd: ERROR: unable to open TCP/IP socket on port ",
	    temp,
	    ((char *) NULL)
	};

	g_serverData -> errorLog.error_sys_quit (argv);
	abort_with_message ("should never get here");   // should never get here
    }

	/* open the Unix socket descriptor that brings in local requests */

    if (unix_setup (g_serverData->socketFilename.value(), g_serverData->unixSockfd) != e_TCPIP_OK) {

	char *argv[] = {
	    "agentd: ERROR: unable to open Unix domain socket \"",
	    g_serverData -> socketFilename.value(),
	    "\"",
	    ((char *) NULL)
	};

	g_serverData -> errorLog.error_sys_quit (argv);
	abort_with_message ("should never get here");   // should never get here
    }

	/* start up the interpreter creation thread */

    hotInterpreters -> startCreationThread();

	/* start watching the Unix domain and TCP/IP sockets */

    Socketd::socketd ();
    abort_with_message ("should never get here"); 
    return (1);	     // prevents compiler warning
}
