#include "platPorting.h"
#include "agentd.h"
#include "servConf.h"
#include "servRandom.h"

/* sigterm_handler

   Purpose: This procedure handles the SIGTERM signal for all server processes.
*/

void sigterm_handler (int, SignalHandlers::ClientData)
{
    RandomFile::RandomFileErrors rc;

	/* log a message that says we have received a SIGTERM */

    char pidString[16];
    sprintf (pidString, "%d", (int) getpid());

    char *argv[] = {
	"agentd: INFO: server process ",
	pidString,
	" received a SIGTERM: exiting",
	((char *) NULL)
    };

    g_serverData -> errorLog.error_app_cont (argv); 

	/* burn the random bits file */

    RandomFile randomFile 
	(g_serverData -> randomFilename, g_serverData -> randomBits);

    if ((rc = randomFile.burnBits()) == RandomFile::e_OK) {

	char *argv[] = {
	    "agentd: INFO: burned random bits file \"",
	    g_serverData -> randomFilename.value(),
	    "\" before exit",
	    ((char *) NULL)
	};

	g_serverData -> errorLog.error_app_cont (argv); 

    } else {

	char temp[16];
	sprintf (temp, "%d %d", rc, errno);

	char *argv[] = {
	    "agentd: ERROR: unable to burn random bits file \"",
	    g_serverData -> randomFilename.value(),
	    "\" before exit",
	    temp,
	    ((char *) NULL)
	};

	g_serverData -> errorLog.error_app_cont (argv); 
    }

	/* exit -- all open file descriptors are automatically closed */

    exit (0);
}

/* sighup_handler

   Purpose: This procedure handles the SIGHUP signal for all server processes.
*/

static sig_atomic_t sighupFired = 0; 

int isSighupPending (void)
{
    sigset_t mask;
    sigpending (&mask);
    return (sigismember (&mask, SIGPROF));
}

int hasSighupFired (void)
{
    int fired = sighupFired;
    sighupFired = 0;
    return (fired);
}

void pretendSighupFired (void) 
{
    sighupFired = 1;
}

void sighup_handler (int, SignalHandlers::ClientData)
{
	/* propagate the signal to the rest of the process group --  */
	/* we propagate a SIGPROF rather than a SIGHUP so that we do */
	/* not get into an infinite signal loop                      */

    kill (-getpgrp(), SIGPROF);
}

/* coredump_handler

   Purpose: This procedure handles the SIGSEGV, SIBABRT, SIGABORT and
            SIGBUS signals for all server processes.
*/

void coredump_handler (int signo, SignalHandlers::ClientData)
{
        /*
 	 * Restore the default action for the SIGSEGV, SIGABRT, SIGABORT and
	 * SIGBUS signals (so that we do not end up in an infinite loop)
         */

    SignalHandlers::install_signal (SIGSEGV, (SignalHandlers::funcType) SIG_DFL, 0);
#ifdef SIGBUS
    SignalHandlers::install_signal (SIGBUS, (SignalHandlers::funcType) SIG_DFL, 0);
#endif
#ifdef SIGABORT
    SignalHandlers::install_signal (SIGABORT, (SignalHandlers::funcType) SIG_DFL, 0);
#endif
#ifdef SIGABRT
    SignalHandlers::install_signal (SIGABRT, (SignalHandlers::funcType) SIG_DFL, 0);
#endif

	/*
	 * Get the name of the signal
	 */

    const char *signalName = SignalHandlers::SignoToString (signo);

	/*
	 * Log that we are attempting to dump core
	 */

    char *argv[] = {
	"agentd: ERROR: caught a ",
	(char *) signalName,
	": attempting to dump core in directory \"",
	(char *) g_serverData -> coredumpDirectory.value(),
	"\"",
	(char *) NULL
    };

    g_serverData -> errorLog.error_app_cont (argv);

	/*
	 * Change directories -- silently ignore any failures
	 */

    if (!(g_serverData -> coredumpDirectory.isEmpty())) {
	const char *directory = g_serverData -> coredumpDirectory.value();
	(void) mkdir (directory, 0700);
	(void) chmod (directory, 0700);
	(void) chdir (directory);
    }

    abort_with_message (signalName);
    exit (1);		// better not ever get here
}
