/* Agent Tcl
   Bob Gray
   26 May 1998

   hotInterpreter.cc

   This file implements the HotInterpreter class, which starts up a pool of
   interpreters so that an interpreter will be available immediately when
   an agent arrives.

   Copyright (c) 1995-1998, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "platDelete.h"
#include "platExclusion.h"
#include "platInterrupt.h"
#include "platThread.h"
#include "suppFileTable.h"
#include "suppHash.h"
#include "genManager.h"
#include "genPipe.h"
#include "genUtility.h"
#include "agentd.h"
#include "hotInterpreter.h"
#include "interp.h"
#include "servConf.h"
#include "truefalse.h"

int hotPid = 0;
int sigchldPid = 0;

#ifdef FIX_LATER
/*
 * Our current arrangement is that even though there are multiple
 * agents per interpreter, there is still one file descriptor PER AGENT,
 * which connects the agent to the server.  This approach is a
 * good intermediate step between the old system where there was only one
 * agent per interpreter, and the FUTURE system where agents in the same
 * interpreter will share a single file descriptor.
 */
#endif

/*
 * Each FullReadyInterpreter holds complete information about a running
 * agent interpreter (e.g., its process id, all the file descriptors that
 * connect the interpreter to the server, and so on).
 */

class FullReadyInterpreter
{
public:

	// errors

    enum Errors {
	e_Ok,		// everything is okay
	e_NoFd,		// there was no free fd
	e_BadFd		// the returned fd does not belong to this interpeter
    };

	// constructor and destructor

    FullReadyInterpreter (pid_t pid, int fdCount, int *fds);
   ~FullReadyInterpreter ();

	// get the process id of the interpreter

    pid_t getPid (void) {
	return (m_pid);
    }

	// get tht total number of fds

    int getNumFds (void) {
	return (m_numFds);
    }

	// check if there is an available file descriptor

    BOOLEAN hasFreeFd (void) {
	return ((m_freeIndex != -1) ? e_TRUE : e_FALSE);
    }

	// get a free file descriptor for an incoming agent

    Errors getFd (int &fd);

	// return the file descriptor after the agent has finished using it
	// (i.e., after the agent has terminated)

    Errors returnFd (int fd);

	// close all FREE fds (used fds are the responsibility of the
	// handler threads that that is currently using them)

#ifdef FIX_LATER
	// this can probably be moved into the destructor 
#endif

    Errors closeFreeFds (void);
    
private:

	// data

    pid_t m_pid;	 // process id of the agent interpeter
    int m_singleList;	 // used/free list when there is only one fd
    int m_singleFd;	 // fd list when there is only one fd;
    int *m_list;	 // used/free list (where used and free refers to the slots inside m_fds)
    int *m_fds;		 // fds
    int m_freeIndex;     // start of the free list inside m_list
    int m_freeIndexTail; // end of the free list inside m_list
    int m_usedIndex;	 // start of the used list inside m_list
    int m_numFds;	 // total number of fds

	// unimplemented

    FullReadyInterpreter (const FullReadyInterpreter &);
    FullReadyInterpreter& operator= (const FullReadyInterpreter &);
};

FullReadyInterpreter::FullReadyInterpreter (pid_t pid, int fdCount, int *fds):
    m_pid (pid),
    m_freeIndex (0),
    m_freeIndexTail (fdCount - 1),
    m_usedIndex (-1),
    m_numFds (fdCount)
{
	// assertions

    assert (fdCount > 0);
    assert (fds != NULL);

	// remember the fds and construct the free/available flags

    if (fdCount == 1) {

	m_list    = &m_singleList;
	m_fds     = &m_singleFd;
	m_fds [0] = fds[0];
	m_list[0] = -1;

    } else {

	m_list = new int [fdCount];
	m_fds  = new int [fdCount];

	for (int i = 0; i < fdCount - 1; ++i) {
	    m_list[i] = i + 1;
	}

	m_list[fdCount-1] = -1;

	for (int j = 0; j < fdCount; ++j) {
	    m_fds[j] = fds[j];
	}
    }
}

FullReadyInterpreter::~FullReadyInterpreter()
{
    if (m_list != &m_singleList) {
	DELETE_ARRAY_OBJECT(m_list);
    }


    if (m_fds != &m_singleFd) {
	DELETE_ARRAY_OBJECT(m_list);
    }
}

FullReadyInterpreter::Errors FullReadyInterpreter::getFd (int &fd) 
{
    Errors rc;

    if (m_freeIndex == -1) {

	fd = -1;
	rc = e_NoFd;

    } else {

	fd = m_fds[m_freeIndex];
	rc = e_Ok;

	int newUsedIndex = m_freeIndex;
	m_freeIndex = m_list[m_freeIndex];
	m_list[newUsedIndex] = m_usedIndex;
	m_usedIndex = newUsedIndex;

	if (m_freeIndex == -1) {
	    m_freeIndexTail = -1;
	}
    }

    return (rc);
}

FullReadyInterpreter::Errors FullReadyInterpreter::returnFd (int fd)
{
    Errors rc;

	// find the fd in the list of used fds

    int newFreeIndex = -1;
    int prevUsedIndex = -1;

    for (int i = m_usedIndex; i != -1; i = m_list[i]) {

	if (m_fds[i] == fd) {
	    newFreeIndex = i;
	    break;
	}

	prevUsedIndex = i;
    }

	// error if we did not find the fd -- otherwise splice the fd
	// out of the used list and return it to the END of the free list;
	// we return it to the end so that we cycle through the fds as much
	// as possible (which has a performance advantage for some of the
	// interpreters where a thread is created per fd per agent, i.e.,
	// cycling through the fds hides the thread creation latency)

    if (newFreeIndex == -1) {

	rc = e_BadFd;

    } else {
	
	if (prevUsedIndex == -1) {
	    m_usedIndex = m_list[newFreeIndex];
	} else {
	    m_list[prevUsedIndex] = m_list[newFreeIndex];
	}

	if (m_freeIndexTail != -1) {
	    m_list[m_freeIndexTail] = newFreeIndex;
	}

	m_list[newFreeIndex] = -1;
	m_freeIndexTail = newFreeIndex;

	if (m_freeIndex == -1) {
	    m_freeIndex = m_freeIndexTail;
	}

	rc = e_Ok;
    }

    return (rc);
}

FullReadyInterpreter::Errors FullReadyInterpreter::closeFreeFds (void)
{
    for (int i = m_freeIndex; i != -1; i = m_list[i]) {
	PipeUtility::closePipe (m_fds[i]);
    }

    return (e_Ok);
}

/*
 * each InterpreterStatus holds the status of an interpreter process (e.g.,
 * its exit code if it has exited).
 */

class InterpreterStatus
{
private:

	// exit code from interpreter process, flag that indicates whether
	// the interpreter process has exited, and flag that indicates
	// whether the interpreter process is in use, interpreter pid,
	// condition variable that is signaled when the interpreter dies,
	// and

    WAIT_STATUS_TYPE m_exitCode;
    BOOLEAN          m_exited;
    int              m_useCount;
    pid_t            m_pid;
    UINT_32          m_interpreterDeath;
    int              m_waitingForInterpreterDeath;
    MonitorLock     &m_monitor;

	// do not allow copying or assignment

    InterpreterStatus (const InterpreterStatus &status):
	m_monitor (status.m_monitor)
    {}

    InterpreterStatus& operator= (const InterpreterStatus &) 
    { return (*this); }

public:

	// set the exit status

    void setExitCode (WAIT_STATUS_TYPE exitCode) {
	m_exitCode = exitCode;
	m_exited   = e_TRUE;
    }

	// check if the interpreter is in use (i.e., if the interpreter
	// process has been handed off to a client)

    BOOLEAN isInUse (void) {
	return ((m_useCount > 0) ? e_TRUE : e_FALSE);
    }

	// check if the interpreter has finished

    BOOLEAN hasExited (void) {
	return (m_exited);
    }

	// get the exit code of a finished interpreter

    WAIT_STATUS_TYPE getExitCode (void) {
	return (m_exitCode);
    }

	// mark that the interpreter is in use by another client (i.e., one
	// of the connections to the interpreter process has just been handed
	// off to a client)

    void markInUse (void) {
	m_useCount += 1;
    }

	// mark that the interpreter is no longer in use by one client (i.e., 
	// a client has just returned one of the connections to the interpreter
	// process to the pool) 

    void markReturned (void) {
	m_useCount -= 1;
    }

	// wait for interpreter death

    void waitForInterpreterDeath (void) {
	while (!m_exited) {
	    m_waitingForInterpreterDeath += 1;
	    m_monitor.wait (m_interpreterDeath);
	    m_waitingForInterpreterDeath -= 1;
	}
    }
	
	// wakeup a thread that is waiting for interpreter death

    void wakeupWaiter (void) {
#ifdef FIX_NOW
	    // We probably need to find a way to do an explicit "wakeup all"
	    // here, since we can now have multiple threads waiting for the
	    // process to exit.
#endif
	if (m_waitingForInterpreterDeath) {
	    m_monitor.wakeup (m_interpreterDeath);
	}
    }

	// constructors and assignment operator

    InterpreterStatus (MonitorLock &monitor, pid_t pid);
   ~InterpreterStatus ();
};

InterpreterStatus::InterpreterStatus (MonitorLock &monitor, pid_t pid):
    m_exited (e_FALSE),
    m_useCount (0),
    m_pid (pid),
    m_waitingForInterpreterDeath (0),
    m_monitor (monitor)
{
    m_monitor.getConditionVariable (m_interpreterDeath);
}

InterpreterStatus::~InterpreterStatus ()
{
    m_monitor.returnConditionVariable (m_interpreterDeath);
}

/*
 * each InterprererQueue is a queue of ready interpreters
 */

class InterpreterQueue
{
    enum DefaultQueueSlots {
	e_DefaultQueueSlots = 0x8
    };
	
	// interpreter description, queue of interpreters that have free
	// file descriptors, queue of interpreters that do not have free
	// file descriptors, and a flag that indicates whether we have 
	// created the full initial complement of interpreters

    INTERP m_interpreter;
    QUEUE m_readyQueue;
    QUEUE m_usedQueue;

public:

	// get the interpreter type, arguments and executable name

    INTERP::InterpreterTypes getInterpreterType (void) {
	return (m_interpreter.getType());
    }

    char **getInterpreterArgv (void) {
	return (m_interpreter.getArgv());
    }

    DynamicString getInterpreterExecutable (void) {
	return (m_interpreter.getExecutable());
    }

	// get the number of agents that can be simultaneously executing
	// in each interpeter process, which determines how many pipes
	// we need to create

    int getAgentsPerInterp (void) {
	return (m_interpreter.getAgentsPerInterp());
    }
	
	// see if we need to add an interpreter

    BOOLEAN needInterpreter (void) {

	BOOLEAN rc;

	int numReady = m_readyQueue.get_count();

	if (numReady < m_interpreter.getMinHotCount()) {
	    rc = e_TRUE; 
	} else {
	    rc = e_FALSE;
	}

	return (rc);
    }

	// add a new ready interpreter

    void addReadyInterpreter 
	(FullReadyInterpreter *fullReadyInterpreter);

	// return a ready interpreter

    BOOLEAN returnReadyInterpreter
	(const HotInterpreters::ReadyInterpreter &readyInterpreter);

	// get a ready interpreter -- returns e_FALSE if there is no
	// interpreter in the queue; otherwise returns e_TRUE and fills in the
	// ReadyInterpreter instance

    BOOLEAN getReadyInterpreter
	(HotInterpreters::ReadyInterpreter &readyInterpreter);

	// find and remove an interpreter -- returns e_TRUE if the interpreter
	// was found and removed; returns a pointer to the removed
	// FullReadyInterpreter instance in fullReadyInterpreter (the caller
	// must delete it)

    BOOLEAN findAndRemoveReadyInterpreter
	(pid_t pid, FullReadyInterpreter *&fullReadyInterpreter);
    
	// constructor and destructor

    InterpreterQueue (const INTERP &interp);
   ~InterpreterQueue ();
};

InterpreterQueue::InterpreterQueue (const INTERP &interp):
    m_interpreter    (interp),
    m_readyQueue     (),
    m_usedQueue	     ()
{
    // empty
}

BOOLEAN InterpreterQueue::returnReadyInterpreter (const HotInterpreters::ReadyInterpreter &readyInterpreter)
{
    BOOLEAN rc = e_FALSE;
    BOOLEAN found = e_FALSE;

#ifdef FIX_LATER
	// scanning the queues to find the process id is fast for those
	// interpreters that allow multiple agents per interpreter, but slow
	// for those interpreters that allow only one agent per interpreter
	// (the first case involves an order of magnitude fewer interpreters
	// than the second case to handle the same number of agents)
#endif

	// information needed for the searches

    pid_t pid    = readyInterpreter.pid;
    int fd       = readyInterpreter.fd;
    int numReady = m_readyQueue.get_count();
    int numUsed  = m_usedQueue.get_count();

	// search the queues for the interpreter -- first search the queue
	// of interpreters that no longer have free fds
 
    for (int i = 0; ((i < numUsed) && (!found)); ++i) {
	
	FullReadyInterpreter *fullReadyInterpreter = 
	    (FullReadyInterpreter *) m_usedQueue.dequeue();

	if (fullReadyInterpreter -> getPid() != pid) {

	    m_usedQueue.enqueue ((void *) fullReadyInterpreter);

	} else {

	    if (fullReadyInterpreter -> returnFd (fd) != FullReadyInterpreter::e_Ok) {

		m_usedQueue.enqueue ((void *) fullReadyInterpreter);
		rc = e_FALSE;

	    } else if (fullReadyInterpreter -> getNumFds() == 1) {

		if (numReady >= m_interpreter.getMaxHotCount()) {
		    DELETE_OBJECT(fullReadyInterpreter);
		    rc = e_FALSE;
		} else {
		    m_readyQueue.enqueue ((void *) fullReadyInterpreter);
		    rc = e_TRUE;
	 	}

	    } else {

#ifdef FIX_LATER
		    // we should take some intelligent action if the ready
		    // queue already contains the maximum number of
		    // interpreters
#endif

		m_readyQueue.enqueue ((void *) fullReadyInterpreter);
		rc = e_TRUE;
	    }

	    found = e_TRUE;
	}
    }

	/* then search the queue of interpreters that do have free fds */

    for (int j = 0; ((j < numReady) && (!found)); ++j) {

	FullReadyInterpreter *fullReadyInterpreter = 
	    (FullReadyInterpreter *) m_readyQueue.dequeue();

	if (fullReadyInterpreter -> getPid() == pid) {

	    if (fullReadyInterpreter -> returnFd (fd) != FullReadyInterpreter::e_Ok) {
		rc = e_FALSE;
	    } else {
		rc = e_TRUE;
	    }

	    found = e_TRUE;
	}

	    // put the interpreter back on the queue no matter what

	m_readyQueue.enqueue ((void *) fullReadyInterpreter);
    }

	/* done */

    return (rc);
}

void InterpreterQueue::addReadyInterpreter (FullReadyInterpreter *fullReadyInterpreter)
{
    m_readyQueue.enqueue ((void *) fullReadyInterpreter);
}

BOOLEAN InterpreterQueue::getReadyInterpreter (HotInterpreters::ReadyInterpreter &readyInterpreter)
{
    BOOLEAN found;

    int numReady = m_readyQueue.get_count(); 

    if (numReady == 0) {

	found = e_FALSE;

    } else {

	FullReadyInterpreter *fullReadyInterpreter =
	    (FullReadyInterpreter *) m_readyQueue.dequeue();
	pid_t pid = fullReadyInterpreter -> getPid();
	assert (fullReadyInterpreter != NULL);

	int fd;
	FullReadyInterpreter::Errors errorCode = 
	    fullReadyInterpreter -> getFd (fd);
	assert (errorCode == FullReadyInterpreter::e_Ok);

	if (fullReadyInterpreter -> hasFreeFd()) {
	    m_readyQueue.enqueue ((void *) fullReadyInterpreter);
	} else {
	    m_usedQueue.enqueue ((void *) fullReadyInterpreter);
	}

	readyInterpreter = HotInterpreters::ReadyInterpreter (pid, fd);
	found = e_TRUE;
    }

    return (found);
}

BOOLEAN InterpreterQueue::findAndRemoveReadyInterpreter (pid_t pid, FullReadyInterpreter *&fullReadyInterpreter)
{
    BOOLEAN found = e_FALSE;
    int numReady  = m_readyQueue.get_count();
    int numUsed   = m_usedQueue.get_count();

#ifdef FIX_LATER
	// scanning the queues to find the process id is fast for those
	// interpreters that allow multiple agents per interpreter, but slow
	// for those interpreters that allow only one agent per interpreter
	// (the first case involves an order of magnitude fewer interpreters
	// than the second case to handle the same number of agents)
#endif

    for (int i = 0; ((i < numReady) && (!found)); ++i) {

	fullReadyInterpreter =
	    (FullReadyInterpreter *) m_readyQueue.dequeue();
	assert (fullReadyInterpreter != NULL);

	if (fullReadyInterpreter -> getPid() != pid) {
	    m_readyQueue.enqueue ((void *) fullReadyInterpreter);
	} else {
	    found = e_TRUE;
	}
    }

    for (int j = 0; ((j < numUsed) && (!found)); ++j) {

	fullReadyInterpreter =
	    (FullReadyInterpreter *) m_usedQueue.dequeue();
	assert (fullReadyInterpreter != NULL);

	if (fullReadyInterpreter -> getPid() != pid) {
	    m_usedQueue.enqueue ((void *) fullReadyInterpreter);
	} else {
	    found = e_TRUE;
	}
    }

    return (found);
}

InterpreterQueue::~InterpreterQueue (void)
{
    int numReady  = m_readyQueue.get_count();
    int numUsed   = m_usedQueue.get_count();

    for (int i = 0; i < numReady; ++i) {
	FullReadyInterpreter *fullReadyInterpreter =
	    (FullReadyInterpreter *) m_readyQueue.dequeue();
	assert (fullReadyInterpreter != NULL);
	DELETE_OBJECT(fullReadyInterpreter);
    }

    for (int i = 0; i < numUsed; ++i) {
	FullReadyInterpreter *fullReadyInterpreter =
	    (FullReadyInterpreter *) m_usedQueue.dequeue();
	assert (fullReadyInterpreter != NULL);
	DELETE_OBJECT(fullReadyInterpreter);
    }
}

/*
 * HotInterpretersImp contains the implementation-specific data members of
 * HotInterpreters.
 */

class HotInterpretersImp  
{
    friend class HotInterpreters;
    friend class HotInterpreterBody;

/*
 * constants
 */

        // default number of slots in the interpreter array

    enum DefaultArraySlots {
	e_DefaultArraySlots = 8
    };

	// number of slots in the hash tables

    enum InterpreterHashSlots {
	e_InterpreterHashSlots = 16
    };

    enum ProcessHashSlots {
	e_ProcessHashSlots = 128
    };

/*
 * data
 */

	// array of interpreter queues, number of slots in the array, number
	// of full slots (i.e., number of available interpreters) and
	// condition variables for signaling that a new interpreter process
	// has been added to one of the queues

#ifdef FIX_LATER
	// m_queues, m_queueConditionVariables and m_waitingForQueue should
	// be combined into an array of structures
#endif

    InterpreterQueue **m_queues;
    int                m_numSlots;
    int                m_numInterpreters;
    UINT_32           *m_queueConditionVariables;
    int               *m_waitingForQueue;

	// hash table that maps the symbolic names to the InterpreterQueue 
	// structures (that are in the array m_queues above -- the table
	// contents are indices into m_queues). 

    Hash m_interpreterTable;

	// monitor lock and condition variable -- m_freeSlotOrDeath 
	// is signaled whenever a new interpreter needs to be execed or an
	// existing interpreter has exited.  The condition variables
	// in m_queueConditionVariables above are signaled whenever a new
	// interpreter has been added to the queue.  The condition variables
	// inside the InterpreterStatus instances are signaled when an
	// interpreter process has died.

    MonitorLock m_monitor;
    UINT_32 m_freeSlotOrDeath;
    int m_waitingForFreeSlotOrDeath;

	// hash table that maps the interpreter process ids to their 
	// current status (the table contents are InterpreterStatus instances),
	// and a flag that indicates whether an interpreter process has 
	// finished

    Hash m_processTable;
    BOOLEAN m_processDone;

	// e_TRUE if the creation thread is already running

    BOOLEAN m_threadRunning;

/*
 * methods
 */

	// check for dead interpreters -- The procedure returns e_TRUE if
	// one of the dead interpreters was in the ready queues (i.e., if we
	// need to create another interpreter).

#ifdef FIX_LATER
	// The procedure currently returns e_TRUE no matter whether the
	// interpreter was in the ready queue or the used queue.  This
	// leads to some unecessary checking in the interpreter-creation
	// thread.
#endif

    BOOLEAN checkForDeadInterpreters (void);

	// constructor and destructor

    HotInterpretersImp (void);
   ~HotInterpretersImp ();
};

HotInterpretersImp::HotInterpretersImp (void):
    m_queues	                (NULL),
    m_numSlots                  (e_DefaultArraySlots),
    m_numInterpreters           (0),
    m_queueConditionVariables   (NULL),
    m_waitingForQueue		(NULL),
    m_interpreterTable          (e_InterpreterHashSlots),
    m_waitingForFreeSlotOrDeath (0),
    m_processTable              (e_ProcessHashSlots),
    m_processDone               (e_FALSE),
    m_threadRunning             (e_FALSE)
{
	// interpreter queues

    m_queues                  = new InterpreterQueue* [m_numSlots];
    m_queueConditionVariables = new UINT_32           [m_numSlots];
    m_waitingForQueue         = new int               [m_numSlots];

    for (int i = 0; i < m_numSlots; ++i) {
	m_queues                 [i] = (InterpreterQueue *) NULL;
        m_queueConditionVariables[i] = 0;
	m_waitingForQueue        [i] = 0;
    }

	// main condition variable

    (void) m_monitor.getConditionVariable (m_freeSlotOrDeath);
}

BOOLEAN HotInterpretersImp::checkForDeadInterpreters (void)
{
    pid_t pid;
    BOOLEAN checkAgain = e_FALSE;

    do {

	WAIT_STATUS_TYPE exitCode;

	    // get the process id of the exiting interpreter
 
	pid = waitpid (0, (int *) &exitCode, WNOHANG);

	if (pid > 0) {

		// remove the interpreter process from the queues no
		// matter what (since if the intrepreter process has
		// exited, it can not exactly handle new agents)

#ifdef FIX_LATER
		// checking all the queues is inefficient.  Instead the 
		// InterpeterStatus instance should contain some kind of
		// "pointer" to the queue location.
#endif

	    FullReadyInterpreter *fullReadyInterpreter;

	    for (int i = 0; i < m_numInterpreters; ++i) {
		if (m_queues[i] -> findAndRemoveReadyInterpreter (pid, fullReadyInterpreter)) {
		    assert (fullReadyInterpreter != NULL);
		    fullReadyInterpreter -> closeFreeFds();
		    DELETE_OBJECT(fullReadyInterpreter);
		    checkAgain = e_TRUE;
		    break;
		}
	    }

		// I used to have the assertion
		//
		//    assert (checkAgain);
		//
		// here.  However this assertion contradicts my intentional
		// design, namely, the HotInterpreter class will not accept
		// back an interpreter if there are too many free interpreters.
		// In this case the interpreter that has "died" is not in the
		// ReadyInterpreter table.  It is still in the process table,
		// however, so we look in the process table as before.

		// lookup the process id in the process table

	    HashNode *node = m_processTable.lookup ((UINT_32) pid, NULL);
	    assert (node != NULL);
	    InterpreterStatus *status = ((InterpreterStatus *) (node -> data));
	    assert (status != NULL);

		// if the interpreter has been handed off to one or more 
		// clients, we record the exit code and wakeup any clients 
		// that are waiting for the interpreter to exit; otherwise 
		// we remove the interpreter from the process table

	    if (status -> isInUse()) {
	        status -> setExitCode (exitCode);
		status -> wakeupWaiter();
	    } else {
		DELETE_OBJECT(status);
		m_processTable.remove ((UINT_32) pid, NULL);
	    }
	}

    } while ((pid > 0) || ((pid < 0) && ((errno == EINTR) || (errno == EAGAIN))));

    return (checkAgain);
}

HotInterpretersImp::~HotInterpretersImp ()      
{
    DELETE_ARRAY_OBJECT(m_queues);
    DELETE_ARRAY_OBJECT(m_queueConditionVariables);
    DELETE_ARRAY_OBJECT(m_waitingForQueue);
}

/*
 * HotInterpreterBody is the thread that actually creates the hot
 * interpreters.
 */

static void sigchld_handler (int, SignalHandlers::ClientData);

class HotInterpreterBody: public ThreadBody
{
    friend void sigchld_handler (int, SignalHandlers::ClientData);

	// interpreter data structures

#ifdef FIX_LATER
	/*
	 * We make m_imp static so that sigchld_handler below can find
	 * the condition variable that it needs to signal.  Thus we are
	 * assuming that only one HotInterpreters instance (and only one
	 * HotInterpreterBody instance) will be ever be created.  We will
	 * enforce this assumption later.  In addition, it would be useful 
	 * to have an InterruptTrap superclass so that we can easily direct 
	 * an interrupt to a particular class instance.
	 */
#endif

    static HotInterpretersImp *m_imp;

public:

	// constructor and destructor

    HotInterpreterBody (HotInterpretersImp *hotInterpreterImp);
   ~HotInterpreterBody ();

	// main thread that creates the interpreter processes and  
	// waits for interpreter deaths

    void run (void);

	// wakeup the main thread if an interpreter has died

    static void wakeupOnChildDeath (void);
};

HotInterpretersImp *HotInterpreterBody::m_imp = NULL;

HotInterpreterBody::HotInterpreterBody (HotInterpretersImp *imp)
{
    m_imp = imp;
}

HotInterpreterBody::~HotInterpreterBody () 
{
    DELETE_OBJECT(m_imp);
    m_imp = NULL;
}

static void cleanupFds (int *fds, int *fdsPtr, int fdSlot)
{
    for (int j = 0; j < fdSlot; ++j) {
	PipeUtility::closePipe (fdsPtr[j]);
    }

    if (fdsPtr != fds) {
	DELETE_ARRAY_OBJECT(fdsPtr);
    }
}

static int execAgentInterpreter (InterpreterQueue &queue, FullReadyInterpreter *&fullReadyInterpreter)
{ 
    pid_t pid;
    int fds[2];
    int *fdsPtr;

	/* determine how many socket pairs we need */

    int agentsPerInterp = queue.getAgentsPerInterp();

	/* create the socket pairs */

    if (agentsPerInterp == 1) {
	fdsPtr = fds;
    } else {
	fdsPtr = new int [2 * agentsPerInterp];
    }
  
    int i      = 0;
    int fdSlot = 0; 

    while (i < agentsPerInterp) {

	if (PipeUtility::createStreamPipe (&fdsPtr[fdSlot]) != PipeUtility::e_OK) {
	    cleanupFds (fds, fdsPtr, fdSlot);
	    g_serverData -> errorLog.error_sys_cont
		("unable to create a pipe");
	    return (-1);
	}

	i      += 1;
	fdSlot += 2;
    }

	/* construct the environment variable */

    i          = 0;
    fdSlot     = 0;
    int length = 0;

    while (i < agentsPerInterp) {
	length += fastLengthAsAscii (fdsPtr[fdSlot+1]) + 1;
	i      += 1;
	fdSlot += 2;
    }

    char *temp = new char [length + 11];
    SmartCharPtr tempDeletor (temp);
    strcpy (temp, "CONNECTION=");

    i           = 0;
    fdSlot      = 0;
    int tempPos = 11;

    while (i < agentsPerInterp) {

	fastIntegerToAscii (fdsPtr[fdSlot+1], &temp[tempPos]);
	tempPos += fastLengthAsAscii (fdsPtr[fdSlot+1]);
	fdSlot  += 2;
        i       += 1;

	if (i < agentsPerInterp) {
	    temp[tempPos] = ' ';
	    tempPos += 1;
	}
    }

	/* fork the process that will become the interpreter -- first    */
	/* We lock the FileTable since we are going to use the FileTable */
	/* in the forked child (and hence the FIleTable must be in a     */
	/* consistent state).  Note that we unlock the FileTable in both */
	/* the parent and child.                                         */

    FileTable::lockTable();
    pid = fork();
    FileTable::unlockTable();

	/* FORK FAILURE */
 
    if (pid < 0) {
	cleanupFds (fds, fdsPtr, fdSlot);
	g_serverData -> errorLog.error_sys_cont
	    ("unable to fork child process");
	return (-1);
    }

	/* IF CHILD PROCESS */

    if (pid == 0) {

	    /* put the CONNECTION entry in the environ list */

	g_serverData -> agentEnviron[g_serverData -> connectionEnvSlot] = temp; 

	    /* close every file descriptor except for the child's    */
	    /* end of the pipe (and stdin, stdout and stderr, which  */
	    /* are connected to either /dev/null or to the terminal, */
	    /* depending on whether the server is currently running  */
	    /* as a daemon)                                          */

#define DEFAULT_NUM_FDS 4
	int fdsToKeep[DEFAULT_NUM_FDS];
#undef DEFAULT_NUM_FDS
	int *fdsToKeepPtr;
	
	if (agentsPerInterp == 1) {
	    fdsToKeepPtr = fdsToKeep;
	} else {
	    fdsToKeepPtr = new int [agentsPerInterp + 3];
	}

	fdsToKeepPtr[0] = 0;
	fdsToKeepPtr[1] = 1;
	fdsToKeepPtr[2] = 2;

	fdSlot = 0;
	i      = 3;

	while (i < agentsPerInterp + 3) {
	    fdsToKeepPtr[i] = fdsPtr[fdSlot+1];
	    fdSlot += 2;
	    i      += 1;
	}

	FileTable::closeAllExcept (agentsPerInterp + 3, fdsToKeepPtr);

	if (fdsToKeepPtr != fdsToKeep) {
	    DELETE_ARRAY_OBJECT(fdsToKeepPtr);
	}

	    /* get rid of any pending signals -- this is VERY important */
	    /* since signals are the basis for all mutual exclusion     */
	    /* within the Tcl interpreter                               */

#ifdef FIX_LATER
	    /* this should be moved into the SignalHandler class */
#endif

	struct sigaction act;
#ifdef SIGACTION_CAST
	act.sa_handler = (void (*)(...)) SIG_DFL;
#endif
	sigemptyset (&act.sa_mask);
	act.sa_flags = 0;
	
	sigaction (SIGIO, &act, 0);
	sigaction (SIGALRM, &act, 0);
	sigaction (SIGPROF, &act, 0);
	sigaction (SIGHUP, &act, 0);
	sigaction (SIGTERM, &act, 0);
	sigaction (SIGSEGV, &act, 0);
#ifdef SIGBUS
	sigaction (SIGBUS, &act, 0);
#endif
#ifdef SIGABORT
	sigaction (SIGABORT, &act, 0);
#endif
#ifdef SIGABRT
	sigaction (SIGABRT, &act, 0);
#endif
	sigaction (SIGCHLD, &act, 0);
	    
	    /* reset the signal mask just to be safe */

	sigset_t emptySet;
	sigemptyset (&emptySet);
	sigprocmask (SIG_SETMASK, &emptySet, NULL);

	    /* do the exec */

	char **argv              = queue.getInterpreterArgv();
	DynamicString executable = queue.getInterpreterExecutable();

	if (execve (executable.value(), argv, g_serverData -> agentEnviron) < 0) {
	    exit (1);
	}

	abort_with_message ("should never get here");
    }

	// ELSE PARENT PROCESS

	// close the fds that connect the interpreter TO the server

    i      = 0;
    fdSlot = 1;

    while (i < agentsPerInterp) {
	PipeUtility::closePipe (fdsPtr[fdSlot]);
	i += 1;
	fdSlot += 2;
    }

	// put the fds that connect the server TO the interpreter into
	// a single array section

    i      = 1;
    fdSlot = 2;

    while (i < agentsPerInterp) {
	fdsPtr[i] = fdsPtr[fdSlot];
	i += 1;
	fdSlot += 2;
    }

	// remember the process id and fds

    fullReadyInterpreter = 
	new FullReadyInterpreter (pid, agentsPerInterp, fdsPtr);

	// free the fds array if we are not using the stack-allocated
	// two-element array

    if (fdsPtr != fds) {
	DELETE_ARRAY_OBJECT(fdsPtr);
    }

    return (0);
}

void HotInterpreterBody::run (void) 
{
#ifdef FIX_LATER
	/*
	 * Note that we can only have one SIGCHLD handler in the current
	 * system so we are assuming that only one HotInterpreterBody instance 
	 * will ever be created.  We will enforce this assumption later.
	 * In addition, it would be useful to have an InterruptTrap superclass 
	 * so that we can easily direct an interrupt to a particular class 
	 * instance.
	 */
#endif

	// install the SIGCHLD handler

    if (SignalHandlers::install_signal_intr 
	(SIGCHLD, (SignalHandlers::funcType) sigchld_handler, 0) < 0) { 

	char *messageArgv[] = {
	    "agentd: ERROR: unable to install handler for ",
	    (char *) SignalHandlers::SignoToString (SIGCHLD),
	    (char *) NULL
	};

	g_serverData -> errorLog.error_sys_quit (messageArgv);
    }

	// main loop -- create interpreters and watch for interpreter death

    hotPid = getpid();

    BOOLEAN checkAgain;
    Guard guard (m_imp -> m_monitor);

    while (1) {

	checkAgain = e_FALSE;

	    // create interpreters

	for (int i = 0; i < m_imp -> m_numInterpreters; ++i) {

	    InterpreterQueue &queue = *(m_imp -> m_queues[i]);

	    if (queue.needInterpreter()) {

		int rc;
		FullReadyInterpreter *fullReadyInterpreter;

		    /* create an interpreter -- release and reacquire the */
		    /* lock since interpreter creation is a lengthy       */
		    /* process                                            */
 
		m_imp -> m_monitor.release();
		rc = execAgentInterpreter (queue, fullReadyInterpreter);
		m_imp -> m_monitor.acquire();

		    /* update the tables */   

		if (rc < 0) {

#ifdef FIX_LATER
			// record that interpreter creation failed
#endif
		} else {

			// we better have a FullReadyInterpreter instance
			// if we did not get an error code

		    assert (fullReadyInterpreter != NULL);

			// add the interpreter to the interpreter queue

		    queue.addReadyInterpreter (fullReadyInterpreter);

			// add the process id to the process table

		    int there;
		    pid_t pid = fullReadyInterpreter -> getPid();
		    InterpreterStatus *status =
			new InterpreterStatus (m_imp -> m_monitor, pid);
		    m_imp -> m_processTable.add
			((UINT_32) pid, NULL, (void *) status, there);
		    assert (!there);

			// wakeup any waiters

		    if (m_imp -> m_waitingForQueue[i]) {
			m_imp -> m_monitor.wakeup
			    (m_imp -> m_queueConditionVariables[i]);
		    }

			// do not go to sleep if we do not have the full
			// complement of interpreters yet

		    if (queue.needInterpreter()) {
			checkAgain = e_TRUE;
		    }
		}
	    }
	}

	    // check for interpreter death

#ifdef CANT_BE_FIXED
	    /*
	     * Note that we are jumping through hoops to do the waitpid
	     * in the same thread that created the interpeter in the first
	     * place.  We do this because LinuxThreads, the most popular
	     * thread package for Linux (which is our main development
	     * platform), treats each thread as a separate process when
	     * delivering signals (such as SIGCHLD).  Thus the only thread 
	     * that can do a waitpid for a particular process is the thread
	     * that forked the process earlier.
	     */
#endif

	if (m_imp -> m_processDone) {

	    m_imp -> m_processDone = e_FALSE;

	    if (m_imp -> checkForDeadInterpreters()) {
		checkAgain = e_TRUE;
	    }
	}

	    // go to sleep if we know that we have enough interpreters

	if (!checkAgain) {
	    m_imp -> m_waitingForFreeSlotOrDeath += 1;
	    m_imp -> m_monitor.wait (m_imp -> m_freeSlotOrDeath);
	    m_imp -> m_waitingForFreeSlotOrDeath -= 1;
	}
    }
}

void HotInterpreterBody::wakeupOnChildDeath (void)
{

    if (sigchldPid == 0) {
	sigchldPid = getpid();
    }

    HotInterpreterBody::m_imp -> m_monitor.acquire();

    HotInterpreterBody::m_imp -> m_processDone = e_TRUE;

    if (m_imp -> m_waitingForFreeSlotOrDeath) {
	HotInterpreterBody::m_imp -> m_monitor.wakeup
	    (HotInterpreterBody::m_imp -> m_freeSlotOrDeath);
    }

    HotInterpreterBody::m_imp -> m_monitor.release();
}

static void sigchld_handler (int, SignalHandlers::ClientData)
{
    HotInterpreterBody::wakeupOnChildDeath ();
}

/*
 * constructor and destructor for HotInterpreters 
 */

HotInterpreters::HotInterpreters (void)
{
    m_imp = new HotInterpretersImp();
}

HotInterpreters::~HotInterpreters ()
{
#ifdef FIX_LATER
	// we need to do more to ensure proper thread shutdown
#endif

    DELETE_OBJECT(m_imp);
}

/*
 * methods for HotInterpreters
 */

HotInterpreters::ErrorCodes HotInterpreters::addInterpreter (const char *symbolicName, const INTERP &interp)
{
    HotInterpreters::ErrorCodes rc;

	// acquire the monitor

    m_imp -> m_monitor.acquire();

	// see if we already know about the interpeter; otherwise add the
	// intepreter to our tables

    if (m_imp -> m_interpreterTable.lookup (0, symbolicName) != NULL) {

	rc = HotInterpreters::e_AlreadyThere;

    } else {

	    // find a free slot in the array

	int freeSlot;

	if (m_imp -> m_numInterpreters < m_imp -> m_numSlots) {

	    freeSlot = m_imp -> m_numInterpreters;

	} else {

	    int numSlots             = m_imp -> m_numSlots << 1;
	    InterpreterQueue **slots = new InterpreterQueue * [numSlots];
	    UINT_32 *vars            = new UINT_32 [numSlots];
	    int *waiting             = new int [numSlots];

	    for (int i = 0; i < m_imp -> m_numSlots; ++i) {
		slots   [i] = m_imp -> m_queues[i];
	 	vars    [i] = m_imp -> m_queueConditionVariables[i];
		waiting [i] = m_imp -> m_waitingForQueue[i];
	    }

	    for (int j = m_imp -> m_numSlots; j < numSlots; ++j) {
		slots[j]   = NULL;
		vars [j]   = 0;
	 	waiting[j] = 0;
	    }

	    freeSlot = m_imp -> m_numSlots;

	    DELETE_OBJECT(m_imp -> m_queueConditionVariables);
	    DELETE_OBJECT(m_imp -> m_queues);
	    DELETE_OBJECT(m_imp -> m_waitingForQueue);
	    m_imp -> m_waitingForQueue = waiting;
	    m_imp -> m_queueConditionVariables = vars;
	    m_imp -> m_queues = slots;
            m_imp -> m_numSlots = numSlots;
	}

	    // add the interpeter to both the array and hash table

	int there;
	UINT_32 handle;
	m_imp -> m_queues[freeSlot] = new InterpreterQueue (interp);
	m_imp -> m_monitor.getConditionVariable (handle);
	m_imp -> m_queueConditionVariables[freeSlot] = handle;
	m_imp -> m_waitingForQueue[freeSlot] = 0;
	m_imp -> m_interpreterTable.add 
	    (0, symbolicName, (void *) freeSlot, there);
	assert (!there);
	m_imp -> m_numInterpreters += 1;
	rc = HotInterpreters::e_Ok;
    }

	// done

    m_imp -> m_monitor.release();

    if (rc == HotInterpreters::e_Ok) {
	m_imp -> m_monitor.wakeup (m_imp -> m_freeSlotOrDeath);
    }

    return (rc);
}

HotInterpreters::ErrorCodes HotInterpreters::getReadyInterpreter (const char *symbolicName, INTERP::InterpreterTypes type, HotInterpreters::ReadyInterpreter &readyInterpreter)
{
    int sendSignal = e_FALSE;
    HotInterpreters::ErrorCodes rc = e_Ok;

	// acquire the lock for mutual exclusion 

    m_imp -> m_monitor.acquire();

	// find the interpreter queue

    HashNode *node = m_imp -> m_interpreterTable.lookup (0, symbolicName);

    if (node == NULL) {

	rc = e_UnknownName;

    } else {

	int queueIndex = (int) (node -> data);
	InterpreterQueue &queue = *(m_imp -> m_queues[queueIndex]);
	
	    // make sure that the interpreter can handle state images (if needed)

	if ((type == INTERP::e_STATE_INTERP) &&
	    (queue.getInterpreterType() != INTERP::e_STATE_INTERP)) {

	    rc = e_WrongType;

	} else {

		// get a ready interpreter

	    while (!queue.getReadyInterpreter (readyInterpreter)) {
		m_imp -> m_waitingForQueue[queueIndex] += 1;
		m_imp -> m_monitor.wait
		    (m_imp->m_queueConditionVariables[queueIndex]);
		m_imp -> m_waitingForQueue[queueIndex] -= 1;
	    }

		// mark that we are handing the interpreter off to a client

	    HashNode *node = m_imp -> m_processTable.lookup 
	 	((UINT_32) readyInterpreter.pid, NULL);
	    assert (node != NULL);
	    InterpreterStatus *status = (InterpreterStatus *) (node -> data);
	    status -> markInUse();

		// see if we need to wakeup the interpreter-creation thread

	    sendSignal = 
		queue.needInterpreter() && m_imp -> m_waitingForFreeSlotOrDeath;
	 
		// everything is okay

	    rc = e_Ok;
	}
    }

	// wakeup the interpreter creation thread if we have just taken an 
	// interpreter out of the queues (and the interpreter creation thread 
	// is asleep), and then release the lock

    if (sendSignal) {
	m_imp -> m_monitor.wakeup (m_imp -> m_freeSlotOrDeath);
    }

    m_imp -> m_monitor.release();
    return (rc);
}

HotInterpreters::ErrorCodes HotInterpreters::returnReadyInterpreter (const char *symbolicName, const HotInterpreters::ReadyInterpreter &readyInterpreter)
{
    HotInterpreters::ErrorCodes rc = e_Ok;

	// acquire the lock for mutual exclusion 

    m_imp -> m_monitor.acquire();

	// lookup (1) the process id in the process table and (2) the
	// interpreter type in the interpreter table; if either could not
	// be found, we have an error

    HashNode *processNode =
	m_imp -> m_processTable.lookup ((UINT_32) readyInterpreter.pid, NULL);
    HashNode *queueNode = 
	m_imp -> m_interpreterTable.lookup (0, symbolicName);

    if (processNode == NULL) {

	rc = e_InvalidPid;

    } else if (queueNode == NULL) {

	rc = e_WrongType;

    } else {

	    // get the interpreter queue and status out of the hash-table nodes

	int queueIndex = (int) (queueNode -> data);
	InterpreterQueue &queue = *(m_imp -> m_queues[queueIndex]);
	InterpreterStatus &status = 
	    * ((InterpreterStatus *) (processNode -> data));

	    // if the interpreter has not exited, try to return it to the
	    // queue

	if (status.hasExited()) {
	    rc = e_HasExited;
	} else if (!queue.returnReadyInterpreter (readyInterpreter)) {	
	    rc = e_TooMany;
	} else {
	    rc = e_Ok;
	}

	    // on success, change the interpreter's status in the process table
	    // and wakeup any waiters

	if (rc == e_Ok) {

	    status.markReturned();

	    if (m_imp -> m_waitingForQueue[queueIndex]) {
		m_imp -> m_monitor.wakeup
		    (m_imp -> m_queueConditionVariables[queueIndex]);
	    }
	}
    }

	// release the lock and return

    m_imp -> m_monitor.release();
    return (rc);
}

HotInterpreters::ErrorCodes HotInterpreters::waitForInterpreter (pid_t pid, WAIT_STATUS_TYPE &exitCode)
{
    HotInterpreters::ErrorCodes rc;

	// acquire the lock and lookup the process

    m_imp -> m_monitor.acquire();
    HashNode *node = m_imp -> m_processTable.lookup ((UINT_32) pid, NULL);

    if (node == NULL) {

	rc = e_InvalidPid;

    } else {

	    // if we found the process, wait until it is finished (if
	    // it has not finished already), then remember the exit code
	    // and remove the process from the process table

	InterpreterStatus *status = (InterpreterStatus *) (node -> data);
	status -> waitForInterpreterDeath();
	exitCode = status -> getExitCode();
	status -> markReturned();

	if (!(status -> isInUse())) {
	    m_imp -> m_processTable.remove ((UINT_32) pid, NULL);
	    DELETE_OBJECT(status);
	}

	rc = e_Ok;
    }

	// release the lock and done

    m_imp -> m_monitor.release();
    return (rc);
}

void HotInterpreters::startCreationThread (void)
{
	// remember that we are starting up the thread

    m_imp -> m_monitor.acquire();
    BOOLEAN threadRunning = m_imp -> m_threadRunning;
    m_imp -> m_threadRunning = e_TRUE;
    m_imp -> m_monitor.release();

	// start the thread if necessary

    if (!threadRunning) {
	HotInterpreterBody *hotInterpreterBody = new HotInterpreterBody (m_imp);
	Thread thread ("hot-interpreter", Thread::e_DAEMON_THREAD, hotInterpreterBody);
	thread.resume ();
    }
}
