/* Agent Tcl
   Bob Gray
   26 May 1998

   hotInterpreter.h

   This file defines the HotInterpreter class, which starts up a pool of
   interpreters so that an interpreter will be available immediately when
   an agent arrives.

   Copyright (c) 1995-1998, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef _HOT_INTERPRETER_H
#define _HOT_INTERPRETER_H

#ifndef NO_PRAGMAS
#pragma interface
#endif

#ifndef _PLAT_PORTING_H
#include "platPorting.h"
#endif
#ifndef _PLAT_SIZES_H
#include "platSizes.h"
#endif
#ifndef _INTERP_H
#include "interp.h"
#endif

class HotInterpretersImp;
class INTERP;

class HotInterpreters
{
	// implementation details

    HotInterpretersImp *m_imp;

	// copy and assignment are not allowed

     HotInterpreters (const HotInterpreters &);
     HotInterpreters& operator= (const HotInterpreters &);

public:

	// error codes

    enum ErrorCodes {
	e_Ok		= 0x0,
	e_InvalidPid	= 0x1,
	e_UnknownName	= 0x2,
	e_NoInterpreter	= 0x3,
	e_AlreadyThere  = 0x4,
	e_WrongType     = 0x5,
	e_TooMany       = 0x6,
	e_HasExited	= 0x7
    };

	// process id and file descriptor of a ready interpreter

    struct ReadyInterpreter {

	pid_t pid;			// process id
	int fd;				// file descriptor

	ReadyInterpreter (void):
	    pid (0),
	    fd (0)
	{
	    // empty
	}

	ReadyInterpreter (pid_t p, int f):
	    pid (p),
	    fd (f)
	{
	    // empty
	}

	ReadyInterpreter (const ReadyInterpreter &readyInterpreter):
	    pid (readyInterpreter.pid),
	    fd  (readyInterpreter.fd)
	{
	    // empty
	}

	ReadyInterpreter& operator= (const ReadyInterpreter &readyInterpreter) {

	    if (this != &readyInterpreter) {
		pid = readyInterpreter.pid;
		fd  = readyInterpreter.fd;
	    }

	    return (*this);
	}
    };

	/* 
	 * get a ready interpreter 
	 *
	 * symbolicName = symbolic name of the desired interpreter
	 *                (e.g., STATE-TCL for a Tcl interpeter that can
	 *                handle state capture)
         *
	 * type         = type of the desired interpreter
	 *                (e.g., e_STATE_INTERP for an interpreter that
	 *                can accept state images, and e_NORMAL_INTERP
	 *                for an interpreter that can accept "object" state 
	 *                only).
	 * 
	 * The function returns one of the following error codes:
	 *
	 * e_Ok            = No error
	 * e_UnknownName   = The given symbolic name does not correspond to a
	 *                   known interpreter.
	 * e_WrongType     = The interpreter with the given name is not of
	 *                   the needed type.
         * e_NoInterpreter = Interpreters are temporary unavailable (perhaps
	 *                   because there are too many processes running on
	 *                   the machine and no more child processes can be
	 *                   forked).
	 */

    ErrorCodes getReadyInterpreter 
	(const char *symbolicName, 
	 INTERP::InterpreterTypes type, 
	 ReadyInterpreter &readyInterpeter);

	/*
	 * return a ready interpreter
	 *
	 * symbolicName     = symbolic name of the interpreter
	 *                    (e.g., STATE-TCL for a Tcl interpeter that can
	 *                    handle state capture)
         *
	 * readyInterpreter = process id of the interpreter plus the socket
	 *                    descriptor that the server uses to communicate
	 *                    with the interpreter
	 *
         * The function returns one of the following error codes:
	 *
	 * e_Ok            = No error
	 * e_UnknownName   = The given symbolic name does not correspond to a
	 *                   known interpreter.
	 * e_TooMany       = There are too many ready interpreters.
	 * e_HasExited     = The interpreter has exited.
	 * e_InvalidPid    = There is no interpreter with the given process id.
	 */

    ErrorCodes returnReadyInterpreter
	(const char *symbolicName,
	 const ReadyInterpreter &readyInterpreter);

	/*
	 * wait for an interpreter to finish and exit
	 *
	 * pid = process id of the interpreter
	 *
	 * The function returns one of the following error codes:
	 *
	 * e_Ok         = No error
	 * e_InvalidPid = The given process id does not correspond to an
	 *                interpreter.
	 *
	 * On e_Ok exitStatus is set to the exit code from the process.
	 */

    ErrorCodes waitForInterpreter 
	(pid_t pid, WAIT_STATUS_TYPE &exitStatus);

	/*
	 * add an interpeter
	 *
	 * symbolicName = symbolic name of the new interperter
	 *                (e.g., STATE-TCL is a Tcl interpeter that can
	 *                handle state capture)
         *
	 * interp       = interpreter description
	 *
	 * The function returns one of the following error codes:
	 *
         * e_Ok		  = No error
	 * e_AlreadyThere = An interpreter with the given symbolic name has
	 *                  already been added.
	 */

    ErrorCodes addInterpreter
	(const char *symbolicName, const INTERP &interp);

	/*
	 * start up the thread that actually creates the hot interpreters
	 */

    void startCreationThread (void);

	/*
	 * constructor and destructor
	 */

    HotInterpreters (void);
   ~HotInterpreters();
};

#endif
