/* Agent Tcl
   Bob Gray
   23 August 1995

   interp.h

   This file defines the routines that keep track of available interpreters.
*/

#ifndef _INTERP_H
#define _INTERP_H

#ifndef NO_PRAGMAS
#pragma interface
#endif

#ifndef _SUPP_DSTRING_H
#include "suppDString.h"		// DynamicString
#endif
#ifndef _SUPP_DSTRING_QUEUE_H
#include "genDStringQueue.h"	// DynamicStringQueue
#endif

  /* struct INTERP holds the information on one interpreter */

class INTERP
{
public:

    enum InterpreterTypes {
	e_STATE_INTERP  = 0,		/* accepts state images */
	e_NORMAL_INTERP = 1		/* accepts scripts only */
    };

    enum DefaultHotCount {
	e_DEFAULT_HOT_COUNT = 4		/* default number of hot interpreters */
    };

    enum DefaultAgentsPerInterp {
	e_DEFAULT_AGENTS_PER_INTERP = 1	/* default number of agents per interpreter */
    };

private:

    DynamicString m_name;           /* symbolic name for the interpreter                 */
    DynamicString m_executable;     /* executable file that starts the interpreter       */
    DynamicString m_argv0;          /* argv[0] that is passed to the executable          */
    DynamicStringQueue m_arguments; /* arguments to pass to the interpreter              */
    InterpreterTypes m_type;        /* e_STATE_INTERP or e_NORMAL_INTERP                 */
    char **m_argv;		    /* argv array for an execve call                     */
    BOOLEAN m_argvReady;            /* e_TRUE if the argv array is up-to-date            */
    int m_minHotCount;              /* minimum number of hot interpreters to have ready  */
    int m_maxHotCount;              /* maximum number of hot interpreters to have ready  */
    int m_agentsPerInterp;	    /* number of agents that the interpreter can execute */
				    /* simultaneously                                    */

private:

	/* fill in the argv array */

    void prepareArgv (void);

public:

	/* constructors and assignment operator */

    INTERP 
	(void);

    INTERP 
	(const INTERP &interp);

    INTERP 
	(DynamicString name, DynamicString executable, DynamicString argv0, 
	 InterpreterTypes type, 
	 int minHotCount, int maxHotCount, int agentsPerInterp);

    INTERP &operator= 
	(const INTERP &interp);
 
	/* access name, executable, argv0, type and hot count*/

    DynamicString getName (void) const {
	return (m_name);
    }

    DynamicString getExecutable (void) const {
	return (m_executable);
    }

    DynamicString getArgv0 (void) const {
	return (m_argv0);
    }

    InterpreterTypes getType (void) const {
	return (m_type);
    }

    int getMinHotCount (void) const {
	return (m_minHotCount);
    }

    int getMaxHotCount (void) const {
	return (m_maxHotCount);
    }

    int getAgentsPerInterp (void) const {
	return (m_agentsPerInterp);
    }

    void setName (DynamicString name) {
	m_name = name;
    }

    void setExecutable (DynamicString executable) {
	m_executable = executable;
    }

    void setArgv0 (DynamicString argv0) {
	m_argv0 = argv0;
    }

    void setType (InterpreterTypes type) {
	m_type = type;
    }

    void setMinHotCount (int minHotCount) {
	m_minHotCount = minHotCount;
    }

    void setMaxHotCount (int maxHotCount) {
	m_maxHotCount = maxHotCount;
    }

    void setAgentsPerInterp (int agentsPerInterp) {
	m_agentsPerInterp = agentsPerInterp;
    }

	/* access the arguments */

    int getArgumentCount (void) const {
	return (m_arguments.get_count());
    }

    DynamicString getArgument (int i) const {
	return (m_arguments.peek(i));
    }

    void addArgument (DynamicString arg) {
	m_arguments.enqueue (arg);
	m_argvReady = e_FALSE;
    }

	/* access argv */

    char **getArgv (void) {
	
	if (m_argvReady == e_FALSE) {
	    prepareArgv ();
	}

	return (m_argv);
    }
	
	/* destructor */

   ~INTERP ();
};



#endif
