/* Agent Tcl
   Bob Gray
   2 March 1998

   messageBuffers.h

   This file defines the class and thread 

   Copyright (c) 1995-1998, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "platDelete.h"
#include "platExclusion.h"	// MonitorLock
#include "mesgPort.h"		// RemotePort
#include "agentId.h"		// INCOMING_MESSAGE
#include "genFile.h"		
#include "genQueue.h"		// QUEUE
#include "genMessage.h"		// MESSAGE
#include "genTransmit.h"
#include "messageBuffers.h"	// IncomingBody
#include "truefalse.h"		// BOOLEAN

struct IncomingBodyImp
{
    int m_sockfd;		   // socket descriptor of the connection
    MonitorLock m_sigioMonitor;	   // lock for the connection
    MonitorLock m_bufferMonitor;   // lock for the buffers
    BOOLEAN m_isSockfdAvailable;   // e_TRUE if socket descriptor is available
    BOOLEAN m_isAgentDone;	   // e_TRUE if the agent is done
    QUEUE m_messages;		   // queue of incoming messages
    QUEUE m_meetings;		   // queue of incoming meetings
    QUEUE m_events;		   // queue of incoming events
    UINT_32 m_items;		   // total number of queued items
};

IncomingBody::IncomingBody (void)
{
    m_this                        = new IncomingBodyImp ();
    m_this -> m_sockfd            = 0; 
    m_this -> m_isSockfdAvailable = e_FALSE;
    m_this -> m_isAgentDone       = e_FALSE;
    m_this -> m_items             = 0;
}

IncomingBody::IncomingBody (int sockfd, const MonitorLock &sigioMonitor)
{
#ifdef FIX_LATER
	// We dup the socket descriptor so that the socket will not be closed
	// until this thread finishes.  We should add some kind of 
	// synchronization so that we do not waste file descriptors.
#endif

    m_this                        = new IncomingBodyImp ();
#ifdef FIX_NOW
    m_this -> m_sockfd            = FileUtility::dup (sockfd);
#else
    m_this -> m_sockfd		  = sockfd;
#endif
    m_this -> m_sigioMonitor      = sigioMonitor;
    m_this -> m_isSockfdAvailable = e_TRUE;
    m_this -> m_isAgentDone       = e_FALSE;
    m_this -> m_items             = 0;
}

void IncomingBody::setBackgroundConnection (int sockfd, const MonitorLock &sigioMonitor)
{
#ifdef FIX_LATER
	// We dup the socket descriptor so that the socket will not be closed
	// until this thread finishes.  We should add some kind of 
	// synchronization so that we do not waste file descriptors.
#endif

    assert (m_this -> m_isSockfdAvailable == e_FALSE);
    Guard guard (m_this -> m_bufferMonitor);
#ifdef FIX_NOW
    m_this -> m_sockfd            = FileUtility::dup (sockfd);
#else
    m_this -> m_sockfd            = sockfd;
#endif
    m_this -> m_sigioMonitor      = sigioMonitor;
    m_this -> m_isSockfdAvailable = e_TRUE;
    m_this -> m_bufferMonitor.wakeup ();
}

void IncomingBody::incomingMessage (INCOMING_MESSAGE *message)
{
    Guard guard (m_this -> m_bufferMonitor);
    m_this -> m_messages.enqueue ((void *) message);
    m_this -> m_items += 1;
    m_this -> m_bufferMonitor.wakeup ();
}

void IncomingBody::incomingEvent (INCOMING_EVENT *event)
{
    Guard guard (m_this -> m_bufferMonitor);
    m_this -> m_events.enqueue ((void *) event);
    m_this -> m_items += 1;
    m_this -> m_bufferMonitor.wakeup ();
}

void IncomingBody::incomingMeeting (INCOMING_MEETING *meeting)
{
    Guard guard (m_this -> m_bufferMonitor);
    m_this -> m_meetings.enqueue ((void *) meeting);
    m_this -> m_items += 1;
    m_this -> m_bufferMonitor.wakeup ();
}

void IncomingBody::agentDone (void)
{
	// NOTE that we acquire the socket monitor too (m_sigioMonitor) so that
	// we can synchronize with the code that writes a message onto the 
	// socket (i.e., if m_isAgentDone is set, we do not want to do the
	// write, and if a write is in progress, we do not want to end the
	// agent quite yet).

	// NOTE that we must acquire bufferMonitor first and then sigioMonitor.
  	// If we acquire bufferMonitor second and then release bufferMonitor
	// first, the IncomingBody ready might wake up BEFORE sigioMonitor is
	// released and run to completion (which DELETES IncomingBody).

    m_this->m_bufferMonitor.acquire();
    m_this->m_sigioMonitor.acquire();

    m_this -> m_isAgentDone = e_TRUE;
    m_this -> m_bufferMonitor.wakeup ();

    m_this->m_sigioMonitor.release();
    m_this->m_bufferMonitor.release();
}

IncomingBody::~IncomingBody ()
{
	// delete the queued items

    INCOMING_MESSAGE *message;
    INCOMING_MEETING *meeting;
    INCOMING_EVENT   *event;

    while ((message = (INCOMING_MESSAGE *) m_this -> m_messages.dequeue ()) != NULL) {
	DELETE_OBJECT(message);
    }
 
    while ((meeting = (INCOMING_MEETING *) m_this -> m_meetings.dequeue ()) != NULL) {
	DELETE_OBJECT(meeting);
    }

    while ((event = (INCOMING_EVENT *) m_this -> m_events.dequeue ()) != NULL) {
	DELETE_OBJECT(event);
    }

	// close our copy of the socket descriptor

#ifdef FIX_NOW
    FileUtility::close (m_this -> m_sockfd);
#endif

	// delete the member data

    DELETE_OBJECT(m_this);
}

#ifdef FIX_LATER
    /*
     * s_sendEvent, s_sendMessage and s_sendMeeting should return an "abort"
     * code if they discover that the agent has ended (i.e., if they discover
     * that m_agentDone is set to true)
     */
#endif

static void s_sendEvent (IncomingBodyImp *m_this, int sockfd, MonitorLock &monitor, INCOMING_EVENT *event)
{
	/* assemble the response */

    MESSAGE response (messagesFromServer.messages[RESP_EVENT]);
    response.addLong (0, 0);
    response.addId (1, event -> id, e_FALSE); 
    response.addSecurity (2, event -> security, e_FALSE);
    response.addString (3, event -> tag, e_FALSE);
    response.addString (4, event -> string, e_FALSE);

	/* send the response down the socket */

    Guard guard (monitor);

    if (!(m_this -> m_isAgentDone)) {
	(void) message_send (sockfd, response);
    }
}

static void s_sendMeeting (IncomingBodyImp *m_this, int sockfd, MonitorLock &monitor, INCOMING_MEETING *meeting)
{
	/* assemble the response */

    MESSAGE response (messagesFromServer.messages[RESP_MEETING]);
    response.addLong (0, 0);
    response.addId (1, meeting -> id, e_FALSE); 
    response.addSecurity (2, meeting -> security, e_FALSE);
    response.addByte (3, meeting -> status);
    response.addLong (4, meeting -> localHandle);
    response.addLong (5, meeting -> remoteHandle);
    response.addString (6, meeting -> port -> machine, e_FALSE);
    response.addIp (7, meeting -> port -> ip);
    response.addLong (8, meeting -> port -> port);

	/* send the response down the socket */

    Guard guard (monitor);

    if (!(m_this -> m_isAgentDone)) {
	(void) message_send (sockfd, response);
    }
}

static void s_sendMessage (IncomingBodyImp *m_this, int sockfd, MonitorLock &monitor, INCOMING_MESSAGE *message)
{
        /* assemble the response */

    MESSAGE response (messagesFromServer.messages[RESP_MESSAGE]);
    response.addLong (0, 0); 
    response.addId (1, message -> id, e_FALSE);
    response.addSecurity (2, message -> security, e_FALSE);
    response.addLong (3, message -> code);
    response.addString (4, message -> string, e_FALSE);

	/* send the response down the socket */

    Guard guard (monitor);

    if (!(m_this -> m_isAgentDone)) {
	(void) message_send (sockfd, response);
    }
}

void IncomingBody::run (void)
{
    INCOMING_MESSAGE *message;
    INCOMING_EVENT   *event;
    INCOMING_MEETING *meeting;

    Guard guard (m_this -> m_bufferMonitor);

    while (1) {

	if (m_this -> m_isAgentDone) {
	    break;
	}

	if (m_this -> m_isSockfdAvailable) {

	    message = (INCOMING_MESSAGE *) m_this -> m_messages.dequeue();
	    meeting = (INCOMING_MEETING *) m_this -> m_meetings.dequeue();
	    event   = (INCOMING_EVENT   *) m_this -> m_events.dequeue();

	    if ((message != NULL) || (event != NULL) || (meeting != NULL)) {

	        if (message != NULL) {
		    m_this -> m_items -= 1;
		}

		if (event != NULL) {
		    m_this -> m_items -= 1;
		}

		if (meeting != NULL) {
		    m_this -> m_items -= 1;
		}

		int         sockfd  = m_this -> m_sockfd;
		MonitorLock monitor = m_this -> m_sigioMonitor;

	 	m_this -> m_bufferMonitor.release();

	 	if (message != NULL) {
		    s_sendMessage (m_this, sockfd, monitor, message);
		    DELETE_OBJECT(message);
		}

		if (event != NULL) {
		    s_sendEvent (m_this, sockfd, monitor, event);
		    DELETE_OBJECT(event);
		}

		if (meeting != NULL) {
		    s_sendMeeting (m_this, sockfd, monitor, meeting);
		    DELETE_OBJECT(meeting);
		}

		m_this -> m_bufferMonitor.acquire();	
		continue;
	    }
	}

	m_this -> m_bufferMonitor.wait();
    }
}
