/* Agent Tcl
   Bob Gray
   23 August 1995

   parse.cc

   This file implements the parsing routines.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "parse.h"
#include "suppDString.h"
#include "suppStrings.h"

/* PARSE::skip_comment

   Purpose: Skip over a comment
 
     Input: None

    Output: The procedure reads through the file until it hits a \n or an EOF.
*/

void PARSE::skip_comment (void)
{
    while ((ch != '\n') && (ch != EOF)) {
	ch = fgetc (fp);
    }
}

/* PARSE::PARSE and PARSE::~PARSE

   Purpose: These procedures are the constructor and destructor for class PARSE.

     Input: fp   = the file to be parsed
                   (FILE *)

            size = maximum word length
                   (int)
 
      Note: The constuctor assumes that the file has been opened for reading 
            and that no I/O operations have been performed on the file yet.
*/

PARSE::PARSE (FILE *p_fp, int p_size)
{
    setvbuf (p_fp, NULL, _IOFBF, 0);
    fp     = p_fp;
    size   = p_size;
    buffer = new char [p_size + 1];
    line   = 1;
    ch     = fgetc (p_fp);
}

PARSE::~PARSE ()
{
    delete (buffer);
}

/* PARSE::getNextLine

   Purpose: Get the next line of words from the file

     Input: None

    Output: The procedure returns -1 on error.  The most common error is
            a word being longer than the buffer size.  Otherwise the procedure
            returns 0, sets wordCount to the number of words on the next
            line, and sets words[0], ..., words[wordCount-1] to dynamically
            allocated strings that contain the words.
*/

int PARSE::getNextLine (int maxWords, DynamicString*words, int& wordCount)
{
    int i = 0;

	/* no words yet */

    wordCount = 0;

	/* skip over whitespace */

    while ((ch == ' ') || (ch == '\t') || (ch == '\n') || (ch == '#')) {
	if (ch == '\n') {
	    line += 1;
	    ch = fgetc (fp);
	} else if (ch == '#') {
	    skip_comment ();
	} else {
	    ch = fgetc (fp);
	}
    }

    if (ch == EOF) {
	return (0);
    }
 
	/* grab the words */

    wordCount = 0;

    while (1) {

	    /* get the word */

        if (ch == '\"') {

	    ch = fgetc (fp);     // skip over the leading quote

	    while ((ch != EOF) && (ch != '\"') && (ch != '\n')) {
	    
	        if (i >= size) {
		    return (-1);
	        } else {
		    buffer[i++] = ch;
		}

	        ch = fgetc (fp);
	    }

	    if (ch != '\"') {
	        return (-1);
	    }

	    ch = fgetc (fp);     // skip over the terminating quote

        } else {

	    while ((ch != EOF) && (ch != ' ') && (ch != '\t') && (ch != '\n') && (ch != '#')) {

	        if (i >= size) {
		    return (-1); 
	        } else {
		    buffer[i++] = ch;
	        }

	        ch = fgetc (fp);
	    }
        }

	buffer[i] = '\0';

	    /* remember the word */

	if (wordCount >= maxWords) {
	    return (-1);
	}

	words[wordCount].empty ();
	words[wordCount].append (buffer);
	wordCount += 1;

	    /* reset i so that the next word goes at the front of the buffer */

	i = 0;

	    /* skip until next word or end of line */

        while ((ch == ' ') || (ch == '\t') || (ch == '#')) {
	    if (ch == '#') {
	        skip_comment ();
	    } else {
	        ch = fgetc (fp);
	    }
        }

	if ((ch == '\n') || (ch == EOF)) {
	    break;
	}
    }

    return (0);
}

/* PARSE::getNextWord
 
   Purpose: Get the next word in the file

     Input: None

    Output: The procedure returns -1 on error.  The most common error is
            a word being longer than the buffer size.  Otherwise the procedure
            returns 0 and sets *bufferPtr to NULL if there are no words left
            in the file or to a dynamically allocated string that contains the
            next word.
*/

int PARSE::getNextWord (char *&bufferPtr)
{
    int i = 0;

	/* skip over whitespace */

    while ((ch == ' ') || (ch == '\t') || (ch == '\n') || (ch == '#')) {
	if (ch == '\n') {
	    line += 1;
	    ch = fgetc (fp);
	} else if (ch == '#') {
	    skip_comment ();
	} else {
	    ch = fgetc (fp);
	}
    }
  
	/* assemble the word */ 

    if (ch == '\"') {

	ch = fgetc (fp);     // skip over the leading quote

	while ((ch != EOF) && (ch != '\"') && (ch != '\n')) {
	    
	    if (i >= size) {
		return -1; 
	    } else {
		buffer[i] = ch;
		i += 1;
	    }

	    ch = fgetc (fp);
	}

	if (ch != '\"') {
	    return -2;
	} 

	ch = fgetc (fp);     // skip over the terminating quote

    } else {

	while ((ch != EOF) && (ch != ' ') && (ch != '\t') && (ch != '\n') && (ch != '#')) {

	    if (i >= size) {
		return -1; 
	    } else {
		buffer[i] = ch;
		i += 1;
	    }

	    ch = fgetc (fp);
	}
  } 

  buffer[i] = '\0';

    /* set the bufferPtr as appropriate */

  if (i == 0) {
    bufferPtr = NULL;
  } else {
    bufferPtr = strcpyWithAlloc (buffer);
  }

  return 0;
}
