/* Bob Gray
   Agent Tcl
   27 January 1996

   servConf.h

   This file implements the routines that load the server configuration files.

   Copyright (c) 1995-1997, Bob Gray, Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "platDelete.h"
#include "platSystem.h"
#include "suppStrings.h"
#include "agentd.h"
#include "mesgMachId.h"		// MachineId 
#include "suppDString.h"	// DynamicString 
#include "genDStringQueue.h"	// DynamicStringQueue
#include "genEncrypt.h"
#include "genFile.h"		// FileUtility 
#include "genManager.h"
#include "mesgTcpip.h"
#include "genPGP.h"
#include "hotInterpreter.h"
#include "interp.h"
#include "parse.h"
#include "noise.h"
#include "random.h"
#include "randpool.h"
#include "servConf.h"
#include "servRandom.h"		// RandomFile
#include "truefalse.h"

    /*
     * configuration limits -- 
     *
     * 1. must allow at least 16 agents at a time
     * 2. must enter at least 512 random bits
     * 3. must have at least 4 processes dedicated to non-creation requests
     */
 
const UINT_32 MINIMUM_AGENTS	= 16;
const UINT_32 MINIMUM_BITS	= 512;
const UINT_32 MINIMUM_PROCESSES = 4;

    /*
     * internal limits --
     *
     * 1. maximum number of arguments per field
     * 2. maximum size of a field name or argument (in bytes)
     * 3. number of buckets in the AllowedMachine hash table
     * 4. number of buckets in the interpreter hash table
     * 5. maximum wait time when attempting to determine the IP address of a machine
     */

const int MAXIMUM_ARGS 		= 2;
const int MAX_WORD_SIZE 	= 2048;
const int ACCESS_TABLE_SIZE 	= 0x020;
const int INTERP_TABLE_SIZE 	= 0x010;
const struct timeval IP_DELAY 	= { 20, 0 };
  
    // forward declarations

struct Field;
struct HandlerData;

    // ReloadAction specifies how the field is handled on a *reload*

enum ReloadAction {
    e_PRELOAD_ONCE,
    e_LOAD_ONCE,
    e_RELOAD
};

    // ArguemntType specifies the argument types for a field

enum ArgumentTypes {
    e_TAKES_ONE,
    e_TAKES_TWO
};

    // HandlerError is the return code from a field handler

enum HandlerError {
    e_OK,
    e_SOFT_FAILURE,
    e_HARD_FAILURE
};

    // handler function type

typedef HandlerError (*Handler) (const HandlerData&);
typedef HandlerError (*DefaultInitializer) (void);

    // more forward declarations

static HandlerError adminHandler (const HandlerData &);
static HandlerError ownerHandler (const HandlerData &);
static HandlerError passPhraseHandler (const HandlerData &);
static HandlerError unixSocketHandler (const HandlerData &);
static HandlerError tcpipSocketHandler (const HandlerData &);
static HandlerError interpFileHandler (const HandlerData &);
static HandlerError logFileHandler (const HandlerData &);
static HandlerError lockFileHandler (const HandlerData &);
static HandlerError pidFileHandler (const HandlerData &);
static HandlerError pgpProgHandler (const HandlerData &);
static HandlerError pgpPathHandler (const HandlerData &);
static HandlerError randomFileHandler (const HandlerData &);
static HandlerError randomBitsHandler (const HandlerData &);
static HandlerError userHandler (const HandlerData &);
static HandlerError groupHandler (const HandlerData &);
static HandlerError encryptionHandler (const HandlerData &);
static HandlerError remoteBeginHandler (const HandlerData &);
static HandlerError remoteCommHandler (const HandlerData &);
static HandlerError remoteForceHandler (const HandlerData &);
static HandlerError remoteInfoHandler (const HandlerData &);
static HandlerError maxAgentsHandler (const HandlerData &);
static HandlerError maxAnonAgentsHandler (const HandlerData &);
static HandlerError maxStatusHandler (const HandlerData &);
static HandlerError extraProcessesHandler (const HandlerData &);
static HandlerError allowSignedHandler (const HandlerData &);
static HandlerError allowedUserHandler (const HandlerData &);
static HandlerError allowedMachineHandler (const HandlerData &);
static HandlerError trustedMachineHandler (const HandlerData &);
static HandlerError reservedNameHandler (const HandlerData &);
static HandlerError startWallHandler (const HandlerData &);
static HandlerError startCpuHandler (const HandlerData &);
static HandlerError startJumpsHandler (const HandlerData &);
static HandlerError startChildrenHandler (const HandlerData &);
static HandlerError startDepthHandler (const HandlerData &);
static HandlerError anonStartWallHandler (const HandlerData &);
static HandlerError anonStartCpuHandler (const HandlerData &);
static HandlerError anonStartJumpsHandler (const HandlerData &);
static HandlerError anonStartChildrenHandler (const HandlerData &);
static HandlerError anonStartDepthHandler (const HandlerData &);
static HandlerError tempDirectoryHandler (const HandlerData &);
static HandlerError coredumpDirectoryHandler (const HandlerData &);
static HandlerError coredumpDirectoryInitializer (void);
static HandlerError timestampFileHandler (const HandlerData &);
static HandlerError logTimestampsHandler (const HandlerData &);

    // data passed to a handler

struct HandlerData
{
    char *field;   			       /* field that we are handling            */
    DynamicString value;       		       /* value of the first field              */
    DynamicString valueArgv[MAXIMUM_ARGS];     /* value of all the fields               */
    char *filename; 			       /* filename from which we got the field  */
    int line;             		       /* line number in that file              */
    ServerData::ReloadLocation reloadLocation; /* server piece that is reloading config */
};

    // configuration fields

struct Field
{
    char *name;		      // field name
    ArgumentTypes args;       // argument type
    ReloadAction reload;      // which part of the server needs to reload the field on SIGHUP
    BOOLEAN required;         // e_TRUE if this field is required
    Handler handler;          // handler for this field 
    BOOLEAN found;            // e_TRUE if this field was found
    DefaultInitializer initializer;  // initialize field to a default value
};

Field configurationFields[] = {
 
    {"TcpipSocket"	, e_TAKES_ONE	, e_LOAD_ONCE		, e_FALSE	, tcpipSocketHandler		, e_FALSE, (DefaultInitializer) NULL },
    {"UnixSocket"	, e_TAKES_ONE	, e_LOAD_ONCE		, e_FALSE	, unixSocketHandler		, e_FALSE, (DefaultInitializer) NULL },
    {"Encryption"	, e_TAKES_ONE	, e_RELOAD		, e_TRUE	, encryptionHandler		, e_FALSE, (DefaultInitializer) NULL },
    {"User"		, e_TAKES_ONE	, e_PRELOAD_ONCE		, e_TRUE	, userHandler			, e_FALSE, (DefaultInitializer) NULL },
    {"Group"		, e_TAKES_ONE	, e_PRELOAD_ONCE		, e_TRUE	, groupHandler			, e_FALSE, (DefaultInitializer) NULL },
    {"Owner"		, e_TAKES_ONE	, e_RELOAD		, e_TRUE	, ownerHandler			, e_FALSE, (DefaultInitializer) NULL },
    {"Admin"		, e_TAKES_ONE	, e_RELOAD		, e_TRUE	, adminHandler			, e_FALSE, (DefaultInitializer) NULL },
    {"PassPhrase"	, e_TAKES_ONE	, e_LOAD_ONCE		, e_TRUE	, passPhraseHandler		, e_FALSE, (DefaultInitializer) NULL },
    {"InterpFile"	, e_TAKES_ONE	, e_RELOAD	, e_TRUE	, interpFileHandler		, e_FALSE, (DefaultInitializer) NULL },
    {"LogFile"		, e_TAKES_ONE	, e_PRELOAD_ONCE	, e_TRUE	, logFileHandler		, e_FALSE, (DefaultInitializer) NULL },
    {"LockFile"		, e_TAKES_ONE	, e_LOAD_ONCE		, e_TRUE	, lockFileHandler		, e_FALSE, (DefaultInitializer) NULL },
    {"PidFile"		, e_TAKES_ONE	, e_LOAD_ONCE		, e_TRUE	, pidFileHandler		, e_FALSE, (DefaultInitializer) NULL },
    {"PGPPROG"		, e_TAKES_ONE	, e_RELOAD		, e_FALSE	, pgpProgHandler		, e_FALSE, (DefaultInitializer) NULL },
    {"PGPPATH"		, e_TAKES_ONE	, e_RELOAD		, e_FALSE	, pgpPathHandler		, e_FALSE, (DefaultInitializer) NULL },
    {"RandomFile"	, e_TAKES_ONE	, e_RELOAD	, e_TRUE	, randomFileHandler		, e_FALSE, (DefaultInitializer) NULL },
    {"RandomBits"	, e_TAKES_ONE	, e_RELOAD	, e_TRUE	, randomBitsHandler		, e_FALSE, (DefaultInitializer) NULL },
    {"MaxAgents"	, e_TAKES_ONE	, e_RELOAD		, e_TRUE	, maxAgentsHandler		, e_FALSE, (DefaultInitializer) NULL },
    {"MaxAnonAgents" 	, e_TAKES_ONE	, e_RELOAD		, e_TRUE	, maxAnonAgentsHandler		, e_FALSE, (DefaultInitializer) NULL },
    {"MaxStatus"	, e_TAKES_ONE	, e_RELOAD		, e_TRUE	, maxStatusHandler		, e_FALSE, (DefaultInitializer) NULL },
    {"ExtraProcesses"	, e_TAKES_ONE	, e_RELOAD		, e_TRUE	, extraProcessesHandler		, e_FALSE, (DefaultInitializer) NULL },
    {"RemoteBegin"	, e_TAKES_ONE	, e_RELOAD		, e_TRUE	, remoteBeginHandler		, e_FALSE, (DefaultInitializer) NULL },
    {"RemoteComm"	, e_TAKES_ONE	, e_RELOAD		, e_TRUE	, remoteCommHandler		, e_FALSE, (DefaultInitializer) NULL },
    {"RemoteForce"	, e_TAKES_ONE	, e_RELOAD		, e_TRUE	, remoteForceHandler		, e_FALSE, (DefaultInitializer) NULL },
    {"RemoteInfo"	, e_TAKES_ONE	, e_RELOAD		, e_TRUE	, remoteInfoHandler		, e_FALSE, (DefaultInitializer) NULL },
    {"AllowSigned"	, e_TAKES_ONE	, e_RELOAD		, e_TRUE	, allowSignedHandler		, e_FALSE, (DefaultInitializer) NULL },
    {"AllowedUser"	, e_TAKES_ONE	, e_RELOAD		, e_FALSE	, allowedUserHandler		, e_FALSE, (DefaultInitializer) NULL },
    {"AllowedMachine"	, e_TAKES_ONE	, e_RELOAD	, e_FALSE	, allowedMachineHandler 	, e_FALSE, (DefaultInitializer) NULL },
    {"TrustedMachine"	, e_TAKES_ONE	, e_RELOAD	, e_FALSE	, trustedMachineHandler		, e_FALSE, (DefaultInitializer) NULL },
    {"ReservedName" 	, e_TAKES_TWO	, e_RELOAD		, e_FALSE	, reservedNameHandler		, e_FALSE, (DefaultInitializer) NULL },
    {"StartWall"	, e_TAKES_ONE	, e_RELOAD		, e_TRUE	, startWallHandler		, e_FALSE, (DefaultInitializer) NULL },
    {"StartCpu"		, e_TAKES_ONE	, e_RELOAD		, e_TRUE	, startCpuHandler		, e_FALSE, (DefaultInitializer) NULL },
    {"StartJumps"	, e_TAKES_ONE	, e_RELOAD		, e_TRUE	, startJumpsHandler		, e_FALSE, (DefaultInitializer) NULL },
    {"StartChildren"	, e_TAKES_ONE	, e_RELOAD		, e_TRUE	, startChildrenHandler		, e_FALSE, (DefaultInitializer) NULL },
    {"StartDepth"	, e_TAKES_ONE	, e_RELOAD		, e_TRUE	, startDepthHandler		, e_FALSE, (DefaultInitializer) NULL },
    {"AnonStartWall"	, e_TAKES_ONE	, e_RELOAD		, e_TRUE	, anonStartWallHandler		, e_FALSE, (DefaultInitializer) NULL },
    {"AnonStartCpu"	, e_TAKES_ONE	, e_RELOAD		, e_TRUE	, anonStartCpuHandler		, e_FALSE, (DefaultInitializer) NULL },
    {"AnonStartJumps"	, e_TAKES_ONE	, e_RELOAD		, e_TRUE	, anonStartJumpsHandler		, e_FALSE, (DefaultInitializer) NULL },
    {"AnonStartChildren", e_TAKES_ONE	, e_RELOAD		, e_TRUE	, anonStartChildrenHandler	, e_FALSE, (DefaultInitializer) NULL },
    {"AnonStartDepth"	, e_TAKES_ONE	, e_RELOAD		, e_TRUE	, anonStartDepthHandler		, e_FALSE, (DefaultInitializer) NULL },
    {"TempDirectory"    , e_TAKES_ONE   , e_PRELOAD_ONCE        , e_TRUE        , tempDirectoryHandler          , e_FALSE, (DefaultInitializer) NULL },
    {"CoredumpDirectory", e_TAKES_ONE   , e_PRELOAD_ONCE        , e_TRUE        , coredumpDirectoryHandler      , e_FALSE, coredumpDirectoryInitializer},
    {"TimestampFile"  	  , e_TAKES_ONE , e_LOAD_ONCE	, e_FALSE	, timestampFileHandler	, e_FALSE, (DefaultInitializer) NULL },
    {"LogTimestamps"      , e_TAKES_ONE , e_LOAD_ONCE	, e_FALSE	, logTimestampsHandler		, e_FALSE, (DefaultInitializer) NULL },
    {(char *) NULL	, e_TAKES_ONE	, e_LOAD_ONCE		, e_FALSE	, NULL				, e_FALSE, (DefaultInitializer) NULL }
};

static char *reloadLocationToString (ServerData::ReloadLocation reloadLocation)
{
    char *string = (char *) NULL;

    switch (reloadLocation) {

	case ServerData::e_INITIAL_LOAD: 
	    string = "initial";
	    break;

	case ServerData::e_RELOADING:
	    string = "reload";
	    break;

	case ServerData::e_PRELOAD:
	    string = "preload";
	    break;
    }

    return (string);
}

static void logReadError (char *filename, int line, ServerData::ReloadLocation reloadLocation)
{
    char temp[16];

	/* assertions */

    assert (filename != NULL);

	/* log the error */
 
    sprintf (temp, "%d", line);

    char *argv[] = {
	"loadConfig: ",
	reloadLocationToString(reloadLocation),
	": ERROR: unable to read configuration file \"",
	filename, "\" (line ", temp,
	"): missing close-quote or excessive word length", ((char *) NULL)
    };

    g_serverData -> errorLog.error_app_cont (argv);
}

static HandlerError handleIPValue (ServerData::ReloadLocation reloadLocation, const DynamicString &machine, UINT_32 &ip)
{

	/* setup */

    MachineId machineId (machine);
    struct timeval stop = TimevalUtil::addTimevals (SystemUtil::getCurrentWall(), IP_DELAY);

	/* try to get the IP address */

    if (tcpip_getIP (machineId, stop) != e_TCPIP_OK) {

	char *argv[] = {
	    "loadConfig: ",
	    reloadLocationToString (reloadLocation),
	    ": ERROR: unable to determine IP address of \"",
	    machine.value(), 
	    "\")", 
            ((char *) NULL)
	};

	g_serverData -> errorLog.error_app_cont (argv);
	return e_HARD_FAILURE;
    }

	/* success! */

    SmartCharPtr ipString (tcpip_IPToString (machineId.getIp()));

    char *argv[] = {
	"loadConfig: ",
	reloadLocationToString (reloadLocation),
	": INFO: IP address of machine \"",
	machine.value(), 
        "\" is ", 
        ipString, 
        ((char *) NULL)
    };

    g_serverData -> errorLog.error_app_cont (argv);
    ip = machineId.getIp();
    return e_OK;
}

static HandlerError handleFilenameValue (const HandlerData &handlerData, DynamicString &filename)
{
    char temp[16];

    if (handlerData.value.value()[0] != '/') {

	sprintf (temp, "%d", handlerData.line); 

	char *argv[] = {
	    "loadConfig: ",
	    reloadLocationToString (handlerData.reloadLocation),
	    ": ERROR:  filename \"", handlerData.value.value(),
	    "\" must be fully qualified (line ", temp,
	    " of configuration file \"", handlerData.filename, "\")",
	    ((char *) NULL)
	};

	g_serverData -> errorLog.error_app_cont (argv);
	return (e_HARD_FAILURE);
    }

    filename = handlerData.value;
    return (e_OK);
}

static HandlerError handleUnsignedValue (const HandlerData &handlerData, UINT_32 minimum, UINT_32& number)
{
    char temp[16];
    char temp2[16];
    UINT_32 scratch;

    if ((sscanf (handlerData.value.value(), "%u", &scratch)) && (scratch >= minimum)) {
	number = scratch;
	return e_OK;
    }

    sprintf (temp, "%u", minimum);
    sprintf (temp2, "%d", handlerData.line);

    char *argv[] = {
	"loadConfig: ",
	reloadLocationToString (handlerData.reloadLocation),
	": ERROR: \"", handlerData.field,
	"\" must be followed by an integer greater than or equal to ",
	temp, " (line ", temp2, " of configuration file \"", 
	handlerData.filename, "\")",
	((char *) NULL)
    };

    g_serverData -> errorLog.error_app_cont (argv);
    return e_HARD_FAILURE;
}

static HandlerError handleBooleanValue (const HandlerData &handlerData, BOOLEAN &flag)
{
    char temp[16];

    if (!strcmp (handlerData.value.value(), "no")) {
	flag = e_FALSE;
	return e_OK;
    }

    if (!strcmp (handlerData.value.value(), "yes")) {
	flag = e_TRUE;
	return e_OK;
    }

    sprintf (temp, "%d", handlerData.line);

    char *argv[] = {
	"loadConfig: ",
	reloadLocationToString (handlerData.reloadLocation),
	": ERROR: \"", handlerData.field,
	 "\" must be followed by \"yes\" or \"no\" (line ", temp,
	" of configuration file \"", handlerData.filename, "\")",
	((char *) NULL)
    };

    g_serverData -> errorLog.error_app_cont (argv);
    return e_HARD_FAILURE;
}

static HandlerError checkDirectoryPermissions (const DynamicString &directory, const FileUtility::FileInformation &stats, const char *description, const HandlerData &handlerData) 
{
    if (stats.type != FileUtility::e_DIRECTORY) {

	char temp[16];
	sprintf (temp, "%d", handlerData.line);

        char *argv[] = {
	    "loadConfig: ",
	    reloadLocationToString (handlerData.reloadLocation),
	    ": ERROR: \"",
	    directory.value(),
	    "\" is not a directory (line ",
	    temp,
	    " of configuration file \"", 
	    handlerData.filename, 
	    "\")",
	    ((char *) NULL)
        };

        g_serverData -> errorLog.error_app_cont (argv);
	return (e_HARD_FAILURE);
    }

    if (!(stats.permissions & FileUtility::e_USER_READ) ||
	!(stats.permissions & FileUtility::e_USER_WRITE) ||
	!(stats.permissions & FileUtility::e_USER_EXEC)) {

        char *argv[] = {
	    "loadConfig: ",
	    reloadLocationToString (handlerData.reloadLocation),
	    ": ERROR: the ",
	    (char *) description,
            " directory \"",
	    directory.value(),
	    "\" must be owned by the server and must have full owner permissions (read, write and execute); please fix the ownership and permissions",
	    ((char *) NULL)
        };

        g_serverData -> errorLog.error_app_cont (argv);
	return (e_HARD_FAILURE);
    }

    if (FileUtility::isGroupAccessible(stats.permissions) ||
	FileUtility::isOtherAccessible(stats.permissions)) {

        char *argv[] = {
	    "loadConfig: ",
	    reloadLocationToString (handlerData.reloadLocation),
	    ": ERROR: the ",
	    (char *) description,
	    " directory \"",
	    directory.value(),
	    "\" should not have any group or other access permissions; please delete the contents of the directory and fix the permissions",
	    ((char *) NULL)
        };

        g_serverData -> errorLog.error_app_cont (argv);
	return (e_HARD_FAILURE);
    }

	/* done */

    return (e_OK);
}

static HandlerError coredumpDirectoryInitializer (void)
{
#ifdef FIX_LATER
	/* 
	 * we are assuming that / is the correct path separator for all
	 * platforms
	 */
#endif

    g_serverData -> coredumpDirectory = g_serverData -> tempDirectory;
    g_serverData -> coredumpDirectory.append ("/coredumps");
    return (e_OK);
}

static HandlerError coredumpDirectoryHandler (const HandlerData &handlerData)
{
    HandlerError rc;
    DynamicString directory;
    FileUtility::FileInformation stats;

	/*
	 * make sure the directory name is fully qualified
	 */

    if ((rc = handleFilenameValue (handlerData, directory)) != e_OK) {
	return (rc);
    }

	/*
	 * If the directory does not exist, it will be created later, so just
	 * return if the directory does not exist.
	 */

    if (FileUtility::getFileStats (directory.value(), stats) != FileUtility::e_OK) {
	return (e_OK);
    }

	/* 
	 * Otherwise make sure that the directory (1) is actually a directory,
	 * (2) is accessible to the server, and (3) is not accessible to
	 * anyone else
	 */

    if ((rc = checkDirectoryPermissions (directory, stats, "coredump", handlerData)) != e_OK) {
	return (rc);
    } 

    g_serverData -> coredumpDirectory = directory;
    return (e_OK);
}

static HandlerError tempDirectoryHandler (const HandlerData &handlerData)
{
    HandlerError rc;
    DynamicString directory;
    FileUtility::FileInformation stats;

	/*
	 * make sure the directory name is fully qualified
	 */

    if ((rc = handleFilenameValue (handlerData, directory)) != e_OK) {
	return (rc);
    }

	/*
	 * make sure that the directory exists
	 */

    if (FileUtility::getFileStats (directory.value(), stats) != FileUtility::e_OK) {

	char temp[16];
	sprintf (temp, "%d", handlerData.line);

        char *argv[] = {
	    "loadConfig: ",
	    reloadLocationToString (handlerData.reloadLocation),
	    ": ERROR: agent temporary directory \"", 
	    directory.value(),
	    "\" does not exist or is not accessible (line ",
	    temp,
	    " of configuration file \"", 
	    handlerData.filename, 
	    "\")",
	    ((char *) NULL)
        };

        g_serverData -> errorLog.error_app_cont (argv);
        return (e_HARD_FAILURE);
    }

	/* 
	 * Otherwise make sure that the directory (1) is actually a directory,
	 * (2) is accessible to the server, and (3) is not accessible to
	 * anyone else
	 */

    if ((rc = checkDirectoryPermissions (directory, stats, "temporary", handlerData)) != e_OK) {
	return (rc);
    }

    g_serverData -> tempDirectory = directory;
    return (rc);
}

static HandlerError remoteInfoHandler (const HandlerData &handlerData)
{
    return (handleBooleanValue (handlerData, g_serverData -> remoteInfo));
}

static HandlerError remoteCommHandler (const HandlerData &handlerData)
{
    return (handleBooleanValue (handlerData, g_serverData -> remoteComm));
}

static HandlerError remoteForceHandler (const HandlerData &handlerData)
{
    return (handleBooleanValue (handlerData, g_serverData -> remoteForce));
}

static HandlerError remoteBeginHandler (const HandlerData &handlerData)
{
    return (handleBooleanValue (handlerData, g_serverData -> remoteBegin));
}

static HandlerError encryptionHandler (const HandlerData &handlerData)
{
    return (handleBooleanValue (handlerData, g_serverData -> encryption));
}

static HandlerError ownerHandler (const HandlerData &handlerData)
{
    int dummy;
    g_serverData -> ownerKeyname = handlerData.value;
    g_serverData -> userTable.add (0, handlerData.value.value(), NULL, dummy);
    return e_OK; 
}

static HandlerError adminHandler (const HandlerData &handlerData)
{
    int dummy;
    g_serverData -> adminKeyname = handlerData.value;
    g_serverData -> userTable.add (0, handlerData.value.value(), NULL, dummy);
    return e_OK; 
}

static HandlerError allowSignedHandler (const HandlerData &handlerData)
{
    return (handleBooleanValue (handlerData, g_serverData -> allowSigned));
}

static HandlerError logTimestampsHandler (const HandlerData &handlerData)
{ 
    return (handleBooleanValue (handlerData, g_serverData -> logTimestamps));
}

static HandlerError allowedUserHandler (const HandlerData &handlerData)
{
    int dummy; 
    g_serverData -> userTable.add (0, handlerData.value.value(), NULL, dummy);
    return e_OK; 
}

static HandlerError allowedMachineHandler (const HandlerData &handlerData)
{
	// convert machine name to lowercase

    DynamicString machine = handlerData.value;
    machine.convertToLowercase();

	// queue up the machine name for later IP address resolution

    DeferredIp *deferredIp = new DeferredIp (machine, DeferredIp::e_ALLOWED_MACHINE);
    g_serverData -> deferredIps.enqueue ((void *) deferredIp);

	// note the we have queued up the machine name

    char *argv[] = {
	"loadConfig: ",
	reloadLocationToString (handlerData.reloadLocation),
	": INFO: AllowedMachine \"",
	machine.value(),
	"\"", 
	((char *) NULL)
    };

    g_serverData -> errorLog.error_app_cont (argv);
    return e_OK;
}

static HandlerError trustedMachineHandler (const HandlerData &handlerData)
{
	// convert machine name to lowercase

    DynamicString machine = handlerData.value;
    machine.convertToLowercase();

	// queue up the machine name for later IP address resolution

    DeferredIp *deferredIp = new DeferredIp (machine, DeferredIp::e_TRUSTED_MACHINE);
    g_serverData -> deferredIps.enqueue ((void *) deferredIp);

	// note the we have queued up the machine name

    char *argv[] = {
	"loadConfig: ",
	reloadLocationToString (handlerData.reloadLocation),
	": INFO: TrustedMachine \"",
	machine.value(),
	"\"",
	((char *) NULL)
    };

    g_serverData -> errorLog.error_app_cont (argv);
    return e_OK;
}

static HandlerError reservedNameHandler (const HandlerData &handlerData)
{
    int dummy;
    char *name = handlerData.valueArgv[0].value();
    char *key = strcpyWithAlloc (handlerData.valueArgv[1].value());    // must copy

    g_serverData -> reservedTable.add (0, name, key, dummy);

    char *argv[] = {
	"loadConfig: ",
	reloadLocationToString (handlerData.reloadLocation),
	": INFO: name \"", name, "\" reserved for owner \"", key, "\"",
	((char *) NULL)
    };

    g_serverData -> errorLog.error_app_cont (argv);
    return e_OK;
}

static HandlerError userHandler (const HandlerData &handlerData)
{
    g_serverData -> userName = handlerData.value;
    return e_OK; 
}

static HandlerError groupHandler (const HandlerData &handlerData)
{
    g_serverData -> groupName = handlerData.value;
    return e_OK; 
}

static HandlerError passPhraseHandler (const HandlerData &handlerData)
{
    g_serverData -> passPhrase = handlerData.value;
    return e_OK;
}

static HandlerError interpFileHandler (const HandlerData &handlerData)
{
    return (handleFilenameValue (handlerData, g_serverData -> interpFilename));
}

static HandlerError lockFileHandler (const HandlerData &handlerData)
{
    DynamicString dummyString;
    return (handleFilenameValue (handlerData, dummyString));
}

static HandlerError pidFileHandler (const HandlerData &handlerData)
{
    return (handleFilenameValue (handlerData, g_serverData -> pidFilename));
}

static HandlerError logFileHandler (const HandlerData &handlerData)
{
    return (handleFilenameValue (handlerData, g_serverData -> logFilename));
}

static HandlerError pgpProgHandler (const HandlerData &handlerData)
{
    HandlerError rc;

    if ((rc = handleFilenameValue (handlerData, g_serverData -> pgpProgname)) == e_OK) {
	PGP_setExecutablePath (g_serverData -> pgpProgname);
    }

    return (rc);
}

static HandlerError pgpPathHandler (const HandlerData &handlerData)
{
    HandlerError rc;

    if ((rc = handleFilenameValue (handlerData, g_serverData -> pgpPathname)) == e_OK) {
	PGP_setKeyDirectory (g_serverData -> pgpPathname);
    }

    return (rc);
}

static HandlerError randomFileHandler (const HandlerData &handlerData)
{
    return (handleFilenameValue (handlerData, g_serverData -> randomFilename));
}

static HandlerError timestampFileHandler (const HandlerData &handlerData)
{
    return (handleFilenameValue (handlerData, g_serverData -> timestampFilename));
}


static HandlerError unixSocketHandler (const HandlerData &handlerData)
{
    return (handleFilenameValue (handlerData, g_serverData -> socketFilename));
}

static HandlerError tcpipSocketHandler (const HandlerData &handlerData)
{
    return (handleUnsignedValue (handlerData, 0, g_serverData -> socketPort));
}

static HandlerError randomBitsHandler (const HandlerData &handlerData)
{
    return (handleUnsignedValue (handlerData, MINIMUM_BITS, g_serverData -> randomBits));
}

static HandlerError maxStatusHandler (const HandlerData &handlerData)
{
    return (handleUnsignedValue (handlerData, 0, g_serverData -> maxStatus));
}

static HandlerError extraProcessesHandler (const HandlerData &handlerData)
{
    return (handleUnsignedValue (handlerData, MINIMUM_PROCESSES, g_serverData -> extraProcesses));
}

static HandlerError maxAgentsHandler (const HandlerData &handlerData)
{
    return (handleUnsignedValue (handlerData, 0, g_serverData -> maxAgents));
}

static HandlerError maxAnonAgentsHandler (const HandlerData &handlerData)
{
    return (handleUnsignedValue (handlerData, 0, g_serverData -> maxAnonAgents));
}

static HandlerError startWallHandler (const HandlerData &handlerData)
{
    UINT_32 seconds;

    if (handleUnsignedValue (handlerData, 0, seconds) != e_OK) {
	return (e_HARD_FAILURE);
    }

    struct timeval wall = {seconds, 0};
    g_serverData -> authorizedPermit.setWallLimit (wall);
    return e_OK;
}

static HandlerError startCpuHandler (const HandlerData &handlerData)
{
    UINT_32 seconds;

    if (handleUnsignedValue (handlerData, 0, seconds) != e_OK) {
	return (e_HARD_FAILURE);
    }

    struct timeval cpu = {seconds, 0};
    g_serverData -> authorizedPermit.setCpuLimit (cpu);
    return e_OK;
}

static HandlerError startJumpsHandler (const HandlerData &handlerData)
{
    UINT_32 jumps;

    if (handleUnsignedValue (handlerData, 0, jumps) != e_OK) {
	return (e_HARD_FAILURE);
    }

    g_serverData -> authorizedPermit.setJumpLimit (jumps);
    return e_OK;
}

static HandlerError startChildrenHandler (const HandlerData &handlerData)
{
    UINT_32 children;

    if (handleUnsignedValue (handlerData, 0, children) != e_OK) {
	return (e_HARD_FAILURE);
    }

    g_serverData -> authorizedPermit.setChildrenLimit (children);
    return e_OK;
}

static HandlerError startDepthHandler (const HandlerData &handlerData)
{
    UINT_32 depth;

    if (handleUnsignedValue (handlerData, 0, depth) != e_OK) {
	return (e_HARD_FAILURE);
    }

    g_serverData -> authorizedPermit.setDepthLimit (depth);
    return e_OK;
}

static HandlerError anonStartWallHandler (const HandlerData &handlerData)
{
    UINT_32 seconds;

    if (handleUnsignedValue (handlerData, 0, seconds) != e_OK) {
	return (e_HARD_FAILURE);
    }

    struct timeval wall = {seconds, 0};
    g_serverData -> anonymousPermit.setWallLimit (wall);
    return e_OK;
}

static HandlerError anonStartCpuHandler (const HandlerData &handlerData)
{
    UINT_32 seconds;

    if (handleUnsignedValue (handlerData, 0, seconds) != e_OK) {
	return (e_HARD_FAILURE);
    }

    struct timeval cpu = {seconds, 0};
    g_serverData -> anonymousPermit.setCpuLimit (cpu);
    return e_OK;
}

static HandlerError anonStartJumpsHandler (const HandlerData &handlerData)
{
    UINT_32 jumps;

    if (handleUnsignedValue (handlerData, 0, jumps) != e_OK) {
	return (e_HARD_FAILURE);
    }

    g_serverData -> anonymousPermit.setJumpLimit (jumps);
    return e_OK;
}

static HandlerError anonStartChildrenHandler (const HandlerData &handlerData)
{
    UINT_32 children;

    if (handleUnsignedValue (handlerData, 0, children) != e_OK) {
	return (e_HARD_FAILURE);
    }

    g_serverData -> anonymousPermit.setChildrenLimit (children);
    return e_OK;
}

static HandlerError anonStartDepthHandler (const HandlerData &handlerData)
{
    UINT_32 depth;

    if (handleUnsignedValue (handlerData, 0, depth) != e_OK) {
	return (e_HARD_FAILURE);
    }

    g_serverData -> anonymousPermit.setDepthLimit (depth);
    return e_OK;
}

static int getConfigurationFields (const CommandLine &commandLine, ServerData::ReloadLocation reloadLocation)
{
#undef  MAX_WORDS
#define MAX_WORDS 4
    int i;
    Field *fptr;
    FILE *fp;
    int wordCount;
    char temp[16];
    DynamicString words[MAX_WORDS];

	/* assertions on the parameters */

    assert (commandLine.configurationFile != NULL);

	/* make sure that we have a fully qualified filename */

    if (commandLine.configurationFile[0] != '/') {

	char *argv[] = {
	    "loadConfig: ",
	    reloadLocationToString (reloadLocation),
	    ": ERROR: configuration filename \"",
	    commandLine.configurationFile, "\" must be fully qualified",
	    ((char *) NULL)
	};

	g_serverData -> errorLog.error_app_cont (argv);
	return -1;
    }

	/* make sure that the configuration file is only owner readable and writeable */

    FileUtility::FileInformation stats;

    if (FileUtility::getFileStats (commandLine.configurationFile, stats) != FileUtility::e_OK) {

	char *argv[] = {
	    "loadConfig: ",
	    reloadLocationToString (reloadLocation),
	    ": ERROR: unable to find configuration file \"",
	    commandLine.configurationFile, 
	    "\"", 
	    ((char *) NULL)
	};

	g_serverData -> errorLog.error_app_cont (argv);
	return -1;
    }

    if (FileUtility::isGroupAccessible(stats.permissions) ||
	FileUtility::isOtherAccessible(stats.permissions)) {

	char *argv[] = {
	    "loadConfig: ",
	    reloadLocationToString (reloadLocation),
	    ": ERROR: configuration file \"",
	     commandLine.configurationFile,
	    "\" should not have any group or other access permissions; please check the contents of the file and fix the permissions",
	    ((char *) NULL)
	};

	g_serverData -> errorLog.error_app_cont (argv);
	return -1;
    }

	/* open the configuration file */

    if ((fp = fopen (commandLine.configurationFile, "r")) == NULL) {

	char *argv[] = {
	    "loadConfig: ",
	    reloadLocationToString (reloadLocation),
	    ": ERROR: unable to open configuration file \"",
	    commandLine.configurationFile, "\"", ((char *) NULL)
	};

	g_serverData -> errorLog.error_app_cont (argv);
	return -1;
    }

	/* make sure that we close the file */

    SmartFilePtr fileManager (fp);

	/* create the parser */

    PARSE parse (fp, MAX_WORD_SIZE);

	/* initialize all the "found" flags to e_FALSE */

    for (fptr = configurationFields; fptr -> name != NULL; fptr++) {
	fptr -> found = e_FALSE;
    }

	/* read through each field-value pair */

    while (1) {

	    /* get the next field */

	if (parse.getNextLine (MAX_WORDS, words, wordCount) < 0) {
	    logReadError (commandLine.configurationFile, parse.getLineNumber(), reloadLocation);
	    return -1;
        }

	    /* done if we have no field */

	if (wordCount == 0) {
	    break;
	}

	    /* see if we have a valid field name */

	Field *fieldData = NULL;

	for (fptr = configurationFields; fptr -> name != NULL; fptr++) {

	    char *sp, *dp;

	    for (sp = words[0].value(), dp = fptr -> name; (*sp != 0) && (*dp != 0); sp++, dp++) {
		if (tolower(*sp) != tolower(*dp)) {
		    break;
		}
	    }

	    if ((*sp == 0) && (*dp == 0)) {
	 	fieldData = fptr;
		break;
	    }
	} 

	if (fieldData == NULL) {
 
	    sprintf (temp, "%d", parse.getLineNumber());

	    char *argv[] = {
		"loadConfig: ",
		reloadLocationToString (reloadLocation),
	        ": ERROR: unknown field type \"", words[0].value(), "\" (line ",
		temp, " of configuration file \"",
		 commandLine.configurationFile, "\")", ((char *) NULL)
	    };

	    g_serverData -> errorLog.error_app_cont (argv);
	    return (-1);
	}

	fieldData -> found = e_TRUE;

	    /* start filling in the data for the handler */ 

	HandlerData handlerData;
	handlerData.field = fieldData -> name;
        handlerData.filename = commandLine.configurationFile;
	handlerData.line = parse.getLineNumber();
	handlerData.reloadLocation = reloadLocation;

	    /* get the field arguments */

	int maxArgs = (fieldData -> args == e_TAKES_ONE) ? 1 : 2;

	if (wordCount != maxArgs + 1) {

	    char *argv[] = {
	        "loadConfig: ",
		reloadLocationToString (reloadLocation),
		": ERROR: field \"", words[0].value(),
		"\" should have ",
		(char *) ((maxArgs == 1) ? "one argument" : "two arguments"),
		" (line ", temp,
		" of configuration file \"", commandLine.configurationFile,
	        "\")", ((char *) NULL)
	    };

	    g_serverData -> errorLog.error_app_cont (argv);
	    return (-1);
	}

	for (i = 0; i < maxArgs; i++) {
	    handlerData.valueArgv[i] = words[i+1];
	}

	    /* determine whether or not we should call the handler */

	BOOLEAN loadField = e_FALSE;

	switch (reloadLocation) {

	    case ServerData::e_PRELOAD:

		if (fieldData->reload == e_PRELOAD_ONCE) {
		    loadField = e_TRUE;
		}

		break;

	    case ServerData::e_INITIAL_LOAD:

		if (fieldData->reload != e_PRELOAD_ONCE) {
		    loadField = e_TRUE;
		}

		break;

	    case ServerData::e_RELOADING:

		if (fieldData->reload == e_RELOAD) {
		    loadField = e_TRUE;
		}

		break;
	}

	    /* call the handler */

	HandlerError errorCode = e_OK;

	if (loadField) {
	    handlerData.value = handlerData.valueArgv[0];
	    errorCode = (fieldData -> handler) (handlerData);
	}

	    /* cleanup and then return on error */

	if (errorCode == e_HARD_FAILURE) {
	    return -1;
	}
    }

	/* check that we have all required fields */

    for (fptr = configurationFields; fptr -> name != NULL; fptr++) {

	if ((reloadLocation == ServerData::e_PRELOAD) && 
	    (fptr->reload != e_PRELOAD_ONCE)) {
	    continue;
	}

	if ((fptr -> required) && !(fptr -> found)) {

	    if (fptr -> initializer != NULL) {

		HandlerError errorCode = (fptr -> initializer) ();

		if (errorCode != e_OK) {
		    return (-1);
		} 	

	    } else {

		char *argv[] = {
		    "loadConfig: ",
		    reloadLocationToString (reloadLocation),
		    ": ERROR: missing field \"", fptr -> name, 
		    "\" (configuration file \"", commandLine.configurationFile, 
		    "\")", ((char *) NULL)
		};

		g_serverData -> errorLog.error_app_cont (argv);
		return (-1); 
	    }
	}
    }

    return 0;
}

/* getInterpreterList 

   Purpose: Get the set of interpreters that are available through the server

     Input: None

   Globals: g_serverData = server characteristics
			 (ServerData *)

    Output: The procedure returns -1 if an error occurs while opening or
            reading from the interpreter configuration  file.  Otherwise 
	    the procedure loads the set of interpreters and returns 0. 
*/

static void logInterpreterError (int line, char *filename, char *message, ServerData::ReloadLocation reloadLocation)
{
    char temp[16];
    sprintf (temp, "%d", line);

    char *argv[] = {
	"loadConfig: ",
	reloadLocationToString (reloadLocation),
	": ERROR: line ",
	temp,
	" of \"",
	filename,
	"\": ",
	message,
	((char *) NULL) 
    };

    g_serverData ->
	errorLog.error_app_cont 
	    (argv);
}

int getInterpreterList (ServerData::ReloadLocation reloadLocation)
{
    FILE *fp;
    struct stat fileInfo;
    char configMessage[1024];

	/* assertions */

    assert (g_serverData != NULL);
    assert (g_serverData -> interpFilename.isEmpty() == e_FALSE);

	/* log message */

    char *argv[] = {
	"loadConfig: ",
	reloadLocationToString (reloadLocation),
	": INFO: loading interpreter list",
	((char *) NULL)
    };

    g_serverData -> errorLog.error_app_cont (argv);

	/* make sure that the interpeter list is not group or other writeable */

    FileUtility::FileInformation stats;
    char *filename = g_serverData -> interpFilename.value();

    if (FileUtility::getFileStats (g_serverData -> interpFilename.value(), stats) != FileUtility::e_OK) {

	char *argv[] = {
	    "loadConfig: ",
	    reloadLocationToString (reloadLocation),
	    ": ERROR: unable to find interpreter file \"",
	    filename,
	    "\"",
	    ((char *) NULL)
	};

	g_serverData -> errorLog.error_app_cont (argv);
	return -1;
    }

    if ((stats.permissions & FileUtility::e_GROUP_WRITE) || 
	(stats.permissions & FileUtility::e_OTHER_WRITE)) {

	char *argv[] = {
	    "loadConfig: ",
	    reloadLocationToString (reloadLocation),
	    ": ERROR: interpeter file \"",
	    filename,
	    "\" is either group- or other-writeable; please check the contents of the file and fix the permissions so that it is only owner-writeable",
	    ((char *) NULL)
	};

	g_serverData -> errorLog.error_app_cont (argv);
	return -1;
    }

	/* open the interpeter list */

    if ((fp = fopen (filename, "r")) == NULL) {

	char *argv[] = {
	    "unable to open interpreter file \"",
	    filename,
	    "\"",
	    ((char *) NULL)
	};
 
	g_serverData -> errorLog.error_app_cont (argv);
	return -1; 
    }

    SmartFilePtr fileManger (fp);   // make sure that fp is closed 

	/* the parser */ 

    PARSE parser (fp, MAX_WORD_SIZE);

	/* parsing flags */ 

    BOOLEAN beginExpected        = e_TRUE;
    BOOLEAN nameFound	         = e_FALSE;
    BOOLEAN executableFound      = e_FALSE;
    BOOLEAN argv0Found           = e_FALSE;
    BOOLEAN typeFound            = e_FALSE;
    BOOLEAN minHotCountFound     = e_FALSE;
    BOOLEAN maxHotCountFound     = e_FALSE;
    BOOLEAN agentsPerInterpFound = e_FALSE;
    DynamicString name;
    DynamicString executable;
    DynamicString argv0;
    DynamicStringQueue arguments;
    INTERP::InterpreterTypes type = INTERP::e_NORMAL_INTERP;
    int minHotCount;
    int maxHotCount;
    int agentsPerInterp;

	/* parse the interpreter file */
 
    while (1) {

	int wordCount;
#undef  MAX_WORDS
#define MAX_WORDS 4
	DynamicString words[MAX_WORDS];

	    /* get the next line */

	if (parser.getNextLine (MAX_WORDS, words, wordCount) < 0) {
	    logReadError (filename, parser.getLineNumber(), reloadLocation);
	    return (-1);
        }

	    /* done if we have no words */

	if (wordCount == 0) {

	    if (!beginExpected) {
		logInterpreterError (parser.getLineNumber(), filename, "expected keyword \"End\" at end of interpreter description", reloadLocation);
		return (-1);
	    }

	    return (0);
	}

	    /* convert the field name to lowercase */

	words[0].convertToLowercase();

	    /* are we expecting a "Begin"? */

	if (beginExpected) {

	    if (words[0] != "begin") {
		logInterpreterError (parser.getLineNumber(), filename, "interpreter description must begin with the keyword \"Begin\"", reloadLocation);
		return (-1);
	    }

	    if (wordCount != 1) {
		logInterpreterError (parser.getLineNumber(), filename, "keyword \"Begin\" must appear on a line by itself", reloadLocation);
		return (-1);
	    }

	    beginExpected = e_FALSE;
	    continue;
	}

	    /* No, we already have the "Begin", so check for the field */

	if (words[0] == "name") {

	    if (wordCount != 2) {
		logInterpreterError (parser.getLineNumber(), filename, "the \"Name\" field takes one argument", reloadLocation);
		return (-1);
	    }
	
	    if (nameFound) {
		logInterpreterError (parser.getLineNumber(), filename, "each interpreter description can have only one \"Name\" field", reloadLocation);
		return (-1);
	    }
		
	    name = words[1];
	    nameFound = e_TRUE;
	    continue;
	}

	if (words[0] == "executable") {

	    if (wordCount != 2) {
		logInterpreterError (parser.getLineNumber(), filename, "the \"Executable\" field takes one argument", reloadLocation);
		return (-1);
	    }

	    if (executableFound) {
		logInterpreterError (parser.getLineNumber(), filename, "each interpreter description can have only one \"Executable\" field", reloadLocation);
		return (-1);
	    }

            if ((words[1].value())[0] != '/') {
	        sprintf (configMessage, "interpreter name \"%s\" is not fully qualified", words[1].value());
	        logInterpreterError (parser.getLineNumber(), filename, configMessage, reloadLocation);
	        return (-1);
	    }

	    if (stat (words[1].value(), &fileInfo) < 0) {
	        sprintf (configMessage, "interpreter \"%s\" does not exist", words[1].value());
	        logInterpreterError (parser.getLineNumber(), filename, configMessage, reloadLocation);
	        return (-1); 
	    }

	    if (fileInfo.st_mode & S_IFDIR) {
	        sprintf (configMessage, "\"%s\" is a directory; should be an interpreter", words[1].value());
	        logInterpreterError (parser.getLineNumber(), filename, configMessage, reloadLocation);
	        return (-1); 
	    }

	    if (access (words[1].value(), X_OK) < 0) {
	        sprintf (configMessage, "interpreter \"%s\" is not executable", words[1].value());
	        logInterpreterError (parser.getLineNumber(), filename, configMessage, reloadLocation);
	        return (-1); 
	    }

	    executable = words[1];
	    executableFound = e_TRUE;
	    continue;
	}

	if (words[0] == "arg") {

	    if (wordCount != 2) {
		logInterpreterError (parser.getLineNumber(), filename, "the \"Arg\" field takes one argument", reloadLocation);
		return (-1);
	    }

	    arguments.enqueue (words[1]);
	    continue;
	}

	if (words[0] == "argv0") {

	    if (wordCount != 2) {
		logInterpreterError (parser.getLineNumber(), filename, "the \"Argv0\" field takes one argument", reloadLocation);
		return (-1);
	    }

	    if (argv0Found) {
		logInterpreterError (parser.getLineNumber(), filename, "each interpreter description can have only one \"Argv0\" field", reloadLocation);
		return (-1);
	    }

	    argv0 = words[1];
	    argv0Found = e_TRUE;
	    continue;
	}

	if (words[0] == "type") {

	    if (wordCount != 2) {
		logInterpreterError (parser.getLineNumber(), filename, "the \"Type\" field takes one argument", reloadLocation);
		return (-1);
	    }

	    if (typeFound) {
		logInterpreterError (parser.getLineNumber(), filename, "each interpreter description can have only one \"Type\" field", reloadLocation);
		return (-1);
	    }

	    words[1].convertToLowercase();

	    if (words[1] == "state") {
		type = INTERP::e_STATE_INTERP;
	    } else if (words[1] == "normal") {
		type = INTERP::e_NORMAL_INTERP;
	    } else {
	        sprintf (configMessage, "expected \"state\" or \"normal\" but got \"%s\"", words[1].value());
	        logInterpreterError (parser.getLineNumber(), filename, configMessage, reloadLocation);
	        return (-1);
    	    }

	    typeFound = e_TRUE;
	    continue;
	}

	if (words[0] == "minhot") {

	    if (wordCount != 2) {
		logInterpreterError (parser.getLineNumber(), filename, "the \"MinHot\" field takes one argument", reloadLocation);
		return (-1);
	    }

	    if (minHotCountFound) {
		logInterpreterError (parser.getLineNumber(), filename, "each interpreter description can have only one \"MinHot\" field", reloadLocation);
		return (-1);
	    }

	    int itemsScanned = sscanf (words[1].value(), "%d", &minHotCount);

	    if ((itemsScanned != 1) || (minHotCount <= 0)) {
	        sprintf (configMessage, "expected a positive integer but got \"%s\"", words[1].value());
	        logInterpreterError (parser.getLineNumber(), filename, configMessage, reloadLocation);
	        return (-1);
    	    }

	    minHotCountFound = e_TRUE;
	    continue;
	}

	if (words[0] == "maxhot") {

	    if (wordCount != 2) {
		logInterpreterError (parser.getLineNumber(), filename, "the \"MaxHot\" field takes one argument", reloadLocation);
		return (-1);
	    }

	    if (maxHotCountFound) {
		logInterpreterError (parser.getLineNumber(), filename, "each interpreter description can have only one \"MaxHot\" field", reloadLocation);
		return (-1);
	    }

	    int itemsScanned = sscanf (words[1].value(), "%d", &maxHotCount);

	    if ((itemsScanned != 1) || (maxHotCount <= 0)) {
	        sprintf (configMessage, "expected a positive integer but got \"%s\"", words[1].value());
	        logInterpreterError (parser.getLineNumber(), filename, configMessage, reloadLocation);
	        return (-1);
    	    }

	    maxHotCountFound = e_TRUE;
	    continue;
	}

	if (words[0] == "agentsperinterp") {

	    if (wordCount != 2) {
		logInterpreterError (parser.getLineNumber(), filename, "the \"AgentsPerInterp\" field takes one argument", reloadLocation);
		return (-1);
	    }

	    if (agentsPerInterpFound) {
		logInterpreterError (parser.getLineNumber(), filename, "each interpreter description can have only one \"AgentsPerInterp\" field", reloadLocation);
		return (-1);
	    }

	    int itemsScanned = sscanf (words[1].value(), "%d", &agentsPerInterp);

	    if ((itemsScanned != 1) || (agentsPerInterp <= 0)) {
	        sprintf (configMessage, "expected a positive integer but got \"%s\"", words[1].value());
	        logInterpreterError (parser.getLineNumber(), filename, configMessage, reloadLocation);
	        return (-1);
    	    }

	    agentsPerInterpFound = e_TRUE;
	    continue;
	}

	if (words[0] == "end") {

	    char *missingField = NULL;

		/* make sure that the "End" is on a line by itself */

	    if (wordCount != 1) {
		logInterpreterError (parser.getLineNumber(), filename, "keyword \"End\" must appear on a line by itself", reloadLocation);
		return (-1);
	    }

		/* make sure that we had all necessary fields */

	    if (!nameFound) {
		missingField = "Name";
	    } else if (!executableFound) {
		missingField = "Executable";
	    } else if (!argv0Found) {
		missingField = "Argv0";
	    } else if (!typeFound) {
		missingField = "Type";
	    }

	    if (missingField != NULL) {
	        sprintf (configMessage, "interpreter description must include a \"%s\" field", missingField);
		logInterpreterError (parser.getLineNumber(), filename, configMessage, reloadLocation);
		return (-1);
	    }

		/* use the default hot count if no hot count was specified */

	    if (!minHotCountFound && !maxHotCountFound) {

		minHotCount = INTERP::e_DEFAULT_HOT_COUNT;
		maxHotCount = minHotCount << 1;

	    } else if (!minHotCountFound) {

		minHotCount = maxHotCount >> 1;

		if (minHotCount < 1) {
		    minHotCount = 1;
		}

	    } else if (!maxHotCountFound) {

		maxHotCount = minHotCount << 1;

	    } else {

	        if (minHotCount > maxHotCount) {
		    maxHotCount = minHotCount;
	        }
	    }

	  	/* assume that the interpreter can only execute one */
		/* agent at a time if there was no AgentsPerInterp  */
		/* field                                            */

	    if (!agentsPerInterpFound) {
		agentsPerInterp = INTERP::e_DEFAULT_AGENTS_PER_INTERP;
	    }

		/* add the interpreter description to the table */

	    int there;
	    INTERP *interp = new INTERP 
		(name, executable, argv0, type, 
		 minHotCount, maxHotCount, agentsPerInterp);

	    while (arguments.get_count() != 0) {
		interp -> addArgument (arguments.dequeue());
	    }

#ifdef FIX_LATER
		/* We currently have redundant tables -- ServerData.interpTable    */
		/* will go away once we have finalized the hot interpreter system. */
#endif

	    g_serverData -> interpTable.add (0, name.value(), (void *) interp, there);

	    HotInterpreters::ErrorCodes rc =
		hotInterpreters -> addInterpreter (name.value(), *interp);

	    if (rc == HotInterpreters::e_AlreadyThere) {
	        sprintf (configMessage, "duplicate interpreter \"%s\"", name.value());
	        logInterpreterError (parser.getLineNumber(), filename, configMessage, reloadLocation);
		DELETE_OBJECT(interp);
	        return (-1); 
	    }

	    char *argv[] = {
		"loadConfig: ",
		reloadLocationToString (reloadLocation),
		": INFO: adding \"",
		name.value(),
		"\" to interpreter list",
		((char *) NULL)
	    };

	    g_serverData -> errorLog.error_app_cont (argv);
 
		/* reset for the next interpeter description */

	    beginExpected        = e_TRUE;
	    nameFound            = e_FALSE;
	    executableFound      = e_FALSE;
  	    argv0Found           = e_FALSE;
            typeFound            = e_FALSE;
	    minHotCountFound     = e_FALSE;
	    maxHotCountFound     = e_FALSE;
	    agentsPerInterpFound = e_FALSE;
	    continue;
	}

	    /* unknown field */

	sprintf (configMessage, "invalid field \"%s\": should be Name, Executable, Arg, Argv0, Type, End, MinHot, MaxHot or AgentsPerInterp", words[0].value());
	logInterpreterError (parser.getLineNumber(), filename, configMessage, reloadLocation);
	return (-1);
    }

	/* better not ever get here! */

    abort_with_message ("should never get here!");    // should never get here
    return (-1);
} 

/* getRandomBits

   Purpose: Load the random bits file if present

     Input: None

   Globals: g_serverData = server characteristics
		         (SeverData &)

    Output: The procedure returns 0 if no random bits file was specified or if
	    it was able to successfully load the random bits file.  Otherwise the
	    procedure returns 0.
*/

static int getRandomBits (ServerData::ReloadLocation reloadLocation)
{
    int fd;
    int nread;
    char temp1[16];
    char temp2[16];
    UINT_32 bitCount;
    UINT_8 randomByte;

	/* nothing to do if we do not have a random bits file */

    if (g_serverData -> randomFilename.isEmpty()) {
	return 0;
    }

	/* make sure that the random bits file is only owner readable and writeable */

    FileUtility::FileInformation stats;

    if (FileUtility::getFileStats (g_serverData -> randomFilename.value(), stats) != FileUtility::e_OK) { 
	
	char *argv[] = {
	    "loadConfig: ",
	    reloadLocationToString (reloadLocation),
	    ": ERROR: unable to find random bits file \"",
	    g_serverData -> randomFilename.value(), 
	    "\"", 
	    ((char *) NULL)
	};

	g_serverData -> errorLog.error_app_cont (argv);
	return -1;
    }

    if (FileUtility::isGroupAccessible(stats.permissions) ||
	FileUtility::isOtherAccessible(stats.permissions)) {

	char *argv[] = {
	    "loadConfig: ",
	    reloadLocationToString (reloadLocation),
	    ": ERROR: random bits file \"", 
	    g_serverData -> randomFilename.value(),
	    "\" should not have any group or other access permissions; please regenerate the file with the \"genrand\" utility", 
	    ((char *) NULL)
	};

	g_serverData -> errorLog.error_app_cont (argv);
	return -1;
    }
    
	/* open the random bits file */

    do {
	fd = FileUtility::open (g_serverData -> randomFilename.value(), O_RDONLY);
    } while ((fd < 0) && ((errno == EINTR) || (errno == EAGAIN)));

    if (fd < 0) {

	char *argv[] = {
	    "loadConfig: ",
	    reloadLocationToString (reloadLocation),
	    ": ERROR: unable to open random bits file \"",
	    g_serverData -> randomFilename.value(), "\"", ((char *) NULL)
	};

	g_serverData -> errorLog.error_sys_cont (argv);
	return -1;
    }

	/* read bytes from the random bits file */

    bitCount = 0;

    (void) noise ();   // not much entropy but some

    while ((tcpip_blockingReadn (fd, (char *) &randomByte, 1, nread) == e_TCPIP_OK) && (nread > 0)) {

	bitCount += 8;
	randPoolAddBytes ((byte *) &randomByte, 1);

	if (bitCount >= g_serverData -> randomBits) {
	    break;
	}
    }

    FileUtility::close (fd);
    (void) noise ();   // not much entropy but some

	/* make sure that we had enough bits */

    if (bitCount < g_serverData -> randomBits) {

	sprintf (temp1, "%u", bitCount);
	sprintf (temp2, "%u", g_serverData -> randomBits);

	char *argv[] = {
	    "loadConfig: ",
	    reloadLocationToString (reloadLocation),
	    ": ERROR: random bits file \"",
	    g_serverData -> randomFilename.value(),
	    "\" only contained ",
	    temp1,
	    " random bits even though configuration file specified ",
	    temp2,
	    ((char *) NULL)
	};

	g_serverData -> errorLog.error_app_cont (argv);
	return -1;
    }

	/* burn the random bits file */

    RandomFile randomFile (g_serverData -> randomFilename, 
	g_serverData -> randomBits);

    if (randomFile.burnBits() != RandomFile::e_OK) {
	
	char *argv[] = {
	    "loadConfig: ",
	    reloadLocationToString (reloadLocation),
	    ": ERROR: unable to burn random bits file \"",
	    g_serverData -> randomFilename.value(),
	    "\": please make sure that the file is user writeable",
	    (char *) NULL
	};

	g_serverData -> errorLog.error_app_cont (argv);
	return -1;
    }
 
	/* done */

    sprintf (temp1, "%u", bitCount);

    char *argv[] = {
	"loadConfig: ",
	reloadLocationToString (reloadLocation),
	": INFO: ",
	temp1,
	" random bits read from file \"",
	g_serverData -> randomFilename.value(),
	"\"",
	((char *) NULL)
    };

    g_serverData -> errorLog.error_app_cont (argv);
    return 0;
}

void ServerData::resolveIp (ServerData::ReloadLocation reloadLocation)
{
    DeferredIp *deferredIp;

#ifdef FIX_LATER
	/*
         * The server is unresponsive while it is resolving the IP addresses,
         * and if it is unable to resolve an IP address, it doesnot put the
         * IP address back into the queue so that it can try again later.
         * It is not worth devoting significant effort to fixing this now,
         * since eventually the server keynames will *not* contain the IP       
         * address as an integral component.
         */
#endif

	/*
         * loop through all the TrustedMachine's and AllowedMachine's
 	 */

    while ((deferredIp = (DeferredIp *) deferredIps.dequeue()) != NULL) {

	DeferredIp::DeferredType type;

	    /*
	     * machine name
	     */

	DynamicString machine = deferredIp -> getMachine();

	    /* 
 	     * AllowedMachine or TrustedMachine?
	     */

	if ((type = deferredIp -> getType()) == DeferredIp::e_ALLOWED_MACHINE) {

	    int dummy;
	    UINT_32 ip;

	    if (handleIPValue (reloadLocation, machine, ip) == e_OK) {
		g_serverData -> machineTable.add (ip, NULL, NULL, dummy);
	    }

	} else {

	    int dummy;
	    UINT_32 ip;

	    if (handleIPValue (reloadLocation, machine, ip) == e_OK) {

		MachineId machineId (machine, ip);
		SmartCharPtr keyname (makeServerKeyname (machineId));
		g_serverData -> trustedTable.add (0, keyname, NULL, dummy);
		g_serverData -> machineTable.add (ip, NULL, NULL, dummy);
    
		char *argv[] = {
		    "loadConfig: ",
		    reloadLocationToString (reloadLocation),
		    ": INFO: keyname for TrustedMachine \"",
		    machine.value(),
		    "\" is \"",
		    keyname,
		    "\"", 
		    ((char *) NULL)
		};

		g_serverData -> errorLog.error_app_cont (argv);
	    }
	}

	    /*
	     * cleanup
	     */

	DELETE_OBJECT(deferredIp);
    }
}

int ServerData::load (const CommandLine &commandLine, ServerData::ReloadLocation reloadLocation)
{
	/* start from a clean copy of the configuration -- we need */
        /* this since we might be calling this routine in response */
	/* to a SIGHUP                                             */

    if (reloadLocation == e_RELOADING) {
	g_serverData -> empty ();
    }

	/* get the configuration fields */

    if (getConfigurationFields (commandLine, reloadLocation) < 0) {
	return -1;
    }

	/* load the interpreter list */

    if ((reloadLocation == e_INITIAL_LOAD) || (reloadLocation == e_RELOADING)) {
	if (getInterpreterList (reloadLocation) < 0) {
	    return -1;	
	}
    }

	/* load the random bits file if present */

    if ((reloadLocation == e_INITIAL_LOAD) || (reloadLocation == e_RELOADING)) {
	if (getRandomBits (reloadLocation) < 0) {
	    return -1;
	}
    }

	/* set up the agent environ list */

    if ((reloadLocation == e_INITIAL_LOAD) || (reloadLocation == e_RELOADING)) {

	int i;
	int count;

	    /* set up the environment entries for the PGP executable and key directory */

	g_serverData -> pgpProgEnv.append ("PGPPROG=");
	g_serverData -> pgpProgEnv.append (g_serverData -> pgpProgname);
	g_serverData -> pgpPathEnv.append ("PGPPATH=");
	g_serverData -> pgpPathEnv.append (g_serverData -> pgpPathname);

	    /* identify the number of non-PGP entries in the current environ */

	count = 0;

	for (i = 0; environ[i] != NULL; ++i) {

	    if (!strncmp (environ[i], "CONNECT", 7)) {
		continue;
	    }

	    if (!strncmp (environ[i], "PGPPATH", 7)) {
		continue;
	    }

	    if (!strncmp (environ[i], "PGPPROG", 7)) {
		continue;
	    }

	    count += 1;
	}

	    /* assemble the agent environent */

	g_serverData -> agentEnviron = new char * [count + 4];

	count = 0;

	for (i = 0; environ[i] != NULL; ++i) {

	    if (!strncmp (environ[i], "CONNECT", 7)) {
		continue;
	    }

	    if (!strncmp (environ[i], "PGPPATH", 7)) {
		continue;
	    }

	    if (!strncmp (environ[i], "PGPPROG", 7)) {
		continue;
	    }

	    g_serverData -> agentEnviron[count++] = environ[i];
	}

	g_serverData -> agentEnviron[count++] = g_serverData->pgpProgEnv.value();
	g_serverData -> agentEnviron[count++] = g_serverData->pgpPathEnv.value();
	g_serverData -> connectionEnvSlot     = count;
	g_serverData -> agentEnviron[count++] = (char *) NULL; 
	g_serverData -> agentEnviron[count++] = (char *) NULL; 
    }

	/* turn on the error log */

    if (commandLine.daemon) {

	if (reloadLocation == e_PRELOAD)  {
	    if (commandLine.truncateLog) {
		unlink (g_serverData -> logFilename.value());
	    }
	}

	g_serverData -> errorLog = ERROR (g_serverData -> logFilename.value());
    }

    return 0;
}

/* ServerData::empty

   Purpose: Empty out the server configuration items that can change on a
            reload
*/

void ServerData::empty (void)
{
    int i;
    HashNode *node;

	// empty out the queue of deferred IP's

    for (i = 0; i < deferredIps.get_count(); ++i) {
	DeferredIp *deferredIp = (DeferredIp *) deferredIps.dequeue();
	DELETE_OBJECT(deferredIp);
    }
 
	// empty out the interpreter table 

    HashSearch search (&g_serverData -> interpTable);

    while ((node = search.next()) != NULL) {
	INTERP *interp = (INTERP *) node -> data;
	DELETE_OBJECT(interp);
    }

    g_serverData -> interpTable.empty ();

	// empty out the ReservedName table

    search = HashSearch (&g_serverData -> reservedTable);

    while ((node = search.next()) != NULL) {
	DELETE_OBJECT_WITH_CAST(char*,node->data);
    }

    g_serverData -> reservedTable.empty ();
 
	// empty out the AllowedMachine, AllowedUser and TrustedMachine
	// tables -- these tables contain keys only so we do not need to
	// delete any data                                               

    g_serverData -> machineTable.empty ();
    g_serverData -> userTable.empty ();
    g_serverData -> trustedTable.empty ();

	// empty out the strings (except for user, group, lockFilename
	// and pidFilename which are ignored on server reload)

    DELETE_ARRAY_OBJECT(agentEnviron);
    g_serverData -> adminKeyname.empty ();
    g_serverData -> ownerKeyname.empty ();
    g_serverData -> interpFilename.empty ();
    g_serverData -> logFilename.empty ();
    g_serverData -> pgpProgname.empty ();
    g_serverData -> pgpPathname.empty ();
    g_serverData -> randomFilename.empty ();
    g_serverData -> pgpProgEnv.empty ();
    g_serverData -> pgpPathEnv.empty ();
    g_serverData -> tempDirectory.empty ();
    g_serverData -> coredumpDirectory.empty ();

	/* empty (reset) the numeric data */

    encryption	   = e_TRUE;
    maxAgents	   = MINIMUM_AGENTS;
    maxAnonAgents  = 0;
    maxStatus 	   = 0;
    allowSigned	   = e_FALSE;
    remoteBegin	   = e_FALSE;
    remoteComm	   = e_FALSE;
    remoteForce	   = e_FALSE;
    remoteInfo	   = e_FALSE;
    allowSigned	   = e_FALSE;
    randomBits	   = MINIMUM_BITS;

	/* empty (reset) the permits */

    authorizedPermit.empty ();
    anonymousPermit.empty ();
}

/* ServerData::ServerData and ServerData::~ServerData

   Purpose: These procedures are the constructor and destructor for class 
	    ServerData. 
*/


ServerData::ServerData (char *hostname):
	userName	((char *) NULL),
  	groupName	((char *) NULL), 
	encryption	(e_TRUE),
	maxAgents	(MINIMUM_AGENTS),
	maxAnonAgents	(0),
	maxStatus	(0),
	extraProcesses  (0),
	remoteBegin	(e_FALSE),
	remoteComm	(e_FALSE),
	remoteForce	(e_FALSE),
	remoteInfo	(e_FALSE),
	ownerKeyname	((char *) NULL),
	adminKeyname 	((char *) NULL),
	allowSigned	(e_FALSE),
	machineTable	(ACCESS_TABLE_SIZE),
	userTable	(ACCESS_TABLE_SIZE),
	trustedTable	(ACCESS_TABLE_SIZE),
	reservedTable	(ACCESS_TABLE_SIZE),
	socketFilename	(UNIX_SOCKET, BufferOwnership::e_COPY),
	socketPort	(SERVER_TCP_PORT),
	logFilename	((char *) NULL),
	interpFilename	((char *) NULL),
	pidFilename	((char *) NULL),
	passPhrase	((char *) NULL),
	randomBits	(MINIMUM_BITS),
	randomFilename	((char *) NULL),
	pgpProgname	((char *) NULL),
	pgpPathname	((char *) NULL),
	interpTable	(INTERP_TABLE_SIZE),
	authorizedPermit(),
	anonymousPermit (),
	timestampFilename ((char *) NULL),
	logTimestamps   (e_FALSE),
	errorLog	(),
	sockfd		(-1),
	unixSockfd	(-1),
	timestampFd	(-1),
	security	(),
	host		(),
	socketdPid	(-1),
	agentdPid	(getpid ()),
	agentEnviron    (NULL),
	connectionEnvSlot (0),
	pgpPathEnv      ((char *) NULL),	
	pgpProgEnv	((char *) NULL)
{
    assert (hostname != (char *) NULL);
    DynamicString name (hostname);
    name.convertToLowercase();
    host.setName (name);
}

ServerData::~ServerData ()
{
    empty ();	// more than necessary but little extra cost
}
