/* Agent Tcl
   Bob Gray
   9 February 1995

   servProcess.cc 

   This file implements the functions that the server uses to HANDLE a client
   request.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "platDelete.h"
#include "platThread.h"
#include "suppStrings.h"
#include "agentAgentId.h"	    // AgentId
#include "genMesgDriver.h"	    // MessageDriver
#include "genServerInterface.h"	    // ServerInterface
#include "hotInterpreter.h"	    // HotInterpreters

#include "platExclusion.h"
#include "platInterrupt.h"
#include "agentd.h"
#include "mesgConnection.h"
#include "genConversion.h"
#include "genError.h"
#include "genFile.h"
#include "genPipe.h"
#include "genManager.h"
#include "genMessage.h"
#include "mesgTcpip.h"
#include "genTransmit.h"
#include "interp.h"
#include "piped.h"
#include "random.h"
#include "servConf.h"
#include "servProcess.h"
#include "servReceive.h"
#include "truefalse.h"

extern void setRecord (BOOLEAN);

    /* error messages */

static char *const transferError = 
	"system error: unable to transfer state image to the interpreter";

    /* forward declarations */

static int sendErrorMessage (
	char *message,
	AgentId& root,
	AgentId& local,
	BASE_SECURITY &security
    );

static int checkAbnormalExit (
	AgentId& root,
	AgentId& local,
	BASE_SECURITY &security,
	WAIT_STATUS_TYPE status
    );

    /* make a background connection */

static int backgroundConnection (
	BASE_SECURITY &sender, 
	int sockfd,
	UINT_32 localId
    );

/* removeDirectory

   Purpose: Delete a directory and all its contents

     Input: directory = name of the directory
			(DynamicString)

    Output: The procedure returns -1 on error and 0 on success.
*/

int removeDirectory (DynamicString directory)
{
    DIR *dir;
    int rc = 0;
    struct dirent *dirEntry;

	/*
	 * open the directory
	 */

    if ((dir = opendir (directory.value())) == NULL) {

	if (errno == ENOENT) {
	    return (0);		// doesn't exist -- no error
	} else {
	    return (-1);	// anything else -- error 
	}
    }

	/*
	 * loop through each directory entry
	 */

    while ((dirEntry = readdir (dir)) != NULL) {

	    // skip over directory entries . and ..
 
	if (!strcmp (dirEntry -> d_name, ".")) {
	    continue;
	}

	if (!strcmp (dirEntry -> d_name, "..")) {
	    continue;
	}

	    // construct the full pathname of the entry

	DynamicString path = directory;
	path.append ("/");
	path.append (dirEntry -> d_name);

	    // see if the entry is a subdirectory; if it is a subdirectory,
	    // then recursively call removeDirectory to remove the
	    // subdirectory; otherwise use unlink to remove the entry

	FileUtility::FileInformation fileInfo;

	if (FileUtility::getFileStats (path.value(), fileInfo) != FileUtility::e_OK) {
	    rc = -1;
	    break;
	}

	if (fileInfo.type == FileUtility::e_DIRECTORY) {

	    if (removeDirectory (path) < 0) {
		rc = -1;
		break;
	    }

	} else {

	    if (unlink (path.value()) < 0) {
		rc = -1;
		break;
	    }
	}
    }

	/*
	 * close the directory, delete the directory if everything else worked,
	 * and then return
	 */

    closedir (dir);

    if (rc == 0) {
	if (rmdir (directory.value()) < 0) {
	    rc = -1;
	}
    }

    return (rc);
}

/* respondWithError

   Purpose: Send a RESP_ERROR message down the socket

     Input: sockfd = socket descriptor
		     (int)

	    code   = server error code
		     (unsigned char)

    Output: The procedure constructs a RESP_ERROR message containing the
            given error code and sends the message down the socket.
*/

void respondWithError (int sockfd, MonitorLock *sockfdLock, UINT_8 code)
{
	// construct the error message

    MESSAGE message (messagesFromServer.messages[RESP_ERROR]);
    message.addByte (0, code);

	// send the error message

    if (sockfdLock != NULL) {
	sockfdLock -> acquire();
    }

    (void) message_send (sockfd, message);

    if (sockfdLock != NULL) {
	sockfdLock -> release();
    }
}

/* sendErrorMessage

   Purpose: Send an error message to the root agent

     Input: string   = the message
		       (char *)

	    root     = the root agent
		       (const AgentId&) 

	    local    = the local agent that caused the error
		       (const AgentId&)

	    security = security information for the local agent
		       (const BASE_SECURITY &)

    Output: The procedure returns -1 on error and 0 otherwise.
*/

const int maximumTries		= 4;
const int initialSleepTime	= 1;
const int maximumConnectionTime = 120;

int sendErrorMessage (char *string, AgentId& root, AgentId& local, BASE_SECURITY &security)
{
	/* northing to do if we do not have a root agent or if the root */
	/* and local agents are the same                                */

    if (root.getServerName().isEmpty()) {
	return 0;
    }

    if ((root.getServerIp() == local.getServerIp()) && (root.getId() == local.getId())) {
	return 0;
    }

	/* if the root agent is on the same machine, just send the message */
	/* to the other half of the server				   */

    if (root.getServerIp() == local.getServerIp()) {

	char *stringCopy;
	MESSAGE *response;

	     /* make a copy of the string since agent_set::send does */
	     /* not make a copy itself -- do not delete stringCopy   */
	     /* ditto for the local id                               */

#ifdef FIX_LATER
	    /* we should not even be constructing a message here */
#endif

	stringCopy = strcpyWithAlloc (string);

	     /* construct the message */

	MESSAGE message (messagesToServer.messages[REQ_ERROR]);
	message.addId (0, new AgentId (local));
	message.addString (1, NULL); 
	message.addLong (2, root.getId());
	message.addLong (3, 1);
	message.addString (4, stringCopy);

	    /* send the message to the other server process */

	if ((response = piped (message, security)) == NULL) {
	    return (e_HANDLER_ERROR);
	}

	DELETE_OBJECT(response);
	return (e_HANDLER_OK);
    }

	/* if the root agent is on a different machine, we must transmit   */
	/* the message with signatures and encryption if desired           */

    int tries = 0;
    int sleepTime = initialSleepTime;
    struct timeval connectionTv = {maximumConnectionTime, 0};
    MachineId remoteMachine (root.getServerName(), root.getServerIp());

    while (1) {

	    /* connect to the remote machine */

	TcpipClientConnection connection;

	struct timeval stop =
	    TimevalUtil::addTimevals (SystemUtil::getCurrentWall(), connectionTv);

	if (connection.connect (remoteMachine, g_serverData -> socketPort, stop) != Connection::e_Ok) {

	    if (++tries > maximumTries) {
		break;
	    }

	    sleep (sleepTime);
	    sleepTime *= 2;
	    continue;
	}

	    /* send the message */

	MESSAGE message (messagesToServer.messages[REQ_MESSAGE]);
	message.addId (0, &local);
	message.addString (1, NULL); 
	message.addLong (2, root.getId());
	message.addLong (3, 1);
	message.addString (4, string);

	if (MessageDriver::secureSend (connection, remoteMachine, security, message) != MESSAGE_OK) {
	    break;
	}

	    /* done */

	return (0);
    }

	/* log an error message if we were unable to send */

    DynamicString serverString = remoteMachine.getName();
    char *agentString          = Agent_IdToString (&local);

    char *argv[] = {
	"handler: ERROR: unable to send abnormal termination message for agent \"",
	agentString,
	"\" to machine \"",
	serverString.value(),
	"\"",
	((char *) NULL)
    };

    g_serverData -> errorLog.error_app_cont (argv);
    DELETE_OBJECT(agentString);
    return (-1);
}

/* checkAbnormalExit 

   Purpose: Check the exit status of the agent interpreter and send an error
            message to the root agent if the exit code was not 0 or if the
            exit was caused by a signal

     Input: root     = the root agent
		       (const AgentId&) 

	    local    = the local agent that caused the error
		       (const AgentId&)

	    security = security information for the local agent
		       (const BASE_SECURITY &)

	    status   = exit status of the agent interpreter
		       (WAIT_STATUS_TYPE)

    Output: The procedure returns -1 on error and 0 otherwise.
*/

int checkAbnormalExit (AgentId& root, AgentId& local, BASE_SECURITY &security, WAIT_STATUS_TYPE status)
{
    int rc;

	/* no error message on a normal exit */

    if (WIFEXITED(status) && (WEXITSTATUS(status) == 0)) {
	return 0;
    }

	/* construct the error message */

    DynamicString errorString;
    errorString.append ("abnormal agent exit: ");

    if (WIFEXITED(status)) {

	int exitcode = WEXITSTATUS(status);

	switch (exitcode) {

	    case ServerInterface::e_GLOBAL_WALL_VIOLATION:
		errorString.append ("violation of global wall-time permit");
		break;

	    case ServerInterface::e_GLOBAL_CPU_VIOLATION:
		errorString.append ("violation of global cpu-time permit");
		break;

	    case ServerInterface::e_GLOBAL_JUMPS_VIOLATION:
		errorString.append ("violation of global jumps permit");
		break;

	    case ServerInterface::e_GLOBAL_CHILDREN_VIOLATION:
		errorString.append ("violation of global children permit");
		break;

	    case ServerInterface::e_GLOBAL_DEPTH_VIOLATION:
		errorString.append ("violation of global depth permit");
		break;

	    case ServerInterface::e_CAN_NOT_RESTORE_STATE:
		errorString.append ("can not restore agent state");
		break;

	    case ServerInterface::e_RECEIVED_SIGTERM:
		errorString.append ("agent was forced or otherwise killed");
		break;

	    default:
		char temp[16];
		sprintf (temp, "%d", exitcode);
		errorString.append ("exited with code ");
		errorString.append (temp);
	}
	
    } else if (WIFSIGNALED (status)) {

	char temp[16];
	sprintf (temp, "%d", WTERMSIG(status));
	errorString.append ("killed with signal ");
	errorString.append (temp);

#ifdef WCOREDUMP
	if (WCOREDUMP(status)) {
	    errorString.append (" (core dump performed)");
	}
#endif

    } else if (WIFSTOPPED (status)) {

	char temp[16];
	sprintf (temp, "%d", WSTOPSIG(status));
	errorString.append ("stopped with signal "); 
	errorString.append (temp);

    } else {

	errorString.append ("unknown exit status");
    } 

	/* send the error message */

    rc = sendErrorMessage (errorString.value(), root, local, security);

	/* log the error message */

    char *stringId = Agent_IdToString (&local);

    char *argv[] = {
	"agent \"",
	stringId,
	"\": ",
	errorString.value(),
	((char *) NULL)
    };

    g_serverData -> errorLog.error_app_cont (argv);
    DELETE_OBJECT(stringId);

	/* done */

    return (rc);
}
 
/* removeAgent

   Purpose: Remove an agent from the server tables.

     Input: id     = numeric id of the agent
		     (UINT_32)

	    sender = security information for the agent that is ending
		     (BASE_SECURITY &)

    Output: The procedure returns -1 on error.  Otherwise the procedure
            removes the agent from the tables and returns 0.
*/

static int removeAgent (UINT_32 id, BASE_SECURITY &sender)
{
    int code;
    MESSAGE *response;

	// construct the REQ_END message

#ifdef FIX_LATER
	// there is no need to actually construct a message
#endif

    MESSAGE message (messagesToServer.messages[REQ_END]);
    message.addString (0, NULL);
    message.addLong (1, id);

	// do the REQ_END

    response = piped (message, sender);
  
    if ((response == NULL) || (response -> getFlag() != RESP_ID)) {
	code = e_HANDLER_ERROR;
    } else {
	code = e_HANDLER_OK;
    }

    DELETE_OBJECT(response);
    return (code);
}

/* getLocalId, registerNewAgent, and replyWithAgentId

   Purpose: Together these three functions allow a new agent to obtain
            an identification and send that identification back to the
            parent (or registering) agent.
*/

static MESSAGE *getLocalId (BASE_SECURITY &sender, pid_t pid)
{
	/* set up the message */

#ifdef FIX_LATER
	/* 
	 * There is no need to actually construct a message. 
	 */
#endif

    MESSAGE message (messagesToServer.messages[REQ_BEGIN]);
    message.addString (0, NULL);
    message.addIp (1, 0);
    message.addLong (2, pid);

	/* send the message to the server */

    MESSAGE *response = piped (message, sender);
    return (response);
}

static AgentId *registerNewAgent (int sockfd, MonitorLock *sockfdLock, BASE_SECURITY &sender, pid_t pid, UINT_32 &randSize, char *&randBuffer) 
{
    AgentId *agentId = (AgentId *) NULL;
    MESSAGE *message = (MESSAGE *) NULL;

	/* get a local id for the agent and return that id */

    if ((message = getLocalId (sender, pid)) == NULL) {

	respondWithError (sockfd, sockfdLock, ServerInterface::e_SERV_ERR_UNKNOWN);

    } else if (message -> getFlag() != RESP_BEGIN) {

	if (message -> getFlag() == RESP_ERROR) {
	    respondWithError (sockfd, sockfdLock, message -> getByte(0));
	} else {
	    respondWithError (sockfd, sockfdLock, ServerInterface::e_SERV_ERR_UNKNOWN);
	}

    } else {

	agentId    = message->getId(0);
	randBuffer = message->getBinary(1,randSize);
    }

    DELETE_OBJECT(message);
    return (agentId);
}

static void replyWithId (int sockfd, MonitorLock *sockfdLock, AgentId *agentId)
{
	/* construct the AgentId response */

    MESSAGE *response = new MESSAGE (messagesFromServer.messages[RESP_ID]);
    response -> addId (0, agentId, e_FALSE);

	/* send the AgentId response */

    if (sockfdLock != NULL) {
	sockfdLock->acquire();
    }
	
    message_send (sockfd, *response);

    if (sockfdLock != NULL) {
	sockfdLock->release();
    }
	
    DELETE_OBJECT(response);
}
 
/* handleNewAgent

   Purpose: This procedure handles the messages

            REQ_SUBMIT

     Input: sockfd     = socket on which the script is arriving
                         (int)
 
	    sockfdLock = lock to provide exclusive access to the sockfd
			 (NULL if not needed)
			 (MonitorLock *)

            sender     = security description of the sender
                         (class BASE_SECURITY &)

            message    = the message
	                 (class MESSAGE *)

	    background = e_TRUE if the agent arrived on the background
			 connection
			 (BOOLEAN)
 
	    timestamp  = time at which the agent was received
			 (struct timeval &)
  
    Output: The procedure returns -1 if it is unable to start execution.
            Otherwise the procedure returns 0.  If the request for a new
	    agent came in over a background connection, the procedure
	    makes a thread so that the new agent has a separate background
            handler.
*/

class NewAgentBody: public ThreadBody
{
	// return code

    int 	  m_rc;
    int           m_rcParent;

	// request information

    int           m_sockfd;
    MonitorLock  *m_sockfdLock;
    BASE_SECURITY m_security;
    MESSAGE      *m_message;
    struct timeval m_timestamp;

	// monitor that we use to signal the parent thread to proceed

    MonitorLock   m_proceedLock;
    BOOLEAN       m_proceed;

	// registration response

    UINT_32 m_randSize;
    char *m_randBuffer;

	// data that is initialized in forkAgentProcess for use in
	// initializeAgentProcess

    ServerInterface::SubmissionTypes m_type;	// type of incoming agent
    AgentId *m_root;				// root identification
    AgentId *m_rootHome;			// root home identification
    AgentId *m_sourceHome;			// source home identification
    UINT_32  m_scriptSize;			// size of the agent code
    char    *m_script;				// agent code
    UINT_32  m_restrictSize;			// size of the restrictions
    char    *m_restrictBuffer;			// restrictions
    UINT_32  m_masksSize;                       // size of the masks
    char    *m_masksBuffer;                     // masks
    int      m_backgroundFd;		 	// fd that connects to the agent process 
    pid_t    m_agentPid;			// process id of agent
    char    *m_language;			// agent language

	// data that is initialized in initializeAgentProcess for use in "run"
	
    AgentId *m_local;				// local identification

private: 

    int forkAgentProcess (void) {

	char temp[256];

	    /* extract the root identification and language name */

	m_type		      = (ServerInterface::SubmissionTypes) m_message -> getByte (0);
	m_root		      = m_message -> getId (1);
	m_rootHome	      = m_message -> getId (2);
	SmartIdPtr source      (m_message -> getId (3));
	m_sourceHome	      = m_message -> getId (4);
	m_language            = m_message -> getString (5);
        m_script              = m_message -> getBinary (6, m_scriptSize); 
	m_restrictBuffer      = m_message -> getBinary (7, m_restrictSize);
        m_masksBuffer         = m_message -> getBinary (8, m_masksSize);

	    /* make sure that we have a valid type flag */

        if (ServerInterface::isSubmissionTypeOutOfRange (m_type)) {
	    respondWithError (m_sockfd, m_sockfdLock, ServerInterface::e_SERV_ERR_INVALID);
	    return (e_HANDLER_ERROR);
        }

	    /* get a ready interpreter */

	HotInterpreters::ErrorCodes rc;
	HotInterpreters::ReadyInterpreter readyInterpreter;

	INTERP::InterpreterTypes type =
	    ((m_type == ServerInterface::e_SUB_JUMP) && 
	     (m_type == ServerInterface::e_SUB_FORK)) ?
	    INTERP::e_STATE_INTERP : INTERP::e_NORMAL_INTERP;

	if (m_language == NULL) {

	    rc = HotInterpreters::e_UnknownName;

	} else {

	    rc = hotInterpreters -> getReadyInterpreter 
		(m_language, type, readyInterpreter);
	}

	if (rc == HotInterpreters::e_UnknownName) {
	    sprintf (temp, "no interpreter for language \"%s\"", m_language); 
	    g_serverData -> errorLog.error_sys_cont (temp);
	    respondWithError (m_sockfd, m_sockfdLock, ServerInterface::e_SERV_ERR_INTERP);
	    return (e_HANDLER_ERROR);
	}

	if (rc == HotInterpreters::e_WrongType) {
	    sprintf (temp, "interpreter for language \"%s\" does not accept state images", m_language);
	    g_serverData -> errorLog.error_sys_cont (temp);
	    respondWithError (m_sockfd, m_sockfdLock, ServerInterface::e_SERV_ERR_UNKNOWN);
	    return (e_HANDLER_ERROR);
	}

	if (rc == HotInterpreters::e_NoInterpreter) {

#ifdef FIX_LATER 
	    // do something here
#endif
	    return (e_HANDLER_ERROR);
	}

	if (rc != HotInterpreters::e_Ok) {

#ifdef FIX_LATER 
	    // do something here
#endif
	    return (e_HANDLER_ERROR);
	}

	    /* remember the file descriptor and return */

	m_backgroundFd = readyInterpreter.fd;
	m_agentPid     = readyInterpreter.pid;
	return (e_HANDLER_OK);
    }

    int initializeAgentProcess (void) {

	    /* get the trusted status of the agent's sending machine AND  */
	    /* THEN slide the current machine into the security structure */

	UINT_16 trustedFlag = 0;

	if (m_security.isMachineTrusted() && m_security.isMachineAuthenticated()) {
	    trustedFlag = e_MACHINE_TRUSTED;
	} else if (m_security.isMachineTrusted()) {
	    trustedFlag = e_MACHINE_TRUSTED_BUT_UNVERIFIED;
	} else {
	    trustedFlag = e_MACHINE_UNTRUSTED;
        }

        m_security.setAgentAuthenticated (e_TRUE, m_local -> getId());
        m_security.transferMachine (g_serverData -> security);
        m_security.setMachineTrusted (e_TRUE);
        m_security.setOwnerSigned (e_FALSE);
 
	     /* send down the message */

        MESSAGE interpMessage (messagesToInterpreter.messages[INTERP_SUBMIT]);
        interpMessage.addByte     ( 0, m_type);
        interpMessage.addId       ( 1, m_root);
        interpMessage.addId       ( 2, m_rootHome);
        interpMessage.addId       ( 3, m_local);
        interpMessage.addId       ( 4, 
	    ((m_type == ServerInterface::e_SUB_JUMP) ||
	     (m_type == ServerInterface::e_SUB_AGLET_JUMP)) ? 
	    m_sourceHome : m_local);
        interpMessage.addBinary   ( 5, m_scriptSize, m_script);
        interpMessage.addSecurity ( 6, &m_security);
        interpMessage.addBinary   ( 7, m_randSize, m_randBuffer);
        interpMessage.addBinary   ( 8, m_restrictSize, m_restrictBuffer);
        interpMessage.addBinary   ( 9, m_masksSize, m_masksBuffer);
        interpMessage.addByte     (10, trustedFlag);

	if (m_security.isOwnerAuthenticated ()) {
	    interpMessage.addPermit (11, &g_serverData -> authorizedPermit);
	} else {
	    interpMessage.addPermit (11, &g_serverData -> anonymousPermit);
	}

	interpMessage.addString (12, g_serverData -> tempDirectory.value());

	if (message_send (m_backgroundFd, interpMessage) < 0) {
	    sendErrorMessage (transferError, *m_root, *m_local, m_security);
	    return (e_HANDLER_ERROR);
        }

	    /* done */

	return (e_HANDLER_OK);
    }

public:

    NewAgentBody (int sockfd, MonitorLock *sockfdLock, const BASE_SECURITY &security, MESSAGE *message, struct timeval &timestamp):
	m_rc		  (0),
	m_rcParent	  (0),
	m_sockfd	  (sockfd),
	m_sockfdLock	  (sockfdLock),
	m_security	  (security),
	m_message	  (message),
	m_timestamp	  (timestamp),
	m_proceedLock     (),
	m_proceed	  (e_FALSE),
	m_randBuffer      (NULL),
	m_type		  ((ServerInterface::SubmissionTypes) 0),
	m_root		  (NULL),
	m_rootHome	  (NULL),
	m_sourceHome      (NULL),
	m_scriptSize	  (0),
	m_script          (NULL),
	m_restrictSize    (0),
	m_restrictBuffer  (NULL),
	m_masksSize       (0),
	m_masksBuffer	  (NULL),
	m_backgroundFd    (0),
	m_agentPid	  (0),
	m_language        (NULL),
	m_local           (NULL)
    {
	// empty
    }

    void freeMessageItems (void) {
	DELETE_ARRAY_OBJECT(m_randBuffer);
	DELETE_OBJECT(m_rootHome);
	DELETE_OBJECT(m_sourceHome);
	DELETE_OBJECT(m_script);
	DELETE_ARRAY_OBJECT(m_restrictBuffer);
	DELETE_ARRAY_OBJECT(m_masksBuffer);
	m_randBuffer     = NULL;
	m_rootHome	 = NULL;
	m_sourceHome	 = NULL;
	m_script         = NULL;
	m_restrictBuffer = NULL;
	m_masksBuffer    = NULL;
    }

   ~NewAgentBody () {
	freeMessageItems();
	DELETE_OBJECT(m_root);
	DELETE_OBJECT(m_local);
	DELETE_OBJECT(m_language);
	m_root = NULL;
	m_local = NULL;
	m_language = NULL;
    }

    void acquireProceedLock (void) {
	m_proceedLock.acquire();
    }

    BOOLEAN canParentProceed (void) {
	return (m_proceed);
    }

    int getRcParent (void) {
	return (m_rcParent);
    }

    int getRc (void) {
	return (m_rc);
    }

    void waitOnProceedLock (void) {
	m_proceedLock.wait();
    }

    void releaseProceedLock (void) {
	m_proceedLock.release();
    }

    struct TimestampEntry {
	UINT_32 id;
	UINT_8 type;
	UINT_32 tv_sec;
	UINT_32 tv_usec;
    };

    void run (void) {

	    // get an identification for the new agent; if we were able to 
	    // get an identification for the new agent, remember the
	    // identification and fork an interpreter for the new agent;
	    // if we were unable to fork an interpreter, remove the agent
	    // identification from the tables

	if ((m_local = registerNewAgent (m_sockfd, m_sockfdLock, m_security, m_agentPid, m_randSize, m_randBuffer)) == (AgentId *) NULL) {

	    m_rc = e_HANDLER_ERROR;

        } else {

	    if (g_serverData->logTimestamps == e_TRUE) {
		TimestampEntry entry;
	        entry.id = htonl (m_local->getId());
		entry.type = 0;
		entry.tv_sec = htonl (m_timestamp.tv_sec);
		entry.tv_usec = htonl (m_timestamp.tv_usec);
		write (g_serverData->timestampFd, (char *) &entry, 
		    sizeof(TimestampEntry));
	    }
	
	    if ((m_rc = forkAgentProcess()) != e_HANDLER_OK) {
		removeAgent (m_local -> getId(), m_security);
	    } else {
		replyWithId (m_sockfd, m_sockfdLock, m_local);
	    }
	}

	    // signal the parent thread that it can proceed

#ifdef FIX_LATER
	    // we only need to signal the parent if there is a parent to
	    // signal
#endif

	m_proceedLock.acquire();
	m_rcParent = m_rc;
	m_proceed = e_TRUE;
	m_proceedLock.wakeup();
	m_proceedLock.release();

	    // if we had any error, bail

	if (m_rc != e_HANDLER_OK) {
	    return;
	}

	    // initialize the agent process; if we were unable to initialize
	    // the agent process, cleanup and bail

#ifdef FIX_LATER
	    // the body of this IF statement is nearly identical to code that 
	    // appears below
#endif

	if ((m_rc = initializeAgentProcess ()) != e_HANDLER_OK) {

		// close the background connection
	    
	    PipeUtility::closePipe (m_backgroundFd);

		// wait for the interpreter to exit

	    WAIT_STATUS_TYPE status;
	    HotInterpreters::ErrorCodes rc =
		hotInterpreters -> waitForInterpreter (m_agentPid, status);
	    assert (rc == HotInterpreters::e_Ok);

		// remove the agent from the tables
	
	    removeAgent (m_local -> getId(), m_security);
	    return;
	}

	    // delete message items that we do not need any more

	freeMessageItems();

	    // background connection that handles all of the agent's requests

	m_rc = backgroundConnection
	    (m_security, m_backgroundFd, m_local -> getId());

	    // if the background connection failed, close the socket
	    // descriptor, wait for the interpreter process to terminate,
	    // and send an abnormal termination message to the agent's
	    // creator if necessary; otherwise return the interpreter to the
	    // hot interpreter pool so that it can be used again

	if (m_rc != e_HANDLER_BACKGROUND_ENDED) {
	
		// close the background connection

	    PipeUtility::closePipe (m_backgroundFd);

		// wait for the interpreter process to finish

	    WAIT_STATUS_TYPE status;
	    HotInterpreters::ErrorCodes rc =
		hotInterpreters -> waitForInterpreter (m_agentPid, status);
	    assert (rc == HotInterpreters::e_Ok);
	    checkAbnormalExit (*m_root, *m_local, m_security, status);

	        // remove the agent from the server tables

	    removeAgent (m_local -> getId(), m_security);

	} else {

		// return the interpreter to the hot interpreter pool

	    HotInterpreters::ReadyInterpreter readyInterpreter 
		(m_agentPid, m_backgroundFd);
	    HotInterpreters::ErrorCodes rc =
		hotInterpreters->returnReadyInterpreter 
		    (m_language, readyInterpreter);
	   
		// if the hot interpreter pool refused to take the interpreter
		// back, we close the background connection and wait for the
		// interpreter to exit

	    if (rc != HotInterpreters::e_Ok) {
		WAIT_STATUS_TYPE status;
		PipeUtility::closePipe (m_backgroundFd);
		(void) hotInterpreters ->
		    waitForInterpreter (m_agentPid, status);
	    }
	}

	    // delete the agent's temporary directory 

	char temp[16];
	sprintf (temp, "%u", m_local -> getId());
	DynamicString directory = g_serverData -> tempDirectory;
	directory.append ("/");
	directory.append (temp);

	if (removeDirectory (directory) < 0) {

	    char *argv[] = {
		"handler: ERROR: unable to remove agent temporary directory \"",
		directory.value(),
		"\"",
		(char *) NULL
	    };

	    g_serverData -> errorLog.error_app_cont (argv);
	}
    }
};

int handleNewAgent (int sockfd, MonitorLock *sockfdLock, BASE_SECURITY &security, MESSAGE *message, BOOLEAN background, struct timeval &timestamp)
{
    int rc;

	/* If the request arrived on a background connection, we create a */
	/* thread for the new background handler.  If the request arrived */
	/* on a non-background connection, we do not create a thread;     */
	/* instead we just use the current thread as the background       */
	/* handler.                                                       */

    if (background) {

	    // create the background thread -- remember that the Thread instance
	    // will delete the NewAgentBody instance

	NewAgentBody *newAgentBody = new NewAgentBody (sockfd, sockfdLock, security, message, timestamp);
	Thread thread ("request-handler", Thread::e_CLIENT_THREAD, newAgentBody);

	    // run the thread and wait for it to tell us that we can proceed

	newAgentBody -> acquireProceedLock();

	thread.resume ();

	while (!(newAgentBody -> canParentProceed())) {
	    newAgentBody -> waitOnProceedLock();
	}

	rc = newAgentBody -> getRcParent();
	newAgentBody -> releaseProceedLock();

    } else {

	    // use the current thread as the background thread

	NewAgentBody *newAgentBody = new NewAgentBody (sockfd, sockfdLock, security, message, timestamp);
	newAgentBody -> run();
	rc = newAgentBody -> getRc();
	DELETE_OBJECT(newAgentBody);
    }

	// done

    return (rc);
}

/* handleSend

   Purpose: This procedure handles the messages

            REQ_BEGIN
            REQ_NAME
            REQ_END
	    REQ_FORCE
            REQ_MESSAGE
            REQ_MEETING
            REQ_EVENT
	    REQ_INFO
	    REQ_STATUS
	    REQ_IP

     Input: sockfd     = socket descriptor
		         (int)

	    sockfdLock = lock to provide exclusive access to the sockfd
			 (NULL if not needed)
			 (MonitorLock *)

            security   = security description of the sender
	                 (class BASE_SECURITY &)

            message    = the message
		         (class MESSAGE *)

	    background = e_TRUE if request arrived over the background
			 connection
			 (BOOLEAN)
 
	    timestamp  = time at which the message was received
			 (struct timeval &)
  
    Output: The procedure returns -1 on error, 0 on a success that should
	    not cause the background handler to terminate, and 1 on a 
	    success that should cause the background handler to terminate
	    (i.e. a successful AGENT_END).  newHandler is always set to NULL.
*/

int handleSend (int sockfd, MonitorLock *sockfdLock, BASE_SECURITY &sender, MESSAGE *message, BOOLEAN background, struct timeval &timestamp)
{
    int rc;
    MESSAGE *response;

	/* 
	 * assertions
	 */

    assert (message != NULL);

	/* 
	 * handle the request
	 */

    response = piped (*message, sender);
    assert (response != NULL);

	/*
	 * send the result done the socket 
	 */

    if (sockfdLock != NULL) {
	sockfdLock -> acquire();
    }

    (void) message_send (sockfd, *response);

    if (sockfdLock != NULL) {
	sockfdLock -> release();
    }

	/* 
	 * we are going to become a background handler if this is an 
	 * AGENT_BEGIN, or we are going to stop being a background 
	 * handler if this is an AGENT_END
	 */

    if ((message -> getFlag() == REQ_BEGIN) && (response -> getFlag() == RESP_BEGIN)) {

	    /* set up information about the new background handler */

	AgentId *localId = response -> getId (0);
	UINT_32 id = localId -> getId();
	DELETE_OBJECT(localId);

	    /* fill in the machine part of security */

	sender.transferMachine (g_serverData -> security);
	sender.setMachineTrusted (e_FALSE);
	sender.setAgentAuthenticated (e_TRUE, id);

	    /* create the new background handler */

	rc = backgroundConnection (sender, sockfd, id);

	    /*  remove the agent from the tables */       

	if (rc == e_HANDLER_BACKGROUND_ENDED) {
	    rc = e_HANDLER_OK;
	} else {
	    removeAgent (id, sender);
	}

    } else if ((message -> getFlag() == REQ_END) && (response -> getFlag() == RESP_ID)) {

	assert (background == e_TRUE);
	rc = e_HANDLER_BACKGROUND_ENDED;

    } else {

	rc = e_HANDLER_OK;
    }

	/* cleanup -- do NOT close the socket */

    DELETE_OBJECT(response);
    return (rc); 
}

/* backgroundConnection

   Purpose: This procedure is the background handler associated with each
	    agent.  It is established on agent creation and then handles
	    all further agent requests and forwards incoming communication to
	    the agent. 

     Input: sender  = security description of the sender
                      (class BASE_SECURITY &)
	
	    sockfd  = socket that connects us to the interpreter process
		      (int)

	    localId = numeric id of the agent executing inside the 
		      interpreter process
		      (UINT_32)
 
    Output: The procedure returns e_HANDLER_BACKGROUND_FAILED if the
	    background connection fails for any reason.  Otherwise the
	    procedure returns e_HANDLER_BACKGROUND_ENDED as soon as the
	    agent (and thus the background handler) has finished.  Until
	    one of these two things occur, the procedure sends all
	    incoming messages, meeting and events (from other agents) down
            the socket to the agent.  The procedure also handles any requests 
            that arrive on the socket.
*/

static int backgroundConnection (BASE_SECURITY &sender, int sockfd, UINT_32 localId)
{
    int rc;
    int selectRc;
    fd_mask readSet[MASK_SIZE];

	/* lock for mutual exclusion */

    MonitorLock sigioLock;

	/* tell the message-handling thread about our lock and sockfd */

    piped_setBackgroundConnection (localId, sockfd, sigioLock);

	/* loop until the agent dies, the server fails, the machine  */
	/* blows up, or the Earth crashes into the Sun, handling all */
	/* messages sent from the agent                              */

    while (1) {

	    // wait for an incoming message

	FDMASK_ZERO (readSet);
	FDMASK_SET (sockfd, readSet);

	do {
	    selectRc = select (sockfd + 1, (SELECT_MASK *) &readSet, (SELECT_MASK *) NULL, (SELECT_MASK *) NULL, (struct timeval *) NULL);
	} while ((selectRc < 0) && ((errno == EINTR) || (errno == EAGAIN)));

	    // handle the incoming message

	if (selectRc > 0) {

	    MESSAGE *backgroundMessage;

	    if ((backgroundMessage = message_receive (sockfd, messagesToServer)) == NULL) {
		rc = e_HANDLER_BACKGROUND_FAILED;
		break;
	    }

	    rc = handleBackgroundClientRequest 
		(sockfd, backgroundMessage, sender, sigioLock);
 	    DELETE_OBJECT(backgroundMessage);

	    if (rc == e_HANDLER_BACKGROUND_ENDED) {
		break;
	    }
	}
    }

	/* done */

    return (rc);
}
