/* Agent Tcl
   Bob Gray
   11 April 1997

   servRandom.cc

   This file implements a class that controls access to a file of random bits.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif 

#include "platPorting.h"
#include "suppDString.h"		// DynamicString
#include "genFile.h"		// FileUtility
#include "mesgTcpip.h"		// tcpip_blockingWriten
#include "servRandom.h"		// RandomFile
#include "random.h"		// trueRandGetByte

/* RandomFile::RandomFile

   Purpose: This procedure is the constructor for class RandomFile.

     Input: filename = name of the random bits file
		       (const DynamicString &)

	    bits     = number of bits we expect to find in the file
		       (unsigned)
*/

RandomFile::RandomFile (const DynamicString& filename, unsigned bits):
    m_filename (filename),
    m_bits (bits)
{
    // empty
}

/* RandomFile::burn

   Purpose: Overwrite the random bits file with new random bits

     Input: None 

    Output: On success the procedure overwrites the random bits file and
	    returns e_OK.  On failure the procedure returns one of the error
            codes.
 
      NOTE: The "new" random bits are generated using the old random bits
	    as a seed.  However, due to the cryptographically-secure nature
	    of our random number generator, the "new" random bits do not
	    provide any leverage for an attacker.
*/

const int BUFFER_SIZE = 1024;

RandomFile::RandomFileErrors RandomFile::burnBits (void)
{
    int fd;
    int nwritten;
    char buffer[BUFFER_SIZE];

	/* open the random bits file if we do not already have the fd */

    do {
	fd = FileUtility::open (m_filename.value(), O_WRONLY);
    } while ((fd < 0) && ((errno == EINTR) || (errno == EAGAIN)));

    if (fd < 0) { 
	return e_CANT_OPEN;
    }

	/* calculate the number of random bytes */ 

    int nbytes = m_bits / 8;

    if (m_bits % 8 != 0) {
	nbytes += 1;
    }

	/* write out the random bytes */ 
   
    int i = 0;
 
    for (int totalBytes = 0; totalBytes < nbytes; ++totalBytes) {

	buffer[i] = trueRandByte();

	if (++i == BUFFER_SIZE) {
	    
	    if ((tcpip_blockingWriten (fd, buffer, BUFFER_SIZE, nwritten) != e_TCPIP_OK)
		|| (nwritten < BUFFER_SIZE)) {
		FileUtility::close (fd);
		return e_CANT_WRITE;
	    }

	    i = 0;
	}
    }

    if ((tcpip_blockingWriten (fd, buffer, i, nwritten) != e_TCPIP_OK) 
	|| (nwritten < i)) {
	FileUtility::close (fd);
	return e_CANT_WRITE;
    }

	/* done */

    FileUtility::close (fd);
    return e_OK;
}
