/* Agent Tcl
   Bob Gray
   9 February 1995

   socketd.cc

   This file implements procedure "socketd" which monitors the TCP/IP port.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "platExclusion.h"
#include "platShortExclusion.h"
#include "platThread.h"
#include "platSystem.h"
#include "mesgTcpip.h"
#include "agentd.h"
#include "servConf.h"
#include "servProcess.h"
#include "servReceive.h"
#include "socketd.h"
#include "truefalse.h"

/*
 * class JobHandlerBody is a handler thread that waits for an accepted socket
 * and then handles the request that is arriving on that socket.
 */

class JobHandlerBody: public ThreadBody, public OnceTrap
{
    	// queue of ready sockets and associated locks

    static MonitorOnce::OnceControl m_onceControl;
    static MutexLock *m_unixLock;
    static MutexLock *m_tcpipLock;

	// e_TRUE if this handler is for the Unix socket

    BOOLEAN m_isUnixSocket;
 
protected:

	// one-time initialization

    void initialize (void) {
	m_unixLock  = new MutexLock ();
	m_tcpipLock = new MutexLock ();
    };

public:

	// constructor

    JobHandlerBody (BOOLEAN isUnixSocket) {

	if (m_onceControl != MonitorOnce::e_DONE) {
	   MonitorOnce::doOnce (m_onceControl, *this);
	}

	m_isUnixSocket = isUnixSocket;
    }

	// thread body

    void run (void) {

	int newSockfd;

	    /* handle incoming requests */

	while (1) {

		/* acquire the lock, wait for a connection and release */
		/* the lock                                            */

	    TcpipErrors tcpipRc;

	    if (m_isUnixSocket) {

		m_unixLock -> acquire();
		tcpipRc = unix_blockingAccept (g_serverData -> unixSockfd, newSockfd);
		m_unixLock -> release();

	    } else {

		m_tcpipLock -> acquire();
		tcpipRc = tcpip_blockingAccept (g_serverData -> sockfd, newSockfd);
		m_tcpipLock -> release();
	    }

		/* handle the request */

	    if (tcpipRc == e_TCPIP_OK) {

		(void) handleClientRequest (newSockfd, m_isUnixSocket);

	  	if (m_isUnixSocket) {
		    unix_close (newSockfd);
		} else {
		    tcpip_close (newSockfd);
		}
	    }
	}
    }
};

MonitorOnce::OnceControl JobHandlerBody::m_onceControl
	= MonitorOnce::e_NEVER;

MutexLock *JobHandlerBody::m_unixLock
	= NULL;

MutexLock *JobHandlerBody::m_tcpipLock
	= NULL;

/*
 * Socketd starts up the initial pool of handler threads (and starts
 * additional handler threads as existing handler threads become long-term
 * background handlers for agents)
 */

int Socketd::m_unixHandlers 
	= 0;

int Socketd::m_tcpipHandlers
	= 0;

ShortMutexLock *Socketd::m_countLock
	= (ShortMutexLock *) NULL;

MonitorLock *Socketd::m_monitor
	= (MonitorLock *) NULL;

void Socketd::socketd (void)
{
	// create the locks

    m_countLock = new ShortMutexLock;
    m_monitor   = new MonitorLock;

	// loop forever

    while (1) {

	    // create Unix and TCP/IP handlers until we reach the minimum 
	    // number of handlers

	BOOLEAN createUnixHandler;
	BOOLEAN createTcpipHandler;

	do {

		// see if we need to create another handler

	    createUnixHandler  = e_FALSE;
	    createTcpipHandler = e_FALSE;

	    m_countLock -> acquire();

	    if (m_unixHandlers < c_minimumUnixHandlers) {
	 	m_unixHandlers += 1;
		createUnixHandler = e_TRUE;
	    }

	    if (m_tcpipHandlers < c_minimumTcpipHandlers) {
		m_tcpipHandlers += 1;
		createTcpipHandler = e_TRUE;
	    }

	    m_countLock -> release();

		// create a handler if we need another one

	    if (createUnixHandler) {
		JobHandlerBody *jobHandler = new JobHandlerBody (e_TRUE);
		Thread thread ("unix-request-thread", Thread::e_CLIENT_THREAD, 
		    jobHandler);
		thread.resume ();
	    }

	    if (createTcpipHandler) {
	 	JobHandlerBody *jobHandler = new JobHandlerBody (e_FALSE);
		Thread thread ("tcpip-request-thread", Thread::e_CLIENT_THREAD, 
		    jobHandler);
		thread.resume ();
	    }

	} while (createUnixHandler || createTcpipHandler);

	    // sleep until we need to create another handler

	BOOLEAN goToSleep = e_FALSE;

	m_countLock -> acquire();

	if ((m_unixHandlers >= c_minimumUnixHandlers) &&
	    (m_tcpipHandlers >= c_minimumTcpipHandlers)) {
	    goToSleep = e_TRUE;
	}

	m_countLock -> release();

	if (goToSleep) {
	    m_monitor -> wait();
	}

	m_monitor -> release();
    }
}

void Socketd::handlerBecomesBackground (BOOLEAN isUnixSocket)
{
    BOOLEAN handlerNeeded = e_FALSE;

    m_monitor   -> acquire();
    m_countLock -> acquire();

    if (isUnixSocket) {

	m_unixHandlers -= 1;	

	if (m_unixHandlers < c_minimumUnixHandlers) {
	    handlerNeeded = e_TRUE;
	}
 
    } else {

	m_tcpipHandlers -= 1;

	if (m_tcpipHandlers < c_minimumTcpipHandlers) {
	    handlerNeeded = e_TRUE;
	}
    }

    m_countLock -> release();

    if (handlerNeeded) {
	m_monitor -> wakeup();
    }

    m_monitor -> release();
}

BOOLEAN Socketd::backgroundBecomesHandler (BOOLEAN isUnixSocket)
{
    BOOLEAN die = e_FALSE;

    m_countLock -> acquire();

    if (isUnixSocket) {

	if (m_unixHandlers >= c_maximumUnixHandlers) {
	    die = e_TRUE;
	} else {
	    m_unixHandlers += 1;
	}

    } else {

	if (m_tcpipHandlers >= c_maximumTcpipHandlers) {
	    die = e_TRUE;
	} else {
	    m_tcpipHandlers += 1;
	}
    }

    m_countLock -> release();
    return (die);
}
