/* Agent Tcl
   Bob Gray
   15 February 1995

   agent_set.cc

   This file implements the structures that keep track of agent data.

   Copyright (c) 1995-1998, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"	// platform-specific definitions
#include "suppStrings.h"
#include "mesgPort.h"		// RemotePort
#include "agentAgentId.h"	// AgentId

#include "agent_set.h"
#include "agentd.h"
#include "agentId.h"
#include "agentSecure.h"
#include "genConversion.h"
#include "genError.h"
#include "suppHash.h"
#include "genManager.h"
#include "genMessage.h"
#include "genSecurity.h"
#include "genServerInterface.h"
#include "servConf.h"
#include "noise.h"
#include "random.h"
#include "truefalse.h"

    /*
     * number of random bytes generated for each agent
     */

const UINT_8 c_RANDOM_BYTES_SIZE = 64;

/* makeErrorResponse

   Purpose: Construct an error response

     Input: code = the error code
		   (UINT_8)

    Output: The procedure returns a dynamically allocated MESSAGE structure
	    that contains the error response.
*/

static MESSAGE *makeErrorResponse (UINT_8 code)
{
    MESSAGE *response = new MESSAGE (messagesFromServer.messages[RESP_ERROR]);
    response -> addByte (0, code);
    return (response);
}

/* AgentSet::lookupCompleteSecurity

   Purpose: Get the complete security information for a local agent

     Input: security = the security information that arrived with the request
		       (BASE_SECURITY &)

    Output: The procedure returns a pointer to the BASE_SECURITY instance
	    that contains the complete security information for the local
	    agent. 

      NOTE: Do not delete the returned BASE_SECURITY instance.
*/

BASE_SECURITY *AgentSet::lookupCompleteSecurity (BASE_SECURITY &security)
{
    HashNode *node;
    BASE_SECURITY *actualSecurity; 

    if (!(security.isAgentAuthenticated())) {
	actualSecurity = &security;
    } else if ((node = agent_table.lookup (security.getAgentId(), NULL)) == NULL) {
	actualSecurity = NULL;
    } else {
	actualSecurity = &(((ServerAgent *) node -> data) -> security);
    }

    return (actualSecurity);
}

/* AgentSet::lookupActualId

   Purpose: Look up the actual id of an agent 

     Input: source   = the agent's claimed identity
		       (AgentId *)

            security = the security information
		       (BASE_SECURITY &)

    Output: The procedure returns a pointer to the AgentId instance that
	    contains the actual id of the agent.

      NOTE: Do not delete the returned AgentId instance.
*/

AgentId *AgentSet::lookupActualId (AgentId *source, BASE_SECURITY &security)
{
    HashNode *node;
    AgentId *actualId;

    if (!(security.isAgentAuthenticated())) {
	actualId = source;
    } else if ((node = agent_table.lookup (security.getAgentId(), NULL)) == NULL) {
	actualId = NULL;
    } else {
	actualId = &(((ServerAgent *) node -> data) -> id);
    }

    return (actualId);
}

/* AgentSet::AgentSet
 
   Purpose: This procedure is the constructor for class AgentSet.
*/

AgentSet::AgentSet (void):
	numAgents	(0),
	numAnonAgents	(0),
	next_id		(1),
	agent_table	(Hash_ENTRIES),
	name_table	(Hash_ENTRIES),
	nameEventsTable (EVENT_Hash_ENTRIES)

{
	/* empty */
}

/* ServerAgent::ServerAgent and ServerAgent::~ServerAgent 

   Purpose: These procedures are the constructor and destructor for class
	    ServerAgent.
*/

ServerAgent::ServerAgent (AgentSet *p_agentSet, char *p_server, UINT_32 p_ip, UINT_32 p_pid, const BASE_SECURITY &p_security):
	agentSet	 (p_agentSet),
	id		 (p_server, p_ip, (char *) NULL, 0),
	security	 (p_security),
	creatorSecurity  (p_security),
	items		 (0),
	pid		 (p_pid),
	handlerPid	 (0),
	handlerBlocked	 (e_FALSE),
	numPendingEvents (0)
{
	/* empty */
}

ServerAgent::~ServerAgent ()
{
	// delete queued messages, meetings and events

    INCOMING_MESSAGE *message;
    INCOMING_MEETING *meeting;
    INCOMING_EVENT *event;

    while ((message = (INCOMING_MESSAGE *) messages.dequeue ()) != NULL) {
	delete (message);
    }
 
    while ((meeting = (INCOMING_MEETING *) meetings.dequeue ()) != NULL) {
	delete meeting;
    }

    while ((event = (INCOMING_EVENT *) events.dequeue ()) != NULL) {
	delete event;
    }

	// delete pending status events

    char *pointer;
    PointerIterator iterator (&pendingBirthEvents);

    while (iterator.nextDistinctPointer (pointer) > 0) {

  	HashNode *node = (HashNode *) pointer;
	PendingEvents *l_events = (PendingEvents *) node -> data;
	l_events -> birthNotifications.removeAll ((char *) this);

	if (l_events -> birthNotifications.getPointerCount() == 0) {
	    delete (l_events);
	    agentSet -> nameEventsTable.remove (0, node -> string_key);
	}
    }
} 

/* ServerAgent::addExistsEvent

   Purpose: Add an "exist" event to the agent's event queue

     Input: newAgent = agent that has come into existence
		       (struct AGENT *)

    Output: The procedure adds the "exist" event to the agent's event queue.
*/

void ServerAgent::addExistsEvent (ServerAgent &newAgent)
{
    char *tag;
    char *string;
    AgentId *source;
    INCOMING_EVENT *event;

	/* first construct the event */

    source = new AgentId (serverData -> host.getName().value(), serverData -> host.getIp(), "server", 0);
    tag    = new char[newAgent.id.getName().length() + 8];
    sprintf (tag, "exists:%s", newAgent.id.getName().value());
    string = Agent_IdToString (&newAgent.id);
    event  = new INCOMING_EVENT (source, tag, string, new BASE_SECURITY (serverData -> security));

	/* now add the event to the queue */

    events.enqueue ((void *) event);
    items += 1;

	/* notify the agent handler */

    if (handlerBlocked) {

	if (kill (handlerPid, SIGUSR1) < 0) {
	    char temp[128];
	    sprintf (temp, "agentd: unable to send signal to process with pid %d\n", handlerPid);
	    serverData -> errorLog.error_sys_cont (temp);
	}

	handlerBlocked = e_FALSE;
    }
}

/* AgentSet::lookupRecipient

   Purpose: Lookup up the recipient agent

     Input: id   = numeric id of the destination agent
            name = name of the destination agent (NULL if not specified)

    Output: The procedure returns NULL if it can not find an agent and a 
            pointer to the appropriate AGENT structure if it does find the
            agent.  Note that the procedure will CREATE an agent if a name
            is given but there is no current agent with the name.  This new
            agent will have a numeric id of UNDEFINED_AGENT. 
*/

ServerAgent *AgentSet::lookupRecipient (UINT_32 id, char *name)
{
    ServerAgent *agent;
    HashNode *hash_node;

	/* lookup the agent using its id if no name is specified or */
	/* if the agent is a "server" agent; otherwise look up the  */
	/* agent using its name 				    */ 

    if ((name == NULL) || (*name == '\0')) {
  
	if ((hash_node = agent_table.lookup (id, NULL)) == NULL) {

	    agent = NULL;

	} else {

	    agent = (ServerAgent *) hash_node -> data;
	}

    } else if (!strcmp (name, "server")) {

	if ((hash_node = agent_table.lookup (id, NULL)) == NULL) {

	    agent = NULL;

	} else {

	    agent = (ServerAgent *) hash_node -> data;

	    if (agent -> id.getName() != "server") {
		agent = NULL;
	    }
	}

    } else {

 	if ((hash_node = name_table.lookup (0, name)) == NULL) {

	    agent = NULL;

	} else {

	    agent = (ServerAgent *) hash_node -> data;
	} 
    }      

    return (agent); 
}

/* AgentSet::agent_info

   Purpose: Get information about the registered agents

     Input: message  = request message
		       (class MESSAGE *)

	    [0] = INFO_ALL_IDS, INFO_ALL_NAMES or INFO_ID
            [1] = name of the agent of interest
            [2] = numeric id of the agent of interest

	    security = security information
		       (class BASE_SECURITY &)

    Output: The procedure returns the response that should be sent to
            socketd.
*/

MESSAGE *AgentSet::agent_info (MESSAGE *message, BASE_SECURITY & /*UNUSED*/)
{
    MESSAGE *response;
    ServerAgent *agent;
    unsigned i;
    unsigned count;
    ID_LIST *idList;
    HashNode *node; 
    NAME_LIST *nameList;
    HashSearch *search;

#ifdef FIX_LATER
	/* Right now we perform no security checks here so any agent can  */
	/* "see" every other agent under the server's control.  It will   */
	/* probably be useful to allow certain agents to be "invisible"   */
	/* to certain other agents.  Also INFO_ALL_IDS and INFO_ALL_NAMES */
	/* take a while if there are a lot of agents, and should probably */
	/* be restrict to "administration" agents                         */
#endif
 
	/* break apart the message */

    UINT_8 flag 	= message -> getByte (0);

	/* information about all numeric ids */
 
    if (flag == INFO_ALL_IDS) {

	count  = agent_table.get_entries ();
	idList = new ID_LIST (count);
	search = new HashSearch (&agent_table); 

	for (i = 0; i < count; i++) {
	    node = search -> next ();
	    idList -> ids[i] = (UINT_32) (node -> key);
        }

	response = new MESSAGE (messagesFromServer.messages[RESP_IDS]); 
	response -> addIds (0, idList, e_TRUE);
	return (response); 
    }

	/* information about all symbolic names */

    if (flag == INFO_ALL_NAMES) {
 
	count    = name_table.get_entries ();
	nameList = new NAME_LIST (count);
	search   = new HashSearch (&name_table);

	for (i = 0; i < count; i++) {
	    node = search -> next ();
	    nameList -> fill (i, node -> string_key);
	}

	response = new MESSAGE (messagesFromServer.messages[RESP_NAMES]);
	response -> addNames (0, nameList, e_TRUE);
	return (response);
    }

	/* information about a particular agent */

    if (flag == INFO_ID) {

	    /* break out the symbolic name and numeric id */

	SmartCharPtr name (message -> getString (1));
	UINT_32 id        = message -> getLong (2);

	    /* find the agent in the tables */

	if ((agent = lookupRecipient (id, name)) == NULL) {
	    return (makeErrorResponse (ServerInterface::e_SERV_ERR_NONEXISTENT));
	}

	    /* construct the AgentId */

	AgentId *fullid = new AgentId (serverData -> host.getName(), serverData -> host.getIp(), agent -> id.getName(), agent -> id.getId());

	    /* assemble the info message */
   
	response = new MESSAGE (messagesFromServer.messages[RESP_INFO]);
	response -> addString (0, agent -> id.getServerName().value());
	response -> addIp (1, agent -> id.getServerIp());
	response -> addId (2, fullid, e_TRUE);
	response -> addSecurity (3, &agent -> creatorSecurity);
	response -> addLong (4, agent -> pid);
	response -> addLong (5, agent -> handlerPid);
	response -> addLong (6, (UINT_32) agent -> messages.get_count ());
	response -> addLong (7, (UINT_32) agent -> meetings.get_count ());
	response -> addLong (8, (UINT_32) agent -> events.get_count ());
	return (response);
    } 

	/* invalid flag */ 

    return (makeErrorResponse (ServerInterface::e_SERV_ERR_INVALID));
}
 
/* AgentSet::agent_req

   Purpose: Add a meeting request to an agent's queue

     Input: message  = request message
		       (class MESSAGE *) 

	    [0] = id of the source agent            (AgentId *)
	    [1] = name of the destination agent     (char *)
	    [2] = id of the destination agent       (UINT_32)
	    [3] = type of meeting request           (UINT_8)
	    [4] = sender's meeting handle           (UINT_32)
	    [5] = recipient's meeting handle        (UINT_32)
	    [6] = actual machine of source agent    (char *)
	    [7] = actual IP address of source agent (UINT_32)
	    [8] = port number                       (UINT_32)

	    security = security information
		       (class BASE_SECURITY &)

    Output: The procedure returns the response that should be sent to
            socketd.
*/

MESSAGE *AgentSet::agent_req (MESSAGE *message, BASE_SECURITY &security)
{
    ServerAgent *agent;
    MESSAGE *response;
    RemotePort *port;
    INCOMING_MEETING *inc_meeting;
    BASE_SECURITY *actualSecurity;
    AgentId *actualSource;

	/* break out the pieces of the id */

    SmartIdPtr source_id     (message -> getId (0));
    SmartCharPtr dest_name   (message -> getString (1));
    UINT_32 dest_id          = message -> getLong (2);
    UINT_8 status            = message -> getByte (3);
    UINT_32 localHandle	     = message -> getLong (4);
    UINT_32 remoteHandle     = message -> getLong (5);
    SmartCharPtr actual_name (message -> getString (6));
    UINT_32 actual_ip        = message -> getIp (7);
    UINT_32 actual_port      = message -> getLong (8);

	/* swap the local and remote handles */

    UINT_32 temp = localHandle;
    localHandle  = remoteHandle;
    remoteHandle = temp;

	/* lookup the destination agent */

    if ((agent = lookupRecipient (dest_id, dest_name)) == NULL) {
	return (makeErrorResponse (ServerInterface::e_SERV_ERR_NONEXISTENT));
    }

	/* lookup the actual id of the sending agent */

    if ((actualSource = lookupActualId (source_id, security)) == NULL) {
	return (makeErrorResponse (ServerInterface::e_SERV_ERR_SYSTEM)); 
    }

	/* get complete security information */

    if ((actualSecurity = lookupCompleteSecurity (security)) == NULL) {
	return (makeErrorResponse (ServerInterface::e_SERV_ERR_SYSTEM));
    }

	/* if the message was sent from a remote agent, make sure that */
	/* the agent is not lying about its id                         */

    if (!(actualSecurity -> isAgentAuthenticated())) {

	if ((actualSource -> getServerIp() == serverData -> host.getIp()) ||
	    (actualSource -> getServerName() == serverData -> host.getName()))  
	{
	    return (makeErrorResponse (ServerInterface::e_SERV_ERR_LIAR));
	}
    }

	/* construct the INCOMING_MEETING structure (make a copy of the */
        /* security information since the original copy will go away)   */

    port        = new RemotePort (actual_name, actual_ip, actual_port);
    inc_meeting = new INCOMING_MEETING (new AgentId (*actualSource), status, localHandle, remoteHandle, port, new BASE_SECURITY (*actualSecurity));

	/* add the meeting to the meeting queue and unblock if necessary */

    agent -> meetings.enqueue ((void *) inc_meeting);
    agent -> items += 1;

    if (agent -> handlerBlocked) {

	if (kill (agent -> handlerPid, SIGUSR1) < 0) {

	    char temp[16];
	    sprintf (temp, "%d", agent -> handlerPid);

	    char *argv[] = {
		"agentd: unable to send signal to process with pid ", 
		temp
	    };

	    serverData -> errorLog.error_sys_cont (argv);
	}

	agent -> handlerBlocked = e_FALSE;
    }

	/* done */ 

    response = new MESSAGE (messagesFromServer.messages[RESP_OK]);
    return response;
}

/* AgentSet::agent_begin

   Purpose: Add an agent to the table

     Input: message  = request message
		       (class MESSAGE *) 

	    [0] = name of the machine on which agent is running (string)
            [1] = IP address of the machine on which agent is running (long)
            [2] = process id of the agent (long)
 
	    security = security information
		       (class BASE_SECURITY &)
 
    Output: The procedure finds an unused id, assigns that id to the agent
            and returns the id.  The procedure SHOULD check for the rare
            occurrence of NO id being available but does not (there are
            over 2 billion ids available on a 32-bit machine and anyone who
	    sets the MaxAgents configuration option to over 2 billion is
	    asking for trouble anyways).
*/

MESSAGE * AgentSet::agent_begin (MESSAGE *message, BASE_SECURITY &security)
{
    int there = TRUE;
    MESSAGE *response;
    BASE_SECURITY *actualSecurity;

	/*
	 * security violation if we have reached the maximum number of agents
	 */

    if (numAgents >= serverData -> maxAgents) {
	return (makeErrorResponse (ServerInterface::e_SERV_ERR_MAX_AGENTS));
    }

	/* 
	 * security violation if we have reached the maximum number of
	 * ANONYMOUS agents
	 */

    if ((actualSecurity = lookupCompleteSecurity (security)) == NULL) {
	return (makeErrorResponse (ServerInterface::e_SERV_ERR_SYSTEM));
    }

    if (!(actualSecurity -> isOwnerAuthenticated()) && (numAnonAgents >= serverData -> maxAnonAgents)) {
	return (makeErrorResponse (ServerInterface::e_SERV_ERR_MAX_ANON));
    }

	/* break apart the message */

    SmartCharPtr host (message -> getString (0));
    UINT_32 ip        = message -> getIp (1);
    UINT_32 pid       = message -> getLong (2);

	/* use the server name and ip if not specified in the message */ 

    if ((char *) host == NULL) {
	DynamicString fullHostName = serverData -> host.getName();
	host = strcpyWithAlloc (fullHostName.value());
        ip   = serverData -> host.getIp();
    }

	/* construct the agent entry */

    ServerAgent *agent = new ServerAgent (this, host, ip, pid, *actualSecurity);

	/* find an unused numeric id */

    while (there) {

	agent_table.add (next_id, NULL, agent, there);

	if (!there) {
	    agent -> id.setId (next_id);
	    agent -> security.setAgentAuthenticated (e_TRUE, next_id);
	    agent -> security.transferMachine (serverData -> security);
	}

	next_id = next_id % MAX_32 + 1;
    }

	/* we now have one more agent */

    numAgents += 1;

    if (!(actualSecurity -> isOwnerAuthenticated())) {
	numAnonAgents += 1;
    }

	/* generate the random bytes and the id */

    noise();	// not a lot of entropy but some

    char *buffer = new char [c_RANDOM_BYTES_SIZE];

    for (int i = 0; i < c_RANDOM_BYTES_SIZE; i++) {
  	buffer[i] = trueRandByte ();
    }

    DynamicString idServerName = serverData -> host.getName();
    AgentId *id = new AgentId 
	(idServerName, serverData -> host.getIp(), NULL, agent -> id.getId());

	/* construct the response */

    response = new MESSAGE (messagesFromServer.messages[RESP_BEGIN]);
    response -> addId (0, id, e_TRUE);
    response -> addBinary (1, c_RANDOM_BYTES_SIZE, buffer, e_TRUE);
    return response;
}

/* AgentSet::agent_name

   Purpose: Assign a name to an agent

     Input: message  = request message
		       (class MESSAGE *)

	    [0] = numeric id of the agent                  (long)
            [1] = name that should be asigned to the agent (char)

	    security = security information
		       (class BASE_SECURITY &)

    Output: The procedure returns the response that should be sent to
            socketd.
*/

MESSAGE *AgentSet::agent_name (MESSAGE *message, BASE_SECURITY &security)
{
    MESSAGE *response;    
    ServerAgent *id_agent;    
    ServerAgent *name_agent;  

	// break out the pieces of the message

    UINT_32 id = message -> getLong (0);
    SmartCharPtr name (message -> getString(1));

	// make sure that the agent making the REQ_NAME request is
	// the same as the agent that made the REQ_BEGIN request

    if (!(security.isAgentAuthenticated()) || (id != security.getAgentId())) {
	return (makeErrorResponse (ServerInterface::e_SERV_ERR_LIAR));
    }

	// make sure that the name is nonempty and does not belong to the server

    if ((name == NULL) || (*name == '\0') || !strcmp(name, "server")) {
	return (makeErrorResponse (ServerInterface::e_SERV_ERR_NAME_RESERVED));
    }

	// find the agent in the tables

    if ((id_agent = lookupRecipient (id, (char *) NULL)) == NULL) {
	return (makeErrorResponse (ServerInterface::e_SERV_ERR_LOST));
    }

	// associate the name with the agent if the name is not in use

    if ((name_agent = lookupRecipient (0, (char *) name)) == NULL) {

	int dummy;                
        HashNode *eventNode;
	HashNode *reservedNode = NULL;

	    // first make sure that the new name is not reserved for some other owner

	if ((serverData -> encryption) && (reservedNode = serverData -> reservedTable.lookup (0, name)) != NULL) {

		// error if the agent is an anonymous agent

	    if (!(id_agent -> security.isOwnerAuthenticated())) {
		return (makeErrorResponse (ServerInterface::e_SERV_ERR_NAME_RESERVED));
	    }

		// error if the agent's owner is not the owner for whom the
		// symbolic name is reserved

	    DynamicString agentOwner = id_agent -> security.getOwnerKeyname();
	    char *nameOwner = (char *) reservedNode -> data;

	    if (agentOwner != nameOwner) {
		return (makeErrorResponse (ServerInterface::e_SERV_ERR_NAME_RESERVED));
	    }
	}

	    // then remove an old name if necessary

        DynamicString oldAgentName = id_agent -> id.getName();

	if (oldAgentName.length() != 0) {

	    name_table.remove (0, oldAgentName.value());

	    if ((eventNode = nameEventsTable.lookup (0, oldAgentName.value())) != NULL) {
		delete ((PendingEvents *) eventNode -> data);
		nameEventsTable.remove (0, oldAgentName.value());
	    }
	}

	    // then set up the new name 

        DynamicString newAgentName (name);
	id_agent -> id.setName (newAgentName);
	name_table.add (0, name, (void *) id_agent, dummy);

	    // then see if there are any events that should be fired 

	if ((eventNode = nameEventsTable.lookup (0, name)) != NULL) {

	    char *pointer;
	    PendingEvents *events = (PendingEvents *) eventNode -> data;
	    PointerIterator iterator (&events -> birthNotifications);

	    while (iterator.nextPointer(pointer) > 0) {
		ServerAgent *receiver = (ServerAgent *) pointer;
		receiver -> addExistsEvent (*id_agent);
		receiver -> numPendingEvents -= 1;
		receiver -> pendingBirthEvents.remove ((char *) eventNode);
	    }

#ifdef FIX_LATER
		/* Once events can hold more than one event type, */
		/* we need to check that events is actually empty */
		/* before we do the deletion below.               */
#endif
 
	    delete (events);
	    nameEventsTable.remove (0, name);
	}

	    // success -- name is now associated with the agent 

	response = new MESSAGE (messagesFromServer.messages[RESP_OK]);

    } else if (name_agent == id_agent) {

	    // success -- name is already associated with the agent

	response = new MESSAGE (messagesFromServer.messages[RESP_OK]);

    } else {

	    // error -- name is associated with some other agent

	response = makeErrorResponse (ServerInterface::e_SERV_ERR_NAME_USED);
    }

    return (response);
}

/* AgentSet::agent_end

   Purpose: Remove an agent from the table

     Input: message  = request message
		       (class MESSAGE *) 

	    [0] = name of the agent
            [1] = numeric id of the agent

	    security = security information
		       (class BASE_SECURITY &)

    Output: The procedure returns the message that should be sent to 
            socketd.
*/

MESSAGE *AgentSet::agent_end (MESSAGE *message, BASE_SECURITY &security)
{ 
    ServerAgent *agent;
    MESSAGE *response;
    HashNode *node;

	/* break out the pieces of the message */

    SmartCharPtr name (message -> getString (0));
    UINT_32 id        = message -> getLong (1);

	/* make sure that the agent saying REQ_END is the agent that said REQ_BEGIN */

    if (!(security.isAgentAuthenticated()) || (id != security.getAgentId())) {
	return (makeErrorResponse (ServerInterface::e_SERV_ERR_LIAR));
    }

	/* find the agent in the tables */

    if ((agent = lookupRecipient (id, name)) == NULL) {
	return (makeErrorResponse (ServerInterface::e_SERV_ERR_LOST));
    }

	/* remove the agent from the tables */

    DynamicString idAgentName = agent -> id.getName();

    agent_table.remove (agent -> id.getId(), NULL); 

    if (idAgentName.length() != 0) {

	name_table.remove (0, idAgentName.value());

	if ((node = nameEventsTable.lookup (0, idAgentName.value())) != NULL) {
	    delete ((PendingEvents *) node -> data);
	    nameEventsTable.remove (0, idAgentName.value());
	}
    }

	/* construct the response message */

    DynamicString idHostName = serverData -> host.getName(); 

    AgentId *localId = new AgentId 
	(idHostName, serverData -> host.getIp(), 
	    idAgentName, agent -> id.getId());
	
    response = new MESSAGE (messagesFromServer.messages[RESP_ID]);
    response -> addId (0, localId, e_TRUE);

	/* finish removing the agent */

    if (!(agent -> security.isOwnerAuthenticated())) {
	numAnonAgents -= 1;
    }

    numAgents -= 1;
    delete (agent);

	/* return the response message */

    return (response);  
}

/* AgentSet::agent_force

   Purpose: Remove an agent from the table

     Input: message  = request message
		       (class MESSAGE *) 

	    [0] = name of the agent
            [1] = numeric id of the agent

	    security = security information
		       (class BASE_SECURITY &)

    Output: The procedure returns the message that should be sent to 
            socketd.
*/

MESSAGE *AgentSet::agent_force (MESSAGE *message, BASE_SECURITY &security)
{ 
    HashNode *node;
    MESSAGE *response;
    ServerAgent *agent;
    BASE_SECURITY *actualSecurity;

	/* break out the pieces of the message */

    SmartCharPtr name (message -> getString (0));
    UINT_32 id        = message -> getLong (1);

	/* find the target agent in the tables */

    if ((agent = lookupRecipient (id, name)) == NULL) {
	return (makeErrorResponse (ServerInterface::e_SERV_ERR_NONEXISTENT));
    }

	/* lookup the complete security information for the forcing agent */

    if ((actualSecurity = lookupCompleteSecurity (security)) == NULL) {
	return (makeErrorResponse (ServerInterface::e_SERV_ERR_SYSTEM));
    }

	/* see if the forcing agent has sufficient authority to force out */
	/* the target agent                                               */

    BOOLEAN allowForce = e_FALSE;

    if (!(serverData -> encryption)) {

	    // allow all forces if we are not using an encryption subsystem --
	    // i.e., allow all forces if we have no signatures on which to 
	    // base a decision

	allowForce = e_TRUE;	

    } else {

	    // if we are using an encryption subsystem, allow a force only
	    // if the source agent is verifiably owned by the machine's owner, 
	    // the server's administrator or the target agent's owner 

	if (actualSecurity -> isOwnerAuthenticated()) {

	    if (actualSecurity -> isOwnerKeyname (serverData -> adminKeyname.value())) {

		allowForce = e_TRUE;

	    } else if (actualSecurity -> isOwnerKeyname (serverData -> ownerKeyname.value())) {

		if (!(agent -> security.isOwnerKeyname (serverData -> adminKeyname.value()))) {
		    allowForce = e_TRUE;
		}

	    } else if (actualSecurity -> isOwnerKeynameSame (agent -> security)) {

		allowForce = e_TRUE;
	    }
	}
    }

    if (!allowForce) {
	return (makeErrorResponse (ServerInterface::e_SERV_ERR_INVALID_FORCE));
    }

	/* kill the agent processes -- do not kill the handler */

#ifdef FIX_LATER
	/* we need to somehow stop the handler if the agent is on a different */
	/* machine							      */
#endif

    if ((agent -> pid != 0) && (agent -> id.getServerIp() == serverData -> host.getIp())) {
	kill (agent -> pid, SIGTERM);
    }

	/* remove the agent from the tables */

    agent_table.remove (agent -> id.getId(), NULL); 

    if (agent -> id.getName().length() != 0) {

	name_table.remove (0, agent -> id.getName().value());

	if ((node = nameEventsTable.lookup (0, agent -> id.getName().value())) != NULL) {
	    delete ((PendingEvents *) node -> data);
	    nameEventsTable.remove (0, agent -> id.getName().value());
	}
    }

	/* construct the response message */

    AgentId *localId = new AgentId 
	(serverData -> host.getName(), serverData -> host.getIp(), 
	    agent -> id.getName(), agent -> id.getId());
	
    response = new MESSAGE (messagesFromServer.messages[RESP_ID]);
    response -> addId (0, localId, e_TRUE);

	/* finish removing the agent */

    if (!(agent -> security.isOwnerAuthenticated())) {
	numAnonAgents -= 1;
    }

    numAgents -= 1;
    delete (agent);

	/* return the response message */

    return (response);  
}

/* AgentSet::agent_event

   Purpose: Add an event to an agent's queue

     Input: message  = request message
		       (class MESSAGE *) 

	    [0] = source agent id        (AgentId ) 
            [1] = destination name       (string)
            [2] = destination numeric id (long)
            [3] = event tag              (string)
            [4] = event string           (string)

	    security = security information
		       (class BASE_SECURITY &)

    Output: The procedure returns the message that should be sent to socketd.
*/

MESSAGE *AgentSet::agent_event (MESSAGE *message, BASE_SECURITY &security)
{
    ServerAgent *agent;
    MESSAGE *response;
    INCOMING_EVENT *inc_event;
    BASE_SECURITY *actualSecurity;
    AgentId *actualSource;

	/* break out the pieces of the message */

    SmartIdPtr source (message -> getId (0));    
    SmartCharPtr name (message -> getString (1));
    UINT_32 id        = message -> getLong (2);
    char *tag         = message -> getString (3);     // don't delete
    char *string      = message -> getString (4);     // don't delete

	/* lookup the destination agent */

    if ((agent = lookupRecipient (id, name)) == NULL) {
	return (makeErrorResponse (ServerInterface::e_SERV_ERR_NONEXISTENT));
    }

	/* lookup the actual id of the sending agent */

    if ((actualSource = lookupActualId (source, security)) == NULL) {
	return (makeErrorResponse (ServerInterface::e_SERV_ERR_SYSTEM)); 
    }

	/* get complete security information */

    if ((actualSecurity = lookupCompleteSecurity (security)) == NULL) {
	return (makeErrorResponse (ServerInterface::e_SERV_ERR_SYSTEM));
    }

	/* if the message was sent from a remote agent, make sure that */
	/* the agent is not lying about its id                         */

    if (!(actualSecurity -> isAgentAuthenticated())) {

	if ((actualSource -> getServerIp() == serverData -> host.getIp()) ||
	    (actualSource -> getServerName() == serverData -> host.getName()))  
	{
	    return (makeErrorResponse (ServerInterface::e_SERV_ERR_LIAR));
	}
    }

	/* construct the INCOMING_EVENT structure (make a copy of the */
	/* security information and the source id since the original  */
	/* copies will go away)                                       */

    inc_event = new INCOMING_EVENT (new AgentId(*actualSource), tag, string, new BASE_SECURITY (*actualSecurity));

	/* add the message to the queue and unblock if necessary */ 

    agent -> events.enqueue ((void *) inc_event);
    agent -> items += 1;

    if (agent -> handlerBlocked) {

	if (kill (agent -> handlerPid, SIGUSR1) < 0) {

	    char temp[16];
	    sprintf (temp, "%d", agent -> handlerPid);

	    char *argv[] = {
		"agentd: unable to send signal to process with pid ", 
		temp
	    };

	    serverData -> errorLog.error_sys_cont (argv);
	}

	agent -> handlerBlocked = e_FALSE;
    }

    response = new MESSAGE (messagesFromServer.messages[RESP_OK]);
    return response;
}
 
/* AgentSet::agent_error

   Purpose: Add an error message generated by the other half of the server
            to an agent's message queue 

     Input: message  = request message
		       (class MESSAGE *)

	    [0] = source id              (AgentId)
            [1] = destination name       (string)
            [2] = destination numeric id (long)
            [3] = message code           (long)
            [4] = message string         (string)

	    security = security information
		       (class BASE_SECURITY &)

    Output: The procedure returns the message that should be sent to socketd.
*/
       
MESSAGE *AgentSet::agent_error (MESSAGE *message, BASE_SECURITY &security)
{
    ServerAgent *agent;
    MESSAGE *response;
    INCOMING_MESSAGE *inc_message;

	/* break out the pieces of the message */

    SmartIdPtr source (message -> getId (0));
    SmartCharPtr name (message -> getString (1));
    UINT_32 id        = message -> getLong (2); 
    UINT_32 code      = message -> getLong (3);
    char *string      = message -> getString (4);       // don't delete

	/* lookup the destination agent */

    if ((agent = lookupRecipient (id, name)) == NULL) {
	return (makeErrorResponse (ServerInterface::e_SERV_ERR_NONEXISTENT));
    }

	/* construct the INCOMING_MESSAGE structure (make a copy of the */
	/* security information and the id since the original copy will */
	/* go away)   							*/

    inc_message = new INCOMING_MESSAGE (new AgentId (*source), code, string, new BASE_SECURITY (security));

	/* add the message to the queue and unblock the background handler */

    agent -> messages.enqueue ((void *) inc_message);
    agent -> items += 1;

    if (agent -> handlerBlocked) {

	if (kill (agent -> handlerPid, SIGUSR1) < 0) {

	    char temp[16];
	    sprintf (temp, "%d", agent -> handlerPid);

	    char *argv[] = {
		"agentd: unable to send signal to process with pid ", 
		temp
	    };

	    serverData -> errorLog.error_sys_cont (argv);
	}

	agent -> handlerBlocked = e_FALSE;
    }

	/* done */

    response = new MESSAGE (messagesFromServer.messages[RESP_OK]);
    return response;
}

/* AgentSet::agent_send

   Purpose: Add a message to an agent's queue

     Input: message  = request message
		       (class MESSAGE *)

	    [0] = source id              (AgentId)
            [1] = destination name       (string)
            [2] = destination numeric id (long)
            [3] = message code           (long)
            [4] = message string         (string)

	    security = security information
		       (class BASE_SECURITY &)

    Output: The procedure returns the message that should be sent to socketd.
*/
       
MESSAGE *AgentSet::agent_send (MESSAGE *message, BASE_SECURITY &security)
{
    ServerAgent *agent;
    MESSAGE *response;
    INCOMING_MESSAGE *inc_message;
    BASE_SECURITY *actualSecurity;
    AgentId *actualSource;

	/* break out the pieces of the message */

    SmartIdPtr source (message -> getId (0));
    SmartCharPtr name (message -> getString (1));
    UINT_32 id        = message -> getLong (2); 
    UINT_32 code      = message -> getLong (3);
    char *string      = message -> getString (4);       // don't delete

	/* lookup the destination agent */

    if ((agent = lookupRecipient (id, name)) == NULL) {
	return (makeErrorResponse (ServerInterface::e_SERV_ERR_NONEXISTENT));
    }

	/* lookup the actual id of the sending agent */

    if ((actualSource = lookupActualId (source, security)) == NULL) {
	return (makeErrorResponse (ServerInterface::e_SERV_ERR_SYSTEM)); 
    }

	/* get complete security information */

    if ((actualSecurity = lookupCompleteSecurity (security)) == NULL) {
	return (makeErrorResponse (ServerInterface::e_SERV_ERR_SYSTEM));
    }

	/* if the message was sent from a remote agent, make sure that */
	/* the agent is not lying about its id                         */

    if (!(actualSecurity -> isAgentAuthenticated())) {

	DynamicString sourceServerName = actualSource -> getServerName();
	DynamicString ourServerName    = serverData -> host.getName();

	if ((actualSource -> getServerIp() == serverData -> host.getIp()) ||
	    (sourceServerName == ourServerName))  
	{
	    return (makeErrorResponse (ServerInterface::e_SERV_ERR_LIAR));
	}
    }

	/* construct the INCOMING_MESSAGE structure (make a copy of the */
	/* security information and the id since the original copy will */
	/* go away)   							*/

    inc_message = new INCOMING_MESSAGE (new AgentId (*actualSource), 
	code, string, new BASE_SECURITY (*actualSecurity));

	/* add the message to the queue and unblock the background handler */

    agent -> messages.enqueue ((void *) inc_message);
    agent -> items += 1;

    if (agent -> handlerBlocked) {

	if (kill (agent -> handlerPid, SIGUSR1) < 0) {

	    char temp[16];
	    sprintf (temp, "%d", agent -> handlerPid);

	    char *argv[] = {
		"agentd: unable to send signal to process with pid ", 
		temp
	    };

	    serverData -> errorLog.error_sys_cont (argv);
	}

	agent -> handlerBlocked = e_FALSE;
    }

	/* done */

    response = new MESSAGE (messagesFromServer.messages[RESP_OK]);
    return response;
}

/* AgentSet::agent_receive

   Purpose: See if there is a result pending for an agent
   
     Input: message  = request message
		       (class MESSAGE *) 

	    [0] = pointer to the agent id      (long)
            [1] = pointer to the process id    (long)

	    security = security information
		       (class BASE_SECURITY &)

    Output: The procedure returns the message that should be sent to socketd.
*/

MESSAGE *AgentSet::agent_receive (MESSAGE *message, BASE_SECURITY & /*UNUSED*/)
{
    ServerAgent *agent;
    MESSAGE *response;
    HashNode *hash_node;
    INCOMING_MESSAGE *inc_message;
    INCOMING_MEETING *inc_meeting;
    INCOMING_EVENT   *inc_event;

#ifdef SECURITY_COMMENT

	/* No security checks are performed here since an "agent_receive"   */
	/* request can only be generated from the other half of the server. */

#endif

	/* break out the pieces of the message */

    UINT_32 id  = message -> getLong (0); 
    UINT_32 pid = message -> getLong (1); 

	/* error if there is no agent with the specified id */

    if ((hash_node = agent_table.lookup (id, NULL)) == NULL) {
	return (makeErrorResponse (ServerInterface::e_SERV_ERR_LOST));
    }

	/* get the first available event, meeting or message for the agent */

    agent = (ServerAgent *) hash_node -> data;

	/* nothing to do if we have no available items */

     if (agent -> items == 0) {

	    /* will send SIGUSR1 when a message, meeting or event arrives */

	response = new MESSAGE (messagesFromServer.messages[RESP_NONE]);
	agent -> handlerBlocked = e_TRUE;
	agent -> handlerPid  = pid;
	return (response);
    }

	/* otherwise return the meeting, event or message */

    if ((inc_event = (INCOMING_EVENT *) agent -> events.dequeue ()) != NULL) {

	agent -> items -= 1;

	    /* assemble the response */

	response = new MESSAGE (messagesFromServer.messages[RESP_EVENT]);
	response -> addLong (0, agent -> items);
	response -> addId (1, inc_event -> id, e_TRUE); 
	response -> addSecurity (2, inc_event -> security, e_TRUE);
	response -> addString (3, inc_event -> tag, e_TRUE);
	response -> addString (4, inc_event -> string, e_TRUE);

	    /* cleanup the INCOMING_EVENT structure (don't delete what we just transferred out) */

	inc_event -> id       = NULL;
	inc_event -> tag      = NULL;
	inc_event -> string   = NULL;
	inc_event -> security = NULL;
	delete (inc_event);

    } else if ((inc_meeting = (INCOMING_MEETING *) agent -> meetings.dequeue ()) != NULL) {

	agent -> items -= 1;

	    /* assemble the response */

	response = new MESSAGE (messagesFromServer.messages[RESP_MEETING]);
	response -> addLong (0, agent -> items);
	response -> addId (1, inc_meeting -> id, e_TRUE); 
	response -> addSecurity (2, inc_meeting -> security, e_TRUE);
	response -> addByte (3, inc_meeting -> status);
	response -> addLong (4, inc_meeting -> localHandle);
	response -> addLong (5, inc_meeting -> remoteHandle);
	response -> addString (6,  strcpyWithAlloc (inc_meeting -> port -> machine), e_TRUE);
	response -> addIp (7, inc_meeting -> port -> ip);
	response -> addLong (8, inc_meeting -> port -> port);

	    /* cleanup the INCOMING_MEETING structure (don't delete what we just transferred out) */

	inc_meeting -> id       = NULL;
	inc_meeting -> security = NULL;
	delete (inc_meeting);

    } else if ((inc_message = (INCOMING_MESSAGE *) agent -> messages.dequeue ()) != NULL) {

	agent -> items -= 1;

	    /* assemble the response */

	response = new MESSAGE (messagesFromServer.messages[RESP_MESSAGE]);
	response -> addLong (0, agent -> items);
	response -> addId (1, inc_message -> id, e_TRUE);
	response -> addSecurity (2, inc_message -> security, e_TRUE);
	response -> addLong (3, inc_message -> code);
	response -> addString (4, inc_message -> string, e_TRUE);

	    /* cleanup the INCOMING_MESSAGE structure (don't delete what we just transferred out) */
    
	inc_message -> id       = NULL;
	inc_message -> string   = NULL;
	inc_message -> security = NULL;
	delete (inc_message);

    } else {

	response = makeErrorResponse (ServerInterface::e_SERV_ERR_SYSTEM);
    }

	/* if we now have no items, make sure we send a SIGUSR1 on the next one */

    if (agent -> items == 0) {
	agent -> handlerBlocked = e_TRUE;
	agent -> handlerPid  = pid;
    }

    return (response);
}

/* AgentSet::agent_status

   Purpose: Check the status of an agent

     Input: message = request message
		      (class MESSAGE *)

	    	      [0] = STATUS_EXISTS 
		      [1] = STATUS_EVENT or STATUS_NO_EVENT
		      [2] = agent name
		      [3] = agent id
		      [4] = requestor agent identification

	    security = security information
		       (class BASE_SECURITY &)
 
    Output: The procedure constructs and returns the message that should be
	    sent down the socket.
*/

MESSAGE *AgentSet::agent_status (MESSAGE *message, BASE_SECURITY &security)
{
    ServerAgent *agent;
    ServerAgent *reqAgent;
 
	/* break out the pieces of the message */

    UINT_8 request       = message -> getByte (0);
    UINT_8 event         = message -> getByte (1);
    SmartCharPtr name    (message -> getString (2));
    UINT_32 id		 = message -> getLong (3); 
    UINT_32 requestorId  = message -> getLong (4);

	/* make sure that the agent saying REQ_STATUS is the agent that said REQ_BEGIN */

    if (!(security.isAgentAuthenticated()) || (requestorId != security.getAgentId())) {
	return (makeErrorResponse (ServerInterface::e_SERV_ERR_LIAR));
    }

	/* find the requesting agent in the tables */

    if ((reqAgent = lookupRecipient (requestorId, (char *) NULL)) == NULL) {
	return (makeErrorResponse (ServerInterface::e_SERV_ERR_LOST));
    }

	/* check the status of the agent of interest */

    if (request == STATUS_EXISTS) {

	    /* look up the agent of interest */

	agent = lookupRecipient (id, name);

	    /* done if we do not want an event */

	if (event == STATUS_NO_EVENT) {

	    if (agent == NULL) {
		return (makeErrorResponse (ServerInterface::e_SERV_ERR_NONEXISTENT));
	    }

	    return (new MESSAGE (messagesFromServer.messages[RESP_OK]));
	}

	    /* we want an event -- first see if we can actually provide an  */
	    /* event -- we can not provide an event if there is no symbolic */
	    /* name                                                         */ 

	if (((char *) name == NULL) || (*name == '\0') || (!strcmp (name, "server"))) {
	    return (makeErrorResponse (ServerInterface::e_SERV_ERR_NO_EVENT));
	}

	    /* provide the event immediately if the agent exists */
	    /* otherwise set up to provide the event later       */

	if (agent != NULL) {

	    reqAgent -> addExistsEvent (*agent);

	} else {

	    int dummy;
	    HashNode *node;
	    PendingEvents *events;

		/* see if the agent can request another event */

	    if (reqAgent -> numPendingEvents >= serverData -> maxStatus) {
		return (makeErrorResponse (ServerInterface::e_SERV_ERR_TOO_MANY_EVENTS));
	    }

		/* remember the new agent */ 

	    if ((node = nameEventsTable.lookup (0, name)) != NULL) {
		events = (PendingEvents *) node -> data;
	    } else {
	        events = new PendingEvents ();
		node = nameEventsTable.add (0, name, (void *) events, dummy);
	    } 

	    events -> birthNotifications.insert ((char *) reqAgent);

		/* and add a back pointer from the agent */

	    reqAgent -> numPendingEvents += 1;
	    reqAgent -> pendingBirthEvents.insert ((char *) node);
        }

	    /* success */

	return (new MESSAGE (messagesFromServer.messages[RESP_OK]));
    }

	/* unknown status type so we have an error */

    return (makeErrorResponse (ServerInterface::e_SERV_ERR_INVALID));
}

/* AgentSet::agent_ip

   Purpose: Return the IP address of the machine on which the server is
	    running.  This function is needed since the only functions 
	    available are "gethostname" which returns the symbolic name of
	    the current host and "gethostbyname" which can convert that
	    symbolic name into an IP address.  The problem is that if the
	    network connection has been pulled out of the wall, which is a
	    very real possible since we are specifically targetting mobile
	    machines, "gethostbyname" will block for several minutes as it 
	    tries to contact a remote name server.  This is an unacceptable 
            delay when an agent is starting up.  Therefore an agent contacts 
	    the server using the localhost IP address 127.0.0.1 in order to 
	    get the real IP address of its machine.

     Input: sender   = security description of the sender
                       (class BASE_SECURITY &)

	    message  = the message
                       (class MESSAGE *)

    Output: The procedure returns the message that should be sent back to
	    the agent. 
*/

MESSAGE *AgentSet::agent_ip (MESSAGE * /*UNUSED*/, BASE_SECURITY & /*UNUSED*/)
{
#ifdef SECURITY_COMMENT

	/* No security checks are performed here since there is no harm */
	/* in returning the server's actual IP address to an anonymous  */
	/* entity (particularly since any remote entity must already    */
	/* know the IP address to have contacted the server in the      */
	/* first place).                                                */

#endif

	/* construct the response that contains the IP address */

    MESSAGE *response = new MESSAGE  (messagesFromServer.messages[RESP_IP]);
    response -> addString (0, serverData -> host.getName().value());
    response -> addIp (1, serverData -> host.getIp());
    return (response); 
}
