/* Agent Tcl
   Bob Gray
   9 February 1995

   agentd.cc

   This is the top-level program for the agent server.  On startup it forks
   the process "socketd" which monitors the agent port.  Then its task is to
   maintain the data structures that keep track of agents executing on its
   machine.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "platInterrupt.h"
#include "platSystem.h"
#include "platTimeval.h"
#include "suppStrings.h"
#include "mesgTcpip.h"
#include "servConf.h"		// ServerData
#include "servRandom.h"		// RandomFile
#include "agent_set.h"
#include "agentd.h"
#include "agentId.h"
#include "genDaemon.h"
#include "genEncrypt.h"
#include "genError.h"
#include "genFile.h"
#include "genPipe.h"
#include "suppHash.h"
#include "genManager.h"
#include "interp.h"
#include "parse.h"
#include "piped.h"
#include "noise.h"
#include "random.h"
#include "randpool.h"
#include "socketd.h"
#include "truefalse.h"

    /* external signal handlers */

extern void sigio_handler (int, SignalHandlers::ClientData);
extern void sighup_handler (int, SignalHandlers::ClientData);
extern void sigprof_handler (int, SignalHandlers::ClientData);
extern void sigchld_handler (int, SignalHandlers::ClientData);
extern void coredump_handler (int, SignalHandlers::ClientData);

    /* defaults are to (1) run the server as a daemonm, (2) verify the */
    /* hostname that was specified after the -host command-line flag,  */
    /* and (3) not truncate the log                                    */

const BOOLEAN DEFAULT_DAEMON	      = e_TRUE;
const BOOLEAN DEFAULT_VERIFY_HOSTNAME = e_TRUE;
const BOOLEAN DEFAULT_TRUNCATE_LOG    = e_FALSE;

    /* size of the pass phrase buffer */

const int PASS_BUFFER_SIZE 	= 2048;

    /* delay when getting an IP address */

const struct timeval IP_DELAY 	= {20, 0};

/* verify_server_ip

   Purpose: Verify that the machine specified with the -host parameter
            is the machine on which the server is running

     Input: verifyHostname = e_TRUE if we are actually going to verify the
                             hostname; e_FALSE if we are going to blindly
                             accept the hostname that was specified after the
                             -host command-line option.

    Output: The procedure returns -1 on error or if the specified machine
            is not the machine on which the server is running.  Otherwise
            the procedure returns 0.
*/

int verify_server_ip (BOOLEAN verifyHostname)
{
    char *hostname;
    struct timeval stop;

	/* get the IP number of machine specified with the -host parameter */
 
    stop = TimevalUtil::addTimevals (SystemUtil::getCurrentWall(), IP_DELAY);

    if (tcpip_getIP (serverData -> host, stop) != e_TCPIP_OK) {

	char *argv[] = {
	    "agentd: ERROR: unable to get IP address of local host \"", 
	    serverData -> host.getName().value(),
	    "\"",
	    ((char *) NULL)
	};

	serverData -> errorLog.error_app_cont (argv);
	return -1;
    }

    if (!verifyHostname) {
	return 0;
    }

	/* get the name of the the actual machine */

    if ((hostname = tcpip_getActualHostname ()) == NULL) {

	char *argv[] = {
	    "agentd: ERROR: unable to get actual name of local host",
	    ((char *) NULL)
	};

	serverData -> errorLog.error_sys_cont (argv);
	return -1;
    }
  
	/* get the IP address of the actual machine */

    MachineId actualMachine (hostname, UNKNOWN_IP);
    stop = TimevalUtil::addTimevals (SystemUtil::getCurrentWall(), IP_DELAY);

    if (tcpip_getIP (actualMachine, stop) != e_TCPIP_OK) {

	char *argv[] = {
	    "agentd: ERROR: unable to get IP address of local host \"", 
	    hostname,
	    "\"",
	    ((char *) NULL)
	};

	serverData -> errorLog.error_app_cont (argv);
	return -1;
    }

	/* compare the IP addresses */

    if (actualMachine.getIp() != serverData -> host.getIp()) {

	char *argv[] = {
	    "machine after \"-host\" (",
	    serverData -> host.getName().value(),
	    ") is *not* the machine on which the server is actually running (",
	    hostname,
	    ")",
	    ((char *) NULL)
	};

	serverData -> errorLog.error_app_cont (argv);
	return -1;
    }

    return 0;
}

/* getPassphrase

   Purpose: Get the passphrase from the keyboard if it was not specified in 
            the configuration file.

     Input: None

    Output: The procedure returns -1 if it is unable to get the pass phrase.
	    Otherwise the procedure returns 0 and updates the server
            configuration with the passphrase.
*/

int getPassphrase (void)
{
    int newline;
    char phrase[PASS_BUFFER_SIZE + 1];

	/* get the pass phrase */

    if (serverData -> passPhrase.isEmpty()) {

	fprintf (stdout, "\nObtaining pass phrase ...\n\n");

	while (1) {

		/* wait for the user to enter the pass phrase */

	    fprintf (stdout, "? ");
	    fgets (phrase, PASS_BUFFER_SIZE, stdin);

		/* remove the trailing newline */

	    newline = strlen(phrase) - 1;
	    phrase[newline] = '\0';

		/* break if we have a nonempty pass phrase */

	    if (phrase[0] != '\0') {
		break;
	    }
	}

	    /* remember the passphrase */

	serverData -> passPhrase.append (phrase);
    }

	/* done */

    return (0);
}

/* construct_server_keyname

   Purpose: Construct the keyname for the server

     Input: None

    Output: The procedure returns -1 if it is unable to construct the
	    keyname.  Otherwise the procedure returns 0 and updates
	    the server information with the appropriate keyname.
*/

int construct_server_keyname (void)
{
    char *keyname;

	/* construct the keyname */

    if ((keyname = makeFullServerKeyname (serverData -> host)) == NULL) {

	char *argv[] = {
	    "agentd: ERROR: unable to construct server keyname",
	    ((char *) NULL)
	};

	serverData -> errorLog.error_app_cont (argv);
	return -1; 
    } 

    char *argv[] = {
	"agentd: INFO: server keyname is \"",
	keyname,
	"\"",
	((char *) NULL)
    };

    serverData -> errorLog.error_app_cont (argv);

	/* record */

    SEC_IDENTITY id (keyname, serverData -> passPhrase.value(), e_TRUE, e_FALSE);
    serverData -> security.setOwner (id);
    serverData -> security.setMachine (id);
    serverData -> security.setOwnerSigned (e_TRUE);

	/* cleanup */

    delete (keyname);
    return 0;
}

/* printUsage

   Purpose: Print usage instructions to standard output

     Input: name = name that was used to start the program

    Output: The procedure outputs usage instructions to standard output.
*/

void printUsage (char *name)
{
    int i;
    int len      = strlen(name);
    char *spaces = new char [len + 1];

	/* initialize the spacing string */

    for (i = 0; i < len; i++) {
	spaces[i] = ' ';
    }

    spaces[i] = '\0';

	/* print usage instructions */

    printf ("\n");
    printf ("USAGE:\n\n");
    printf ("%s -host <hostname>\n", name);
    printf ("%s -conf <filename>\n", spaces);
    printf ("%s [-nodaemon]\n", spaces);
    printf ("%s [-noverify]\n", spaces);
    printf ("%s [-trunc]\n", spaces);

	/* cleanup */

    delete (spaces);
}

/* getCommandLine

   Purpose: Get the command line parameters

     Input: argc = number of command line parameters
		   (int)

            argv = the command line arguments
		   (char **)

    Output: The procedure returns -1 if the command line parameters are
            specified incorrectly.  Otherwise the procedure fills in the
	    CommandLine structure and returns 0.
*/

static int getCommandLine (int argc, char **argv, CommandLine &commandLine)
{
    int confFound    = FALSE;     // configuration file specified 
    int hostFound    = FALSE;     // host specified
    unsigned numArgs = argc - 1;  // number of arguments       
    unsigned curArg  = 1;         // current argument         

	/* set the command line defaults */

    commandLine.daemon         = DEFAULT_DAEMON;
    commandLine.verifyHostname = DEFAULT_VERIFY_HOSTNAME;
    commandLine.truncateLog    = DEFAULT_TRUNCATE_LOG;

	/* get the arguments from the command line */

    while (curArg <= numArgs) {

	if (!strcmp(argv[curArg], "-host")) {

	    if (++curArg > numArgs) {
		return (-1);
	    }  

	    DynamicString name (argv[curArg]);
	    name.convertToLowercase ();
	    serverData -> host.setName (name);
	    hostFound = TRUE;

	} else if (!strcmp(argv[curArg], "-conf")) {

	    if (++curArg > numArgs) {
		return (-1); 
	    }

	    commandLine.configurationFile = strcpyWithAlloc (argv[curArg]);
	    confFound = TRUE;

	} else if (!strcmp(argv[curArg], "-nodaemon")) {
	    commandLine.daemon = e_FALSE;
	} else if (!strcmp(argv[curArg], "-daemon")) {
	    commandLine.daemon = e_TRUE;
	} else if (!strcmp(argv[curArg], "-noverify")) {
	    commandLine.verifyHostname = e_FALSE;
	} else if (!strcmp(argv[curArg], "-verify")) {
	    commandLine.verifyHostname = e_TRUE;
	} else if (!strcmp(argv[curArg], "-notrunc")) {
	    commandLine.truncateLog = e_FALSE;
	} else if (!strcmp(argv[curArg], "-trunc")) {
	    commandLine.truncateLog = e_TRUE;
	} else {
	    return (-1); 
	}

	curArg++;
    }

    return (confFound && hostFound ? 0 : -1);
}

/* switchUser

   Purpose: Reset user priveleges

     Input: None

    Output: The procedure changes the user priveleges -- i.e. the procedure
	    changes the user id of the process.
   
     NOTES: This procedure was taken from the Apache Web server.
*/

static uid_t getUserId (char *name) 
{
    struct passwd *ent;

    if(name[0] == '#') {

        return(atoi(&name[1]));
    }

    if(!(ent = getpwnam(name))) {

	char *argv[] = {
	    "ERROR: agentd: bad user name \"",
	    name,
	    "\"",
	    ((char *) NULL)
	};

	serverData -> errorLog.error_sys_quit (argv);
	abort_with_message ("should never get here");   // should never get here
    } 

    return (ent->pw_uid);
}

static void switchUser (void)
{
    if(!geteuid()) {    // only switch if running as root

	uid_t uid = getUserId (serverData -> userName.value());

	if (setuid(uid) < 0) {

	    char *argv[] = {
		"ERROR: agentd: unable to change uid",
		((char *) NULL)
	    };

	    serverData -> errorLog.error_sys_quit (argv);
	    abort_with_message ("should never get here");   // should never get here
	}

	char *argv[] = {
	    "INFO: agentd: user has been switched to \"",
	    serverData -> userName.value(),
	    "\"",
	    ((char *) NULL)
	};

	serverData -> errorLog.error_app_cont (argv);
    }
}

/* switchGroup

   Purpose: Reset group priveleges

     Input: None

    Output: The procedure resets the group priveleges -- i.e. the procedure
	    changes the group id of the process.

     NOTES: We use the username as set in the config files rather than
	    a lookup on uid -- the same uid may have multiple passwd entries 
	    with different sets of groups for each.

	    This code is taken from the Apache web server.
*/
 
static gid_t getGroupId (char *name) 
{
    struct group *ent;

    if(name[0] == '#') {
        return(atoi(&name[1]));
    }

    if(!(ent = getgrnam(name))) {

	char *argv[] = {
	    "ERROR: agentd: bad group name \"",
	    name,
	    "\"",
	    ((char *) NULL)
	};

	serverData -> errorLog.error_sys_quit (argv);
	abort_with_message ("should never get here");   // should never get here
    }
 
    return (ent -> gr_gid);
}

static void switchGroup (void)
{
    if(!geteuid()) {    // can only do this if we are root

	char *name;
 	char *userName = serverData -> userName.value();
	char *groupName = serverData -> groupName.value();

	    /* get the group id */

	gid_t group_id = getGroupId (groupName);

	    /* get username if passed as a uid */
    
	if (userName[0] == '#') {

	    struct passwd* ent;
	    uid_t uid = atoi(&userName[1]);

	    if ((ent = getpwuid(uid)) == NULL) {

		char *argv[] = {
		    "ERROR: agentd: couldn't determine username from uid\"",
		    userName,
		    "\"",
	    	    ((char *) NULL)
		};

		serverData -> errorLog.error_sys_quit (argv);
		abort_with_message ("should never get here");   // should never get here
	    }

	    name = ent -> pw_name;

	} else {

	    name = userName; 
	}

#ifndef __EMX__ 

	    /* OS/2 dosen't support groups. */
	    /* Reset `groups' attributes. */
    
	if (initgroups(name, group_id) == -1) {

	    char *argv[] = {
		"ERROR: agentd: unable to set groups",
	    	 ((char *) NULL)
	    };
	 
	    serverData -> errorLog.error_sys_quit (argv); 
	    abort_with_message ("should never get here");   // should never get here
	}

	if (setgid(group_id) == -1) {

	    char *argv[] = {
		"ERROR: agentd: unable to set group id",
	    	 ((char *) NULL)
	    };

	    serverData -> errorLog.error_sys_quit (argv); 
	    abort_with_message ("should never get here");   // should never get here
	}
#endif 

	char *argv[] = {
	    "INFO: agentd: group has been switched to \"",
	    groupName,
	    "\"",
	    ((char *) NULL)
	};

	serverData -> errorLog.error_app_cont (argv);
    }
}

/* redirectOutput

   Purpose: Redirect stdin, stdout and stdout to /dev/null.

     Input: None

    Output: The procedure halts the server and logs an error message if
            it is unable to redirect stdin, stdout and stdout.  Otherwise
	    the procedure redirects stdin, stdout and stdout and then
	    returns.
*/

static void redirectOutput (void)
{
    if (FileUtility::fileRedirect() != FileUtility::e_OK) {
	serverData -> errorLog.error_sys_quit 
	    ("agentd: unable to redirect stdin, stdout and stdout");
    }
}

/* check_environment

    Purpose: Make sure that the necessary environment variables are defined

      Input: None
  
     Output: The procedure returns -1 if the necessary environment variables
	     are not defined.  Otherwise the procedure returns 0.  Currently
	     the procedure only checks for the HOME environment variable which
	     should contain the full pathname of the server account's home 
	     directory.
*/

static int check_environment (void)
{
    char *value;

	/* check for the HOME environment variable */

    if ((value = getenv ("HOME")) == NULL) {

	char *argv[] = {
	     "agentd: ERROR: environment variable \"HOME\" must be set to the full pathname of the agent account's home directory",
	     ((char *) NULL)
	};

	serverData -> errorLog.error_app_cont (argv);
	return -1;
    }

	/* success */

    char *argv[] = {
	"agentd: INFO: environment variable \"HOME\" is set to \"",
	value,
	"\"",
	((char *) NULL) 
    };

    serverData -> errorLog.error_app_cont (argv);
    return 0;
}

/* writePid

   Purpose: Write out the process id (of the main server process) to the
	    process id file

     Input: None

    Output: The function returns -1 on error.  Otherwise the function returns 0
	    and writes out the process id.
*/

int writePid (void)
{
    int fd;
    char temp[16];

	/* open the file */

    if ((fd = open (serverData -> pidFilename.value(), O_CREAT | O_TRUNC | O_WRONLY, 0644)) < 0) {
	
	char *argv[] = {
	    "agentd: ERROR: unable to open the pid file \"",
	    serverData -> pidFilename.value(),
	    "\"",
	    ((char *) NULL)
	};

	serverData -> errorLog.error_app_cont (argv);
	return (-1);
    }

	/* write out the process id */

    sprintf (temp, "%d\n", serverData -> agentdPid);
    write (fd, temp, strlen(temp));

	/* done */

    close (fd);
    return (0);
}

int main (int argc, char **argv)
{
	/* create the server configuration, agents, and command line */
	/* data structures                                           */

    serverData  = new ServerData ();
    agentSet    = new AgentSet ();
    commandLine = new CommandLine ();

	/* catch all signals that might cause a core dump -- the */
	/* signal handler uses serverData so we can not install  */
	/* the handlers until after we create serverData above   */

    const int c_endFlag = 0xFFFF;

    static int coredumpSignals[] = {
	SIGSEGV,
#ifdef SIGBUS
	SIGBUS,
#endif
#ifdef SIGABORT
	SIGABORT,
#endif
#ifdef SIGABRT
	SIGABRT,
#endif
	c_endFlag
    };

    for (int i = 0; coredumpSignals[i] != c_endFlag; ++i) { 

	if (SignalHandlers::install_signal_intr (coredumpSignals[i], (SignalHandlers::funcType) coredump_handler, 0) < 0) { 

	    char *messageArgv[] = {
	 	"agentd: ERROR: unable to install handler for ",
		(char *) SignalHandlers::SignoToString (coredumpSignals[i]),
		(char *) NULL
	    };

	    serverData -> errorLog.error_sys_quit (messageArgv);
	}
    }
 
	/* get the command line parameters */

    if (getCommandLine (argc, argv, *commandLine) < 0) {
	printUsage (argv[0]);
	exit (1);  
    }

	/* load the configuration file */

    if (serverData -> load (*commandLine, ServerData::e_INITIAL_LOAD) < 0) {
	exit (1);
    }

	/* get the pass phrase from the keyboard if we do not have it yet */

    if (getPassphrase() < 0) {
	exit (1);
    }

	/* initialize the cryptographically-secure PRNG if it is not already */

    if (serverData -> randomFilename.isEmpty()) {
	trueRandAccum ((int) serverData -> randomBits);
    }

	/* turn the server into a daemon */

    if (commandLine -> daemon) {
	make_daemon ();
	redirectOutput ();
    }

	/* switch the user and group id's (if possible) */

    switchGroup ();
    switchUser ();

	/* set the process group id to the process id */

    if (commandLine -> daemon) {
	setpgid (0, 0);
    }

	/* clear the file creation mask (otherwise the Unix socket might */
	/* be created with the wrong permissions)                        */

    umask (0);

	/* get the IP address of the current machine */

    if (verify_server_ip (commandLine -> verifyHostname) < 0) {
	exit (1);
    }

	/* construct the server keyname */

    if (construct_server_keyname () < 0) {
	exit (1);
    }

	/* resolve the IP address of the AllowedMachine's and TrustedMachine's */

    serverData -> resolveIp (ServerData::e_INITIAL_LOAD);

	/* make sure the the HOME environment variable is defined */

    if (check_environment () < 0) {
	exit (1);
    }
  
	/* write the main server process id out to the file */

    serverData -> agentdPid = getpid ();

    if (writePid() < 0) {
	exit (1);
    }

	/* ignore SIGPIPE so that we do not die on pipe breakdown    */
	/* and catch SIGCHLD so that we can detect if socketd dies,  */
	/* SIGTERM, SIGINT and SIGQUIT so that we can cleanup at     */
	/* server death, and SIGHUP so that we can reload the server */ 
	/* configuration                                             */

    if (SignalHandlers::install_signal_intr (SIGPIPE, (SignalHandlers::funcType) SIG_IGN, 0) < 0) { 
	serverData -> errorLog.error_sys_quit 
	    ("agentd: unable to ignore SIGPIPE");
    }

    if (SignalHandlers::install_signal_intr (SIGCHLD, sigchld_handler, 0) < 0) {
	serverData -> errorLog.error_sys_quit 
	    ("agentd: unable to install SIGCHLD handler");
    }

    if (SignalHandlers::install_signal_intr (SIGTERM, sigterm_handler, 0) < 0) {
	serverData -> errorLog.error_sys_quit 
	    ("agentd: unable to install SIGTERM handler");
    }

    if (SignalHandlers::install_signal_intr (SIGINT, sigterm_handler, 0) < 0) {
	serverData -> errorLog.error_sys_quit 
	    ("agentd: unable to install SIGINT handler");
    }

    if (SignalHandlers::install_signal_intr (SIGQUIT, sigterm_handler, 0) < 0) {
	serverData -> errorLog.error_sys_quit 
	    ("agentd: unable to install SIGQUIT handler");
    }

    if (SignalHandlers::install_signal_intr (SIGIO, sigio_handler, 0) < 0) {
	serverData -> errorLog.error_sys_quit 
	    ("agentd: unable to install SIGIO handler");
    }

    if (SignalHandlers::install_signal_intr (SIGHUP, sighup_handler, 0) < 0) {
	serverData -> errorLog.error_sys_quit 
	    ("agentd: unable to install SIGHUP handler");
    }

    if (SignalHandlers::install_signal_intr (SIGPROF, sigprof_handler, 0) < 0) {
	serverData -> errorLog.error_sys_quit 
	    ("agentd: unable to install SIGPROF handler");
    }

	/* create the pipes */ 

    int fd[2];

    if (PipeUtility::createStreamPipe (fd) != PipeUtility::e_OK) {
	serverData -> errorLog.error_sys_quit 
	    ("agentd: unable to create the stream pipe\n");
    }

    serverData -> streamSocketd = fd[0];
    serverData -> streamAgentd  = fd[1]; 

	/* open the TCP/IP socket descriptor that brings in remote requests */

    int chosenPort;

    if (tcpip_setup (serverData->socketPort, serverData->sockfd, chosenPort) != e_TCPIP_OK) {

	char temp[16];
	sprintf (temp, "%u", serverData -> socketPort);

	char *argv[] = {
	    "agentd: ERROR: unable to open TCP/IP socket on port ",
	    temp,
	    ((char *) NULL)
	};

	serverData -> errorLog.error_sys_quit (argv);
	abort_with_message ("should never get here");   // should never get here
    }

	/* open the Unix socket descriptor that brings in local requests */

    if (unix_setup (serverData->socketFilename.value(), serverData -> unixSockfd) != e_TCPIP_OK) {

	char *argv[] = {
	    "agentd: ERROR: unable to open Unix domain socket \"",
	    serverData -> socketFilename.value(),
	    "\"",
	    ((char *) NULL)
	};

	serverData -> errorLog.error_sys_quit (argv);
	abort_with_message ("should never get here");   // should never get here
    }

	/* fork off socketd */

    if ((serverData -> socketdPid = fork()) < 0) {

	    /* FORK failed */

	serverData -> errorLog.error_sys_quit 
	    ("agentd: unable to fork \"socketd\"");

    } else if (serverData -> socketdPid == 0) {
      
	    /* CHILD = socketd */

	close (serverData -> streamSocketd);       
	socketd ();

    } else {

	    /* PARENT = agentd; first log the start message */

	char temp[64];
	sprintf (temp, "agentd: INFO: socketd has been started (process id %ld)", (long) serverData -> socketdPid);
	serverData -> errorLog. error_app_cont (temp);

	    /* then go and handle the table */

	close (serverData -> sockfd);
	close (serverData -> unixSockfd);
	close (serverData -> streamAgentd);
	piped ();
    }

	/* should never get here -- prevents a compiler warning with CC */

    return 1; 
}
