#include "platPorting.h"
#include "agentd.h"
#include "servConf.h"
#include "servRandom.h"

    /* server and agent information */

ServerData *serverData	 = NULL;
AgentSet  *agentSet	 = NULL;
CommandLine *commandLine = NULL;

/* sigterm_handler

   Purpose: This procedure handles the SIGTERM signal for all server processes.
*/

static sig_atomic_t sigtermInProgress = 0;

void sigterm_handler (int, SignalHandlers::ClientData)
{
    sigset_t mask;
    RandomFile::RandomFileErrors rc;

	/* bail if a SIGTERM is in progress -- this is required   */
	/* for IRIX where the "kill" statement below appears to   */
	/* be causing a SIGTERM interrupt even though SIGTERM is  */
	/* blocked automatically                                  */

    if (sigtermInProgress) {
	return;
    }

    sigtermInProgress = 1;

	/* block the SIGCHLD signal -- we don't save the old mask */
	/* since we are about to exit from the process and don't  */
	/* need to restore the old mask                           */

    sigemptyset (&mask);
    sigaddset (&mask, SIGCHLD);
    sigprocmask (SIG_BLOCK, &mask, (sigset_t *) NULL);

	/* propagate the signal to the rest of the process group */

    kill (-getpgrp(), SIGTERM);

	/* log a message that says we have received a SIGTERM */

    char pidString[16];
    sprintf (pidString, "%d", (int) getpid());

    char *argv[] = {
	"agentd: INFO: server process ",
	pidString,
	" received a SIGTERM: exiting",
	((char *) NULL)
    };

    serverData -> errorLog.error_app_cont (argv); 

	/* burn the random bits file if we are in agentd */

    if (getpid() == serverData -> agentdPid) {

	RandomFile randomFile (serverData -> randomFilename, serverData -> randomBits);

	if ((rc = randomFile.burnBits()) == RandomFile::e_OK) {

	    char *argv[] = {
		"agentd: INFO: burned random bits file \"",
		serverData -> randomFilename.value(),
		"\" before exit",
		((char *) NULL)
	    };

	    serverData -> errorLog.error_app_cont (argv); 

	} else {

	    char temp[16];
	    sprintf (temp, "%d %d", rc, errno);

	    char *argv[] = {
		"agentd: ERROR: unable to burn random bits file \"",
		serverData -> randomFilename.value(),
		"\" before exit",
		temp,
		((char *) NULL)
	    };

	    serverData -> errorLog.error_app_cont (argv); 
	}
    }

	/* exit -- all open file descriptors are automatically closed */

    exit (0);
}

/* sigio_handler

   Purpose: This procedure handles the SIGIO signal for all server processes.
*/

static sig_atomic_t sigioFired = 0;

int isSigioPending (void)
{
    sigset_t mask;
    sigpending (&mask);
    return (sigismember (&mask, SIGIO));
}
 
int hasSigioFired (void)
{
    int fired = sigioFired;
    sigioFired = 0;
    return (fired);
}

void pretendSigioFired (void) 
{
    sigioFired = 1;
}

void sigio_handler (int, SignalHandlers::ClientData)
{
    sigioFired = 1;
}

/* sighup_handler and sigprof_handler

   Purpose: These procedures handle the SIGHUP signal for all server processes.
*/

static sig_atomic_t sighupFired = 0; 

int isSighupPending (void)
{
    sigset_t mask;
    sigpending (&mask);
    return (sigismember (&mask, SIGPROF));
}

int hasSighupFired (void)
{
    int fired = sighupFired;
    sighupFired = 0;
    return (fired);
}

void pretendSighupFired (void) 
{
    sighupFired = 1;
}

void sighup_handler (int, SignalHandlers::ClientData)
{
	/* propagate the signal to the rest of the process group --  */
	/* we propagate a SIGPROF rather than a SIGHUP so that we do */
	/* not get into an infinite signal loop                      */

    kill (-getpgrp(), SIGPROF);
}

void sigprof_handler (int, SignalHandlers::ClientData)
{
    sighupFired = 1;
}

/* sigchld_handler

   Purpose: This procedure handles the SIGCHLD signal in the "agentd" process.
	    It detects when the "socketd" process dies and brings down the rest
            of the server processes.
*/

const int c_REASON_LENGTH = 256;

void sigchld_handler (int, SignalHandlers::ClientData)
{
    sigset_t mask;
    WAIT_STATUS_TYPE status;    

	/* block the SIGTERM signal -- we don't save the old mask */
	/* since we are about to exit from the process and don't  */
	/* need to restore the old mask                           */

    sigemptyset (&mask);
    sigaddset (&mask, SIGTERM);
    sigprocmask (SIG_BLOCK, &mask, (sigset_t *) NULL);

	/* remove the zombie */

    while (waitpid(0, (int *) &status, WNOHANG) > 0) {
	// empty
    }

	/* log an error message and then generate a SIGTERM to cleanup -- */
	/* include the reason for socketd's failure in the error message  */

    char failureReason[c_REASON_LENGTH];
    char *failureCore = "";

    if (WIFEXITED(status)) {

	int exitCode = WEXITSTATUS(status);
	sprintf (failureReason, "; socketd exited with exit code %d", exitCode);

    } else if (WIFSIGNALED(status)) {

	int signo = WTERMSIG(status);
	const char *signoString = SignalHandlers::SignoToString (signo);
	sprintf (failureReason, "; socketd was killed with %s signal (signo %d)", 
	    signoString, signo);

#ifdef WCOREDUMP
	if (WCOREDUMP(status)) {
	    failureCore = "; core dump performed";
	}
#endif

    } else if (WIFSTOPPED(status)) {

	int signo = WSTOPSIG(status);
	const char *signoString = SignalHandlers::SignoToString (signo);
	sprintf (failureReason, "; socketd was stopped with %s signal (signo %d)", 
	    signoString, signo);

    } else {

	sprintf (failureReason, "; socketd stopped for undetermined reason");
    }

    char *argv[] = {
	"agentd: ERROR: socketd failed so agentd is terminating",
	failureReason,
	failureCore,
	((char *) NULL)
    };

    serverData -> errorLog.error_app_cont (argv);
    sigterm_handler (SIGTERM, 0);

#ifdef FIX_LATER
	/* we might actually end up here, since sigterm_handler returns */
	/* if a sigterm shutdown is already in progress                 */
#endif

    abort_with_message ("should never get here");   // should never get here
}

/* coredump_handler

   Purpose: This procedure handles the SIGSEGV, SIBABRT, SIGABORT and
            SIGBUS signals for all server processes.
*/

void coredump_handler (int signo, SignalHandlers::ClientData)
{
        /*
 	 * Restore the default action for the SIGSEGV, SIGABRT, SIGABORT and
	 * SIGBUS signals (so that we do not end up in an infinite loop)
         */

    SignalHandlers::install_signal (SIGSEGV, (SignalHandlers::funcType) SIG_DFL, 0);
#ifdef SIGBUS
    SignalHandlers::install_signal (SIGBUS, (SignalHandlers::funcType) SIG_DFL, 0);
#endif
#ifdef SIGABORT
    SignalHandlers::install_signal (SIGABORT, (SignalHandlers::funcType) SIG_DFL, 0);
#endif
#ifdef SIGABRT
    SignalHandlers::install_signal (SIGABRT, (SignalHandlers::funcType) SIG_DFL, 0);
#endif

	/*
	 * Get the name of the signal
	 */

    const char *signalName = SignalHandlers::SignoToString (signo);

	/*
	 * Log that we are attempting to dump core
	 */

    char *argv[] = {
	"agentd: ERROR: caught a ",
	(char *) signalName,
	": attempting to dump core in directory \"",
	(char *) serverData -> coredumpDirectory.value(),
	"\"",
	(char *) NULL
    };

    serverData -> errorLog.error_app_cont (argv);

	/*
	 * Change directories -- silently ignore any failures
	 */

    if (!(serverData -> coredumpDirectory.isEmpty())) {
	const char *directory = serverData -> coredumpDirectory.value();
	(void) mkdir (directory, 0700);
	(void) chmod (directory, 0700);
	(void) chdir (directory);
    }

	/*
	 * Abort
	 */

    abort ();
    exit (1);		// better not ever get here
}
