/* Agent Tcl
   Bob Gray
   15 February 1995

   piped.cc

   This file implements "piped" which monitors the pipe from socketd to
   agentd.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "platExclusion.h"
#include "platInterrupt.h"
#include "platSocket.h"
#include "mesgTcpip.h"
#include "piped.h"
#include "agent_set.h"
#include "agentd.h"
#include "genError.h"
#include "genMessage.h"
#include "genTransmit.h"
#include "noise.h"
#include "random.h"
#include "randpool.h"
#include "servConf.h"

typedef MESSAGE * (AgentSet::*Action) (MESSAGE *, BASE_SECURITY &);

    /* maximum number of requests between stirs of the random pool */

const unsigned MAXIMUM_COUNTDOWN = 24;

    /* jump table for requests */

static Action jumpTable[NUM_REQUESTS] =
{
  &AgentSet::agent_begin,       /* REQ_BEGIN     */
  &AgentSet::agent_name,        /* REQ_NAME      */
  &AgentSet::agent_end,         /* REQ_END       */
  &AgentSet::agent_force,       /* REQ_FORCE     */
  NULL,                         /* REQ_SUBMIT    */
  &AgentSet::agent_send,        /* REQ_MESSAGE   */
  &AgentSet::agent_req,         /* REQ_MEETING   */
  &AgentSet::agent_event,       /* REQ_EVENT     */
  &AgentSet::agent_receive,     /* REQ_GET       */
  &AgentSet::agent_info,	/* REQ_INFO      */ 
  &AgentSet::agent_status,      /* REQ_STATUS    */
  &AgentSet::agent_ip,	    	/* REQ_IP        */
  NULL,				/* REQ_AUTH      */
  NULL,				/* REQ_SECURITY  */
  &AgentSet::agent_error	/* REQ_ERROR     */
};

/* piped

   Purpose: Monitor the pipe that carries requests from socketd to agentd

     Input: None

    Output: The procedure should never return.
*/

static MESSAGE *receiveMessageAndSecurity (int fd, BASE_SECURITY *&messageSecurity)
{
    MESSAGE *message = NULL;
    MESSAGE *secureMessage = NULL;

	/* receive the security information */ 

    if ((secureMessage = message_receive (fd, messagesToServer)) == NULL) {
	return (NULL);
    } 

	/* no security information means we already have the actual message */

    UINT_8 flag = secureMessage -> getFlag ();

    if ((flag != REQ_AUTH) && (flag != REQ_SECURITY)) {
	messageSecurity = new BASE_SECURITY ();
	message = secureMessage;
	return (message);
    }

	/* remember the security information and receive the actual message */

    if (flag == REQ_AUTH) {

	messageSecurity = new BASE_SECURITY ();
	messageSecurity -> setAgentAuthenticated (e_TRUE, secureMessage -> getLong(0));

    } else {

	messageSecurity = secureMessage -> getSecurity (0);
    }
   
	/* and then receive the actual message */

    message = message_receive (fd, messagesToServer);
    delete (secureMessage);
    return (message);
}

void piped (void)
{
    int bytes;
    unsigned randCountdown;
    MESSAGE *message;
    MESSAGE *response;
    BASE_SECURITY *security;
    
	/* turn on asynchronous I/O for the pipe */

    if (SocketIO::markAsynchronous (serverData->streamSocketd) != SocketIO::e_Ok) {
	serverData -> errorLog.error_sys_quit
		("piped: ERROR: unable to turn on asynchronous I/O for pipe");
    }

	/* pretend that we have received a SIGIO so that we do not miss anything */

    pretendSigioFired ();

	/* block signals */

    MonitorLock monitor;
    Guard guard (monitor);

	/* number of requests before we stir the random pool */

    randCountdown = ((unsigned) trueRandByte()) % MAXIMUM_COUNTDOWN + 1;

	/* loop forver, handling requests from the other server processes */
 
    while (1) {

	    /* see if SIGHUP has fired */

	if (hasSighupFired()) {

		/* log a message */

	    char *argv[] = {
		"piped: INFO: received the SIGHUP: reloading configuration",
		((char *) NULL)
	    };

	    serverData -> errorLog.error_app_cont (argv);
	
		/* reload the configuration and resolve the IP addresses */

	    if (serverData -> load (*commandLine, ServerData::e_RELOADING_IN_AGENTD) < 0) {

	        char *argv[] = {
		    "piped: ERROR: unable to reload configuration",
		    ((char *) NULL)
	        };

		serverData -> errorLog.error_app_cont (argv);
		sigterm_handler (SIGTERM, 0);	// bail on error
		abort_with_message ("should never get here!");  // shouldnever get here
	    }

	    serverData -> resolveIp (ServerData::e_RELOADING_IN_AGENTD);
	}
 
	    /* see if SIGIO has fired */

	if (hasSigioFired()) {

	    while (1) {

		response = NULL;
		security = NULL;

		    /* see if there is an incoming message */

		if ((ioctl (serverData -> streamSocketd, FIONREAD, (char *) &bytes) < 0) || (bytes <= 0)) {
		    break;
		}
	
		    /* receive the incoming message */
		
		if ((message = receiveMessageAndSecurity (serverData -> streamSocketd, security)) == NULL) {
		    serverData -> errorLog.error_app_quit 
			("piped: ERROR: invalid or interrupted message\n");
		}

#ifdef FIX_LATER
		    /* We should use the jump table defined at the top */
		    /* of this file.                                   */
#endif

		    /* handle the message */
	
		UINT_8 flag = message -> getFlag();

		if ((flag > REQ_ERROR) || (jumpTable[flag] == (Action) NULL)) {
		    response = (MESSAGE *) NULL;
		    serverData -> errorLog.error_app_quit 
			("piped: ERROR: REQ_SUBMIT, REQ_SECURITY and REQ_IP are not supported");

		} else {

		    response = (agentSet->*jumpTable[flag]) (message,*security);
		}

		    /* send the response message down the pipe */

		(void) message_send (serverData -> streamSocketd, *response);

		    /* decrement the countdown and stir the random pool */
		    /* if we have reached zero                          */

		if (--randCountdown == 0) {
		    UINT_8 flagByte = ((response -> getFlag()) << 4) | (message -> getFlag());
		    noise ();
		    randPoolAddBytes ((byte *) &flagByte, 1);
		    randCountdown = ((unsigned) trueRandByte()) % MAXIMUM_COUNTDOWN + 1;
		}

		    /* cleanup */

		delete (security);
		delete (response);
		delete (message);
	    }
	}

	    /* wait for a signal */

	monitor.wait ();
    } 
}
