/* Agent Tcl
   Bob Gray
   25 March 1997 

   genPointerList.cc

   This file implements a list of pointers that supports expected 
   linear-time insertion and deletion (i.e., the list is implemented as a
   hash table). 

   Copyright (c) 1997, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.

   RCS INFORMATION:

   $Id: servPointerList.cc,v 1.2 1999/03/06 20:22:37 rgray Exp $

   $Log: servPointerList.cc,v $
   Revision 1.2  1999/03/06 20:22:37  rgray
   fixed a lot of Java state capture bugs -- about to try to make it thread safe
   Sat Mar  6 15:22:37 EST 1999

   Revision 1.1  1998/08/10 14:57:03  rgray
   Initial revision

   Revision 1.2  1998/02/18 15:08:25  rgray
   compiled under FreeBsd and fixed some Makefile errors
   Wed Feb 18 10:08:25 EST 1998

   Revision 1.1  1997/10/17 06:02:12  rgray
   Initial revision

   Revision 1.1  1997/10/17 05:51:34  rgray
   Initial revision

   Revision 1.1  1997/10/17 05:43:45  rgray
   Initial revision

   Revision 1.5  1997/05/03 18:24:43  rgray
   fixed pragmas

// Revision 1.4  1997/03/26  16:02:19  rgray
// fixed runtime bug
//
   Revision 1.3  1997/03/26 15:07:46  rgray
   server now users a PointerList for name events

   Revision 1.2  1997/03/25 22:53:05  rgray
   fixed compiler errors

   Revision 1.1  1997/03/25 22:44:21  rgray
   Initial revision

*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "servPointerList.h"
#include "genUtility.h"
#include "truefalse.h"

#ifdef FIX_LATER
    /* we need to make the hash table dynamically shrink and expand     */
    /* right now we can get away with a fixed size since we only        */
    /* use the table in situations where the expected number of entries */
    /* is extremely likely to be small				    */
#endif

const int DEFAULT_BUCKETS = 8;    // MUST BE A POWER OF TWO

/* PointerIterator::PointerIterator 

   Purpose: This procedure is the constructor for class PointerIterator.
*/

PointerIterator::PointerIterator (PointerList *p_list)
{
	/* assertions on the parameters */

    assert (p_list != NULL);

	/* initialize */

    PointerIterator::list     = p_list;
    PointerIterator::bucket   = 0;
    PointerIterator::node     = p_list -> table[0];
    PointerIterator::instance = 0;
}

/* PointerIterator::nextPointer

   Purpose: Get the next pointer in the list

     Input: None

    Output: The procedure returns 0 if there are no more pointers in the list.
            Otherwise the procedure returns 1 and sets "pointer" to the next
            pointer.
*/

int PointerIterator::nextPointer (char *& pointer)
{
    while (bucket < list -> maxIndex) {

	while (node != NULL) {

	    while (instance < node -> m_count) {
		pointer = node -> m_pointer;
		instance += 1;
		return (1);
	    }

	    node = node -> m_next;

	    instance = 0;
	}

	bucket += 1;

	if (bucket < list -> maxIndex) {
	    node = list -> table[bucket];
	}
    }

    return (0);
}

/* PointerIterator::nextDistinctPointer

   Purpose: Get the next *distint* pointer in the list

     Input: None

    Output: The procedure returns 0 if there are no more pointers in the list.
            Otherwise the procedure returns 1 and sets "pointer" to the next
            distinct pointer.
*/

int PointerIterator::nextDistinctPointer (char *& pointer)
{
    instance = 0;

    while (bucket < list -> maxIndex) {

	while (node != NULL) {
	    pointer = node -> m_pointer;
	    node = node -> m_next;
	    return (1);
	}

	bucket += 1;

	if (bucket < list -> maxIndex) {
	    node = list -> table[bucket];
	}
    }

    return (0);
}

/* PointerList::PointerList

   Purpose: This procedure is the constructor for class PointerList
*/

PointerList::PointerList (void)
{
    maxIndex = DEFAULT_BUCKETS - 1;
    entries  = 0;
    table    = new PointerNode * [DEFAULT_BUCKETS];

    for (unsigned i = 0; i <= maxIndex; i++) {
	table[i] = (PointerNode *) NULL;
    }
}

/* PointerList::empty

   Purpose: This procedure empties out the pointer list.
*/

void PointerList::empty (void)
{
    register PointerNode *current;
    register PointerNode *next;

	/* delete the table entries */

    for (unsigned i = 0; i <= maxIndex; i++) {

	for (current = table[i]; current != NULL; current = next) {
	    next = current -> m_next;
	    delete current;
        } 

	table[i] = NULL;
    }
}

/* PointerList::~PointerList

   Purpose: This procedure is the destructor for class PointerList.
*/

PointerList::~PointerList ()
{
    empty ();
    delete [] table;
}

/* PointerList::insert

   Purpose: Insert a pointer into the list

     Input: pointer = the pointer
		      (char *) 
*/

void PointerList::insert (char *pointer)
{
    PointerNode *current;

    long key = hash (pointer);

	/* see if the pointer is already in the list */

    for (current = table[key]; current != NULL; current = current -> m_next) {
	if (current -> m_pointer == pointer) {
	    current -> m_count += 1;
	    entries += 1;
	    return;
	}
    }

	/* otherwise add the pointer */

    current = new PointerNode (pointer, table[key]);
    table[key] = current;
    entries += 1;
}

/* PointerList:remove

   Purpose: Remove (a single instance of) a pointer from the list

     Input: pointer = the pointer
                      (char *)
 
    Output: The procedure returns 0 if the pointer does not appear in the list
            and 1 otherwise.
*/

int PointerList::remove (char *pointer)
{
    register PointerNode *current;
    register PointerNode *previous;

       /* check if the entry is in the table */

     long index = hash(pointer);

     for (previous = NULL, current = table[index]; current != NULL; previous = current, current = current -> m_next) {

	if (current -> m_pointer == pointer) {

		// just decrement the count if we have more than one instance

	    if (current -> m_count > 1) {
		current -> m_count -= 1;
		entries -= 1;
		return (1);
	    }

		// otherwise delete the pointer's entire entry

	    if (previous == NULL) {
		table[index] = current -> m_next;
	    } else {
		previous -> m_next = current -> m_next;
	    }

	    delete (current);
	    entries -= 1;
            return (1);
	}
    }

    return (0);
}
 
/* PointerList:removeAll

   Purpose: Remove (all instances of) a pointer from the list

     Input: pointer = the pointer
                      (char *)
 
    Output: The procedure returns 0 if the pointer does not appear in the list
            and 1 otherwise.
*/

int PointerList::removeAll (char *pointer)
{
    register PointerNode *current;
    register PointerNode *previous;

       /* check if the entry is in the table */

     long index = hash(pointer);

     for (previous = NULL, current = table[index]; current != NULL; previous = current, current = current -> m_next) {

	if (current -> m_pointer == pointer) {

	    if (previous == NULL) {
		table[index] = current -> m_next;
	    } else {
		previous -> m_next = current -> m_next;
	    }

	    entries -= current -> m_count;
	    delete (current);
            return (1);
	}
    }

    return (0);
} 
