/* Agent Tcl
   Bob Gray
   9 February 1995

   servProcess.cc 

   This file implements the functions that the server uses to HANDLE a client
   request.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "platExclusion.h"
#include "platInterrupt.h"
#include "platSocket.h"		    // SocketIO
#include "mesgConnection.h"
#include "mesgTcpip.h"
#include "agentAgentId.h"	    // AgentId
#include "genMesgDriver.h"
#include "genServerInterface.h"	    // ServerInterface

#include "agentd.h"
#include "genConversion.h"
#include "genError.h"
#include "genFile.h"
#include "genPipe.h"
#include "genManager.h"
#include "genMessage.h"
#include "genTransmit.h"
#include "genUtility.h"
#include "interp.h"
#include "piped.h"
#include "random.h"
#include "servConf.h"
#include "servProcess.h"
#include "servReceive.h"
#include "truefalse.h"

    /* error messages */

static char *const transferError = 
	"system error: unable to transfer state image to the interpreter";

    /* forward declarations */

static void sigchld_handler (
	int,
	SignalHandlers::ClientData
    );

static int forkAgentProcess (
	int sockfd, 
	BASE_SECURITY &security, 
	MESSAGE *message, 
	int &agentFd, 
	pid_t &agentPid,
	AgentId &localId, 
	AgentId &rootId
    );

static int sendErrorMessage (
	char *message,
	AgentId& root,
	AgentId& local,
	BASE_SECURITY &security
    );

static int checkAbnormalExit (
	AgentId& root,
	AgentId& local,
	BASE_SECURITY &security,
	WAIT_STATUS_TYPE status
    );

/* removeDirectory

   Purpose: Delete a directory and all its contents

     Input: directory = name of the directory
			(DynamicString)

    Output: The procedure returns -1 on error and 0 on success.
*/

int removeDirectory (DynamicString directory)
{
    DIR *dir;
    int rc = 0;
    struct dirent *dirEntry;

	/*
	 * open the directory
	 */

    if ((dir = opendir (directory.value())) == NULL) {

	if (errno == ENOENT) {
	    return (0);		// doesn't exist -- no error
	} else {
	    return (-1);	// anything else -- error 
	}
    }

	/*
	 * loop through each directory entry
	 */

    while ((dirEntry = readdir (dir)) != NULL) {

	    // skip over directory entries . and ..
 
	if (!strcmp (dirEntry -> d_name, ".")) {
	    continue;
	}

	if (!strcmp (dirEntry -> d_name, "..")) {
	    continue;
	}

	    // construct the full pathname of the entry

	DynamicString path = directory;
	path.append ("/");
	path.append (dirEntry -> d_name);

	    // see if the entry is a subdirectory; if it is a subdirectory,
	    // then recursively call removeDirectory to remove the
	    // subdirectory; otherwise use unlink to remove the entry

	FileUtility::FileInformation fileInfo;

	if (FileUtility::getFileStats (path.value(), fileInfo) != FileUtility::e_OK) {
	    rc = -1;
	    break;
	}

	if (fileInfo.type == FileUtility::e_DIRECTORY) {

	    if (removeDirectory (path) < 0) {
		rc = -1;
		break;
	    }

	} else {

	    if (unlink (path.value()) < 0) {
		rc = -1;
		break;
	    }
	}
    }

	/*
	 * close the directory, delete the directory if everything else worked,
	 * and then return
	 */

    closedir (dir);

    if (rc == 0) {
	if (rmdir (directory.value()) < 0) {
	    rc = -1;
	}
    }

    return (rc);
}

/* sendMessage

   Purpose: Send an incoming message to the other half of the server 

     Input: message = the incoming message
		      (struct MESSAGE *)

    Output: The procedure returns NULL on error and a pointer to the
	    a dynamically allocated MESSAGE structure that contains the
            server's response on success.
*/

static MESSAGE *sendMessage (MESSAGE *message)
{
    MESSAGE *response = NULL;

	/* prevent interrupts -- i.e., we must do the agent_receive! */

    MonitorLock lock;
    Guard guard (lock);

	/* send the message to the server */

#ifdef FIX_LATER
	/* the error_sys_quit gives us a way out of process without */
	/* doing proper cleanup                                     */
#endif

    if (FileUtility::acquireFileLock (serverData -> lockfd) != FileUtility::e_OK) {
	serverData -> errorLog.error_sys_quit ("handler: unable to acquire connection lock");
    }

    if (message_send (serverData -> streamAgentd, *message) >= 0) {
	response = message_receive (serverData -> streamAgentd, messagesFromServer);
    }

    (void) FileUtility::releaseFileLock (serverData -> lockfd);
    
	/* done */

    return (response);
}

/* sendMessageAndSecurity

   Purpose: Send an incoming message and its security information up to
	    the other half of the server

     Input: sender  = security information for the incoming message
		      (class BASE_SECURITY &)

	    message = the incoming message
		      (struct MESSAGE *)

    Output: The procedure returns NULL on error and a pointer to the
	    a dynamically allocated MESSAGE structure that contains the
            server's response on success.
*/

enum SecurityType {
    e_DEFAULT_SECURITY,
    e_FULL_SECURITY
};

static MESSAGE *sendMessageAndSecurity (BASE_SECURITY &sender, MESSAGE *message, enum SecurityType securityType)
{
    MESSAGE *response;
    MESSAGE *secureMessage;

	/* construct the security message */

    if (sender.isAgentAuthenticated() && (securityType == e_DEFAULT_SECURITY)) {

	secureMessage = new MESSAGE (messagesToServer.messages[REQ_AUTH]);
	secureMessage -> addLong (0, sender.getAgentId());

    } else {

	secureMessage = new MESSAGE (messagesToServer.messages[REQ_SECURITY]);
	secureMessage -> addSecurity (0, &sender);
    }

	/* prevent interrupts -- i.e., we must do the agent_receive */ 

    MonitorLock lock;
    Guard guard (lock);

	/* send the message */

#ifdef FIX_LATER
	/* the error_sys_quit gives us a way out of the process without */
	/* doing proper cleanup                                         */
#endif

    if (FileUtility::acquireFileLock (serverData -> lockfd) != FileUtility::e_OK) {
	serverData -> errorLog.error_sys_quit ("handler: unable to acquire connection lock");
    }

    if (message_send (serverData -> streamAgentd, *secureMessage) < 0) {
	response = NULL;
    } else if (message_send (serverData -> streamAgentd, *message) < 0) {
	response = NULL;
    } else {
	response = message_receive (serverData -> streamAgentd, messagesFromServer);
    }

    (void) FileUtility::releaseFileLock (serverData -> lockfd);

	/* done */

    delete (secureMessage);
    return (response);
}

/* respondWithError

   Purpose: Send a RESP_ERROR message down the socket

     Input: sockfd = socket descriptor
		     (int)

	    code   = server error code
		     (unsigned char)

    Output: The procedure constructs a RESP_ERROR message containing the
            given error code and sends the message down the socket.
*/

void respondWithError (int sockfd, UINT_8 code)
{
    MESSAGE message (messagesFromServer.messages[RESP_ERROR]);
    message.addByte (0, code);
    (void) message_send (sockfd, message);
}

/* sendErrorMessage

   Purpose: Send an error message to the root agent

     Input: string   = the message
		       (char *)

	    root     = the root agent
		       (const AgentId&) 

	    local    = the local agent that caused the error
		       (const AgentId&)

	    security = security information for the local agent
		       (const BASE_SECURITY &)

    Output: The procedure returns -1 on error and 0 otherwise.
*/

const int maximumTries		= 4;
const int initialSleepTime	= 1;
const int maximumConnectionTime = 120;

int sendErrorMessage (char *string, AgentId& root, AgentId& local, BASE_SECURITY &security)
{
	/* northing to do if we do not have a root agent or if the root */
	/* and local agents are the same                                */

    if (root.getServerName().isEmpty()) {
	return 0;
    }

    if ((root.getServerIp() == local.getServerIp()) && (root.getId() == local.getId())) {
	return 0;
    }

	/* if the root agent is on the same machine, just send the message */
	/* to the other half of the server				   */

    if (root.getServerIp() == local.getServerIp()) {

	MESSAGE *response;

	     /* construct the message */

	MESSAGE message (messagesToServer.messages[REQ_ERROR]);
	message.addId (0, &local);
	message.addString (1, NULL); 
	message.addLong (2, root.getId());
	message.addLong (3, 1);
	message.addString (4, string);

	    /* send the message to the other server process */

	if ((response = sendMessageAndSecurity (security, &message, e_FULL_SECURITY)) == NULL) {
	    return -1;
	}

	delete (response);
	return 0;
    }

	/* if the root agent is on a different machine, we must transmit   */
	/* the message with signatures and encryption if desired           */

    int rc = 0;
    int tries = 0;
    int sleepTime = initialSleepTime;
    struct timeval connectionTv = {maximumConnectionTime, 0};
    MachineId remoteMachine (root.getServerName(), root.getServerIp());

    while (1) {

	    /* connect to the remote machine */

	TcpipClientConnection connection;

	struct timeval stop =
	    TimevalUtil::addTimevals (SystemUtil::getCurrentWall(), connectionTv);

	if ((rc = connection.connect (remoteMachine, serverData -> socketPort, stop)) < 0) {

	    if (++tries > maximumTries) {
		break;
	    }

	    sleep (sleepTime);
	    sleepTime *= 2;
	    continue;
	}

	    /* send the message */

	MESSAGE message (messagesToServer.messages[REQ_MESSAGE]);
	message.addId (0, &local);
	message.addString (1, NULL); 
	message.addLong (2, root.getId());
	message.addLong (3, 1);
	message.addString (4, string);

	if ((rc = MessageDriver::secureSend (connection, remoteMachine, security, message)) != MESSAGE_OK) {
	    break;
	}

	    /* done */

	return (0);
    }

	/* log an error message if we were unable to send */

    DynamicString serverString = remoteMachine.getName();
    char *agentString          = Agent_IdToString (&local);

    char *argv[] = {
	"handler: ERROR: unable to send abnormal termination message for agent \"",
	agentString,
	"\" to machine \"",
	serverString.value(),
	"\"",
	((char *) NULL)
    };

    serverData -> errorLog.error_app_cont (argv);
    delete (agentString);
    return (-1);
}

/* checkAbnormalExit 

   Purpose: Check the exit status of the agent interpreter and send an error
            message to the root agent if the exit code was not 0 or if the
            exit was caused by a signal

     Input: root     = the root agent
		       (const AgentId&) 

	    local    = the local agent that caused the error
		       (const AgentId&)

	    security = security information for the local agent
		       (const BASE_SECURITY &)

	    status   = exit status of the agent interpreter
		       (WAIT_STATUS_TYPE)

    Output: The procedure returns -1 on error and 0 otherwise.
*/

int checkAbnormalExit (AgentId& root, AgentId& local, BASE_SECURITY &security, WAIT_STATUS_TYPE status)
{
    int rc;

	/* no error message on a normal exit */

    if (WIFEXITED(status) && (WEXITSTATUS(status) == 0)) {
	return 0;
    }

	/* construct the error message */

    DynamicString errorString;
    errorString.append ("abnormal agent exit: ");

    if (WIFEXITED(status)) {

	int exitcode = WEXITSTATUS(status);

	switch (exitcode) {

	    case ServerInterface::e_GLOBAL_WALL_VIOLATION:
		errorString.append ("violation of global wall-time permit");
		break;

	    case ServerInterface::e_GLOBAL_CPU_VIOLATION:
		errorString.append ("violation of global cpu-time permit");
		break;

	    case ServerInterface::e_GLOBAL_JUMPS_VIOLATION:
		errorString.append ("violation of global jumps permit");
		break;

	    case ServerInterface::e_GLOBAL_CHILDREN_VIOLATION:
		errorString.append ("violation of global children permit");
		break;

	    case ServerInterface::e_GLOBAL_DEPTH_VIOLATION:
		errorString.append ("violation of global depth permit");
		break;

	    case ServerInterface::e_CAN_NOT_RESTORE_STATE:
		errorString.append ("can not restore agent state");
		break;

	    case ServerInterface::e_RECEIVED_SIGTERM:
		errorString.append ("agent received a SIGTERM");
		break;

	    default:
		char temp[16];
		sprintf (temp, "%d", exitcode);
		errorString.append ("exited with code ");
		errorString.append (temp);
	}
	
    } else if (WIFSIGNALED (status)) {

	char temp[16];
	sprintf (temp, "%d", WTERMSIG(status));
	errorString.append ("killed with signal ");
	errorString.append (temp);

#ifdef WCOREDUMP
	if (WCOREDUMP(status)) {
	    errorString.append (" (core dump performed)");
	}
#endif

    } else if (WIFSTOPPED (status)) {

	char temp[16];
	sprintf (temp, "%d", WSTOPSIG(status));
	errorString.append ("stopped with signal "); 
	errorString.append (temp);

    } else {

	errorString.append ("unknown exit status");
    } 

	/* send the error message */

    rc = sendErrorMessage (errorString.value(), root, local, security);

	/* log the error message */

    char *stringId = Agent_IdToString (&local);

    char *argv[] = {
	"agent \"",
	stringId,
	"\": ",
	errorString.value(),
	((char *) NULL)
    };

    serverData -> errorLog.error_app_cont (argv);
    delete (stringId);

	/* done */

    return (rc);
}
 
/* remove_id

   Purpose: Remove an agent from the server tables.

     Input: id     = numeric id of the agent
		     (UINT_32)

	    sender = security information for the agent that is ending
		     (BASE_SECURITY &)

    Output: The procedure returns -1 on error.  Otherwise the procedure
            removes the agent from the tables and returns 0.
*/

static int remove_id (UINT_32 id, BASE_SECURITY &sender)
{
    int code = 0;
    MESSAGE *response = NULL;

	/* construct the REQ_END message */

    MESSAGE message (messagesToServer.messages[REQ_END]);
    message.addString (0, NULL);
    message.addLong (1, id);

	/* send the message to the server */

    response = sendMessageAndSecurity (sender, &message, e_DEFAULT_SECURITY);

	/* return -1 on error */

    UINT_8 flag = response -> getFlag();

    if ((response == NULL) || (flag != RESP_ID)) {
	code = -1;
    }

    delete (response);
    return (code);
}

static MESSAGE *getLocalid (BASE_SECURITY &sender, pid_t pid)
{
	/* set up the message */

    MESSAGE message (messagesToServer.messages[REQ_BEGIN]);
    message.addString (0, NULL);
    message.addIp (1, 0);
    message.addLong (2, pid);

	/* send the message to the server */

    MESSAGE *response = sendMessageAndSecurity (sender, &message, e_DEFAULT_SECURITY);
    return (response);
}
 
static MESSAGE *registerAndReply (int sockfd, BASE_SECURITY &sender, pid_t pid) 
{
    MESSAGE *message;

	/* get a local id for the agent and return that id */

    if ((message = getLocalid (sender, pid)) == NULL) {

	respondWithError (sockfd, ServerInterface::e_SERV_ERR_UNKNOWN);

    } else if (message -> getFlag() != RESP_BEGIN) {

	if (message -> getFlag() == RESP_ERROR) {
	    respondWithError (sockfd, message -> getByte(0));
	} else {
	    respondWithError (sockfd, ServerInterface::e_SERV_ERR_UNKNOWN);
	}

	delete (message);
	message = NULL;

    } else {

	    /* construct the AgentId response */

	MESSAGE *response = new MESSAGE (messagesFromServer.messages[RESP_ID]);
	AgentId *localId = message -> getId (0);
	response -> addId (0, localId, e_FALSE);
	message -> addId (0, localId, e_TRUE);

	    /* send the AgentId response */

	message_send (sockfd, *response);
	delete (response);
    }

    return (message);
}

/* handleNewAgent

   Purpose: This procedure handles the messages

            REQ_SUBMIT

     Input: sockfd     = socket on which the script is arriving
                         (int)
 
            sender     = security description of the sender
                         (class BASE_SECURITY &)

            message    = the message
	                 (class MESSAGE *)

	    background = e_TRUE if the agent arrived on the background
			 connection
			 (BOOLEAN)
 
    Output: The procedure returns -1 if it is unable to start execution.
            Otherwise the procedure returns 0.  If the request for a new
	    agent came in over a background connection, the procedure
	    forks in order to create a new background connection for the
	    new agent.  In this case the procedure essentially returns
	    twice.  It returns in the existing background process with
	    newHandler set to NULL.  It returns in the new background process
	    with newHandler set to a dynamically allocated NewBackgroundHandler
	    that contains enough information to finish setting up the new 
	    background connection.
*/

static BOOLEAN sigusr2Fired = e_FALSE;

void sigusr2_handler (int, SignalHandlers::ClientData)
{
    sigusr2Fired = e_TRUE;
    return;
}

int handleNewAgent (int sockfd, BASE_SECURITY &security, MESSAGE *message, BOOLEAN background, NewBackgroundHandler*& newHandler)
{
    int rc;
    pid_t pid;
    int agentFd;
    pid_t agentPid;
    pid_t handlerPid = 0;    
    AgentId localId;
    AgentId rootId;

	/* assume that there will not be a new background handler */

    newHandler = NULL;

	/* if the request arrived on a background connection, we need to do */
	/* a fork here so that we end up with another background handler -- */
	/* the parent does wait for a signal from the child since otherwise */
	/* the parent and child will be using the same socket               */

    if (background) { 

	    /* get the process id */

	handlerPid = getpid ();

	    /* SIGUSR2 has not yet fired for this agent */

	sigusr2Fired = e_FALSE;
	
	    /* mutual exclusion */

	MonitorLock monitor;
	Guard guard (monitor);

	    /* fork */	
	
	if ((pid = fork()) < 0) {	// ERROR in the fork

	    serverData -> errorLog.error_sys_cont
		    ("unable to fork child process");

	    respondWithError (sockfd, ServerInterface::e_SERV_ERR_UNKNOWN);
	    return -1;
	}

	if (pid > 0) {			// PARENT PROCESS 
	
	    if (isSighupPending()) {	// make sure child does not miss a SIGHUP
		kill (pid, SIGPROF);
	    }

	    while (!sigusr2Fired) {
		monitor.wait ();
	    }

	    return 0;
	}

	    // CHILD PROCESS -- everything below here

	    /*
	     * we reopen the lock file to avoid weirdness involving  
	     * locks that are inherited across forks
	     */

	close (serverData -> lockfd);

	do {
	    serverData -> lockfd = FileUtility::open (serverData -> lockFilename.value(), O_RDWR | O_TRUNC | O_CREAT, 0600);
	} while ((serverData -> lockfd < 0) && ((errno == EINTR) || (errno == EAGAIN)));
	   
	if (serverData -> lockfd < 0) {

	    serverData -> errorLog.error_sys_quit 
		    ("handleNewAgent: unable to open lock file");

	    respondWithError (sockfd, ServerInterface::e_SERV_ERR_UNKNOWN);
	    exit (1);
	}

	    /*
	     * and we turn off any SIGCHLD handler so that we don't 
	     * detect and process child death before we're ready
	     */

	if (SignalHandlers::install_signal (SIGCHLD, (SignalHandlers::funcType) SIG_DFL, 0) < 0) {
	    serverData -> errorLog.error_sys_quit
		("handleNewAgent: unable to install default SIGCHLD");
	}
    }

	/* now start up the agent interpreter */

    rc = forkAgentProcess (sockfd, security, message, agentFd, agentPid, localId, rootId);

	/* tell the parent to proceed */

    if (background) {
	kill (handlerPid, SIGUSR2);
    }

	/* check for failure */ 

    if (rc < 0) {

	if (background) {	// child EXITS on failure
	    exit (1);
	}

	return (-1);		// non-child RETURNS an error code
    }

	/* need a background handler on success */

    newHandler = new NewBackgroundHandler ();
    newHandler -> localId = localId;
    newHandler -> rootId = rootId;
    newHandler -> sockfd = agentFd;
    newHandler -> pid = agentPid;
    return 0;
}

int forkAgentProcess (int sockfd, BASE_SECURITY &security, MESSAGE *message, int &backgroundFd, pid_t &agentPid, AgentId &localId, AgentId &rootId)
{
    int fd[2];
    char temp[256]; 
    HashNode *node = NULL;  
    MESSAGE *registerMessage;

	/* extract the root identification and language name */

    UINT_32 masksSize;
    UINT_32 scriptSize;
    UINT_32 restrictSize;
    ServerInterface::SubmissionTypes type = (ServerInterface::SubmissionTypes) message -> getByte (0);
    SmartIdPtr root             (message -> getId (1));
    SmartIdPtr rootHome         (message -> getId (2));
    SmartIdPtr source           (message -> getId (3));
    SmartIdPtr sourceHome       (message -> getId (4));
    char *language              = message -> getString (5);
    SmartCharPtr languageMgr (language);
    SmartCharPtr script         (message -> getBinary (6, scriptSize)); 
    SmartCharPtr restrictBuffer (message -> getBinary (7, restrictSize));
    SmartCharPtr masksBuffer    (message -> getBinary (8, masksSize));

	/* make sure that we have a valid type flag */

    if (ServerInterface::isSubmissionTypeOutOfRange (type)) {
	respondWithError (sockfd, ServerInterface::e_SERV_ERR_INVALID);
	return -1;
    }

	/* find the language in the list of supported interpreters */

    if (((char *) language == NULL) || ((node = serverData -> interpTable.lookup (0, language)) == NULL)) {
	sprintf (temp, "no interpreter for language \"%s\"", (char *) language); 
	serverData -> errorLog.error_sys_cont (temp);
	respondWithError (sockfd, ServerInterface::e_SERV_ERR_INTERP);
	return -1;
    }

    INTERP *interp = (INTERP *) node -> data;

	/* check that the langauge supports state if we have a state image */

    if ((type != ServerInterface::e_SUB_SCRIPT) && 
	(type != ServerInterface::e_SUB_AGLET_JUMP) &&
	(type != ServerInterface::e_SUB_AGLET_SUBMIT) &&
	(interp -> getType() != INTERP::e_STATE_INTERP)) {

	sprintf (temp, "interpreter for language \"%s\" does not accept state images", (char *) language);
	serverData -> errorLog.error_sys_cont
		(temp);

	respondWithError (sockfd, ServerInterface::e_SERV_ERR_UNKNOWN);
	return -1;
    }

	/* create the socket pair */

    if (PipeUtility::createStreamPipe (fd) != PipeUtility::e_OK) {

	serverData -> errorLog.error_sys_cont
		("unable to create the socket pair");

	respondWithError (sockfd, ServerInterface::e_SERV_ERR_UNKNOWN);
	return -1;
    }

	/* construct the environment variable */

    strcpy (temp, "CONNECTION=");
    fastIntegerToAscii (fd[1], &temp[11]);

	/* fork the child that will then exec the interpreter */

    if ((agentPid = fork()) < 0) {		// ERROR 

	serverData -> errorLog.error_sys_cont
		("unable to fork child process");

	respondWithError (sockfd, ServerInterface::e_SERV_ERR_UNKNOWN);
	close (fd[1]);
	close (fd[0]);
	return -1;
    }

    if (agentPid == 0) {			// CHILD PROCESS

	    /* put the CONNECTION entry in the environ list */

	serverData -> agentEnviron[serverData -> connectionEnvSlot] = temp; 

	    /* close down the files */

	close (fd[0]);
	close (sockfd);
	close (serverData -> lockfd);
	close (serverData -> streamAgentd);

	    /* get rid of any pending signals -- this is VERY important */
	    /* since signals are the basis for all mutual exclusion     */
	    /* within the agent interpreters                            */

	(void) SignalHandlers::install_signal (SIGIO  , (SignalHandlers::funcType) SIG_IGN, 0);
	(void) SignalHandlers::install_signal (SIGALRM, (SignalHandlers::funcType) SIG_IGN, 0);
	(void) SignalHandlers::install_signal (SIGPROF, (SignalHandlers::funcType) SIG_IGN, 0);
	(void) SignalHandlers::install_signal (SIGUSR1, (SignalHandlers::funcType) SIG_IGN, 0);
	(void) SignalHandlers::install_signal (SIGUSR2, (SignalHandlers::funcType) SIG_IGN, 0);
	(void) SignalHandlers::install_signal (SIGHUP , (SignalHandlers::funcType) SIG_IGN, 0);
	(void) SignalHandlers::install_signal (SIGTERM, (SignalHandlers::funcType) SIG_IGN, 0);
	(void) SignalHandlers::install_signal (SIGSEGV, (SignalHandlers::funcType) SIG_IGN, 0);
#ifdef SIGBUS
	(void) SignalHandlers::install_signal (SIGBUS, (SignalHandlers::funcType) SIG_IGN, 0);
#endif
#ifdef SIGABORT
	(void) SignalHandlers::install_signal (SIGABORT, (SignalHandlers::funcType) SIG_IGN, 0);
#endif
#ifdef SIGABRT
	(void) SignalHandlers::install_signal (SIGABRT, (SignalHandlers::funcType) SIG_IGN, 0);
#endif
	(void) SignalHandlers::install_signal (SIGCHLD, (SignalHandlers::funcType) SIG_DFL, 0);

	    /* reset the signal mask just to be safe */

	sigset_t emptySet;
	sigemptyset (&emptySet);
	sigprocmask (SIG_SETMASK, &emptySet, NULL);

	    /* do the exec */

	char **argv = interp -> getArgv();

	if (execve (interp -> getExecutable().value(), argv, serverData -> agentEnviron) < 0) {
	    exit (1);
	}

	abort_with_message ("should never get here");   // should never get here
    }

    close (fd[1]);			// PARENT PROCESS

	/* get a new agent identification */

    if ((registerMessage = registerAndReply (sockfd, security, agentPid)) == NULL) {
	close (fd[0]);
	return (-1);
    }

    UINT_32 randSize;
    SmartIdPtr local        (registerMessage -> getId (0));
    SmartCharPtr randBuffer (registerMessage -> getBinary (1, randSize));
  
	/* get the trusted status of the agent's sending machine AND THEN */
	/* slide the current machine into the security structure          */

    UINT_16 trustedFlag = 0;

    if (security.isMachineTrusted() && security.isMachineAuthenticated()) {
	trustedFlag = e_MACHINE_TRUSTED;
    } else if (security.isMachineTrusted()) {
	trustedFlag = e_MACHINE_TRUSTED_BUT_UNVERIFIED;
    } else {
	trustedFlag = e_MACHINE_UNTRUSTED;
    }

    security.setAgentAuthenticated (e_TRUE, local -> getId());
    security.transferMachine (serverData -> security);
    security.setMachineTrusted (e_TRUE);
    security.setOwnerSigned (e_FALSE);
 
	/* send down the message */

    MESSAGE interpMessage (messagesToInterpreter.messages[INTERP_SUBMIT]);
    interpMessage.addByte (0, type);
    interpMessage.addId (1, root);
    interpMessage.addId (2, rootHome);
    interpMessage.addId (3, local);
    interpMessage.addId (4, 
	((type == ServerInterface::e_SUB_JUMP) ||
	 (type == ServerInterface::e_SUB_AGLET_JUMP))
	? sourceHome : local);
    interpMessage.addBinary (5, scriptSize, script);
    interpMessage.addSecurity (6, &security);
    interpMessage.addBinary (7, randSize, randBuffer);
    interpMessage.addBinary (8, restrictSize, restrictBuffer);
    interpMessage.addBinary (9, masksSize, masksBuffer);
    interpMessage.addByte (10, trustedFlag);

    if (security.isOwnerAuthenticated ()) {
	interpMessage.addPermit (11, &serverData -> authorizedPermit);
    } else {
	interpMessage.addPermit (11, &serverData -> anonymousPermit);
    }

    interpMessage.addString (12, serverData -> tempDirectory.value());

    if (message_send (fd[0], interpMessage) < 0) {
	sendErrorMessage (transferError, *root, *local, security);
	close (fd[0]);
	return -1;
    }

	/* remember the root and local identifications and background fd */

    rootId	 = *root;
    localId	 = *local;
    backgroundFd = fd[0];
    return 0;
}

/* handleSend

   Purpose: This procedure handles the messages

            REQ_BEGIN
            REQ_NAME
            REQ_END
	    REQ_FORCE
            REQ_MESSAGE
            REQ_MEETING
            REQ_EVENT
	    REQ_INFO
	    REQ_STATUS
	    REQ_IP

     Input: sockfd     = socket descriptor
		         (int)

            security   = security description of the sender
	                 (class BASE_SECURITY &)

            message    = the message
		         (class MESSAGE *)

	    background = e_TRUE if request arrived over the background
			 connection
			 (BOOLEAN)
  
    Output: The procedure returns -1 on error, 0 on a success that should
	    not cause the background handler to terminate, and 1 on a 
	    success that should cause the background handler to terminate
	    (i.e. a successful AGENT_END).  newHandler is always set to NULL.
*/

int handleSend (int sockfd, BASE_SECURITY &sender, MESSAGE *message, BOOLEAN background, NewBackgroundHandler*& newHandler)
{
    MESSAGE *response = NULL;

	/* assertions */

    assert (message != NULL);

	/* there is never a new background handler */

    newHandler = NULL;
 
	/* send the security information and the message to "agentd" */

    if ((response = sendMessageAndSecurity (sender, message, e_DEFAULT_SECURITY)) == NULL) {
	respondWithError (sockfd, ServerInterface::e_SERV_ERR_UNKNOWN);
	close (sockfd);
	return -1;
    }

	/* send down the response */

    (void) message_send (sockfd, *response);

	/* we are going to become a background handler if this is a BEGIN */

    if ((message -> getFlag() == REQ_BEGIN) && (response -> getFlag() == RESP_BEGIN)) {

	    /* make the new handler structure */ 

	AgentId *localId = response -> getId (0);
	newHandler = new NewBackgroundHandler ();
	newHandler -> sockfd = sockfd;
	newHandler -> localId = *localId;
	newHandler -> rootId = *localId;
	newHandler -> pid = 0;
	delete (localId);

	    /* fill in the machine part of security */

	sender.transferMachine (serverData -> security);
	sender.setMachineTrusted (e_FALSE);
	sender.setAgentAuthenticated (e_TRUE, newHandler -> localId.getId());

	    /* cleanup -- do NOT close the socket */

	delete (response);
	return (0);
    }

	/* the background handler is done if this was an AGENT_END */

    if ((message -> getFlag() == REQ_END) && (response -> getFlag() == RESP_ID)) {

	    /* better be a background handler */

	assert (background == e_TRUE);

	    /* cleanup -- do NOT close the socket */

	delete (response);
	return (1);
    }

	/* cleanup -- do NOT close the socket */

    delete (response);
    return (0); 
}

/* backgroundConnection

   Purpose: This procedure is the background handler associated with each
	    agent.  It is established on agent creation and then handles
	    all further agent requests and forwards incoming communication to
	    the agent. 

     Input: sockfd   = socket descriptor
                       (int)

            sender   = security description of the sender
                       (class BASE_SECURITY &)

	    localId  = local id of the agent
		       (const AgentId &)
 
    Output: The procedure returns -1 and sends a RESP_ERROR message
            along the socket if an error occurs.  Otherwise the procedure
            loops forver and sends all incoming messages, meetings and
            events down the socket to the agent.  The procedure also handles
	    any requests that arrive on the socket.
*/

#ifdef FIX_LATER
    /* we will need a per-thread copy of all this data once the */
    /* server is multi-threaded (assuming that we have not      */
    /* eliminated the need to use all these signals             */
#endif

static BOOLEAN          sigusr1Fired;     // flag that indicates SIGUSR1 has fired
static BOOLEAN          sigchldFired;     // flag that indicates SIGCHLD has fired
static pid_t            sigchldPid;       // pid of the associated agent interpreter
static WAIT_STATUS_TYPE sigchldStatus;    // exit status of the associated agent interpreter
static BOOLEAN		sigalrmFired;	  // flag that indicates SIGALRM has fired

void sigusr1_handler (int, SignalHandlers::ClientData)
{
    sigusr1Fired = e_TRUE;
    return;
}

void sigalrm_handler (int, SignalHandlers::ClientData)
{
	/* we set sigioFired to e_TRUE since the purpose of the SIGALRM  */
	/* is to force a check of the background connection -- on Linux  */
	/* machines a SIGIO is not generated (even though it should be)  */
	/* when the interpreter closes the background connection         */

    sigalrmFired = e_TRUE;
    return;
}

static void sigchld_handler (int, SignalHandlers::ClientData) 
{
    pid_t pid;
    int oldErrno;
    WAIT_STATUS_TYPE status;

	/* save the old errno */

    oldErrno = errno;

	/* reap the zombie */

    while ((pid = waitpid (0, (int *) &status, WNOHANG)) > 0) {
	if (pid == sigchldPid) {
	    sigchldFired = e_TRUE;
	    sigchldStatus = status;
	}
    }

	/* restore the old errno */

    errno = oldErrno;
}

int backgroundConnection (BASE_SECURITY &sender, NewBackgroundHandler &handler, NewBackgroundHandler *&newHandler) 
{
    int code;
    fd_set readset; 
    BOOLEAN goToSleep;
    struct timeval noTimeout;
    MESSAGE *response = NULL;
    BOOLEAN removeId = e_FALSE;
    struct itimerval pollTimeout; 

	/* assume that there is no new background handler */

    newHandler = NULL;

	/* create the REQ_GET message */ 

    MESSAGE message (messagesToServer.messages[REQ_GET]);
    message.addLong (0, handler.localId.getId());
    message.addLong (1, (UINT_32) getpid());
    message.linearize ();

	/* turn on asynchronous I/O for the socket */

    if (SocketIO::markAsynchronous (handler.sockfd) != SocketIO::e_Ok) {
	serverData -> errorLog.error_sys_quit
	    ("backgroundConnection: unable to turn on asychronous I/O for agent socket");
    }

	/* initialize the "setitimer" timeout */

    pollTimeout.it_interval.tv_sec 	= 0;
    pollTimeout.it_interval.tv_usec 	= 0;
    pollTimeout.it_value.tv_sec		= 1;
    pollTimeout.it_value.tv_usec	= 0;

	/* block signals for mutual exclusion */

    MonitorLock monitor;
    Guard guard (monitor); 
 
	/* set up to handle I/O events and child death */

    pretendSigioFired ();
    sigusr1Fired = e_TRUE;
    sigchldFired = e_FALSE;
    sigchldPid = handler.pid;

    if (SignalHandlers::install_signal_intr (SIGCHLD, sigchld_handler, 0) < 0) {
	serverData -> errorLog.error_sys_quit 
	  	("backgroundConnection: unable to install SIGCHLD handler");
    }

    sigchld_handler (SIGCHLD, 0);

	/* set up a timer to periodically poll for connection death */

    sigalrmFired = e_FALSE;
    (void) setitimer (ITIMER_REAL, &pollTimeout, NULL);

	/* loop until the agent dies, the server fails, the machine  */
	/* blows up, or the Earth crashes into the Sun, handling all */
	/* messages sent from the agent                              */

    while (1) {

 	goToSleep = e_TRUE;

	    /* done if the associated agent interpreter has exited */

	if (sigchldFired == e_TRUE) {
	    removeId = e_TRUE;
	    break;
	}

	    /* see if we need to reload configuration */

	if (hasSighupFired()) {

		/* log a message */

	    char temp[16];
	    sprintf (temp, "%d", (int) getpid());

	    char *argv[] = {
		"handler: INFO: handler process ",
		temp,
		" received the SIGHUP: reloading configuration",
		((char *) NULL)
	    };

	    serverData -> errorLog.error_app_cont (argv);

		/* reload the configuration and resolve the IP addresses */

	    if (serverData -> load (*commandLine, ServerData::e_RELOADING_IN_HANDLER) < 0) {

	        char *argv[] = {
		    "handler: ERROR: unable to reload configuration",
		    ((char *) NULL)
	        };

#ifdef FIX_LATER
		/* this error_app_quit means that it is possible to get out */
		/* of this process without removing the agent temporary     */
		/* directory                                                */
#endif

		serverData -> errorLog.error_app_quit (argv);
		abort_with_message ("should never get here");   // should never get here
	    }

	    serverData -> resolveIp (ServerData::e_RELOADING_IN_HANDLER);
	    goToSleep = e_FALSE;
	}

	    /* see if there is an incoming request from the agent */

	if (hasSigioFired() || sigalrmFired) {

	    if (sigalrmFired) {
		sigalrmFired = e_FALSE;
		(void) setitimer (ITIMER_REAL, &pollTimeout, NULL);
	    }
		
	    FD_ZERO (&readset);
	    FD_SET (handler.sockfd, &readset);
	
	    do {
	        noTimeout.tv_sec  = 0;
	        noTimeout.tv_usec = 0;
		code = select (handler.sockfd + 1, (SELECT_MASK *) &readset, (SELECT_MASK *) NULL, (SELECT_MASK *) NULL, &noTimeout);
	    } while ((code < 0) && ((errno == EINTR) || (errno == EAGAIN)));

	    if (code > 0) {

		MESSAGE *backgroundMessage;

		if ((backgroundMessage = message_receive (handler.sockfd, messagesToServer)) == NULL) {
		    removeId = e_TRUE;
		    break;
		}

		monitor.release ();
		code = handleBackgroundClientRequest (handler.sockfd, backgroundMessage, sender, newHandler);
		monitor.acquire ();

		delete (backgroundMessage);
		backgroundMessage = NULL;

		if ((code > 0) || (newHandler != NULL)) {
		    break;
		}

		pretendSigioFired ();	// force another check -- CRITICAL! 
		goToSleep = e_FALSE;
	    }
	}

	    /* see if there is incoming communication from another agent */

	if (sigusr1Fired == e_TRUE) {

	    response = sendMessage (&message);

	    if ((response == NULL) || (response -> getFlag() == RESP_ERROR)) {
		MESSAGE clientMessage (messagesFromServer.messages[RESP_ERROR]); 
		message_send (handler.sockfd, clientMessage);
		delete (response);
	        response = NULL;
		break;
	    }

	    if (response -> getFlag() != RESP_NONE) {

		message_send (handler.sockfd, *response);

		if (response -> getLong(0) != 0) {
		    sigusr1Fired = e_TRUE;   // force another check -- CRITICAL!
		    goToSleep    = e_FALSE;
		} else {
		    sigusr1Fired = e_FALSE;  // we will get SIGUSR on next item
		}

	    } else {

		sigusr1Fired = e_FALSE;      // we will get SIGUSR on next item 
	    }

	    delete (response);
	    response = NULL;
	}

	    /* wait for a SIGIO or SIGUSR1 */

	if (goToSleep) {
	    monitor.wait ();
	}
    }

	/* wait for the associated agent interpreter to exit and send an */
	/* abnormal exit message if necessary, remove the agent from the */
	/* tables if necessary, and delete the agent-specific directory  */
	/* in the agent temporary directory                              */

    if (newHandler == NULL) {

	if (handler.pid != 0) {

	    while (!sigchldFired) {
	        monitor.wait ();
	    }

	    checkAbnormalExit (handler.rootId, handler.localId, sender, sigchldStatus);
	}

        if (removeId) {
	    remove_id (handler.localId.getId(), sender);
        }

	char temp[16];
	sprintf (temp, "%u", handler.localId.getId());
	DynamicString directory = serverData -> tempDirectory;
	directory.append ("/");
	directory.append (temp);

	if (removeDirectory (directory) < 0) {

	    char *argv[] = {
		"handler: ERROR: unable to remove agent temporary directory \"",
		directory.value(),
		"\"",
		(char *) NULL
	    };

	    serverData -> errorLog.error_app_cont (argv);
	}
    }

	/* clean up */

    return (0);
}
