/* Agent Tcl
   Bob Gray
   9 February 1995

   servReceive.cc

   This file implements the routines that the server uses to RECEIVE a 
   client request.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "mesgTcpip.h"
#include "agentd.h"
#include "genError.h"
#include "suppHash.h"
#include "platInterrupt.h"
#include "genManager.h"
#include "genMessage.h"
#include "genPGP.h"
#include "genServerInterface.h"
#include "genTransmit.h"
#include "genUtility.h"
#include "piped.h"
#include "servConf.h"
#include "servProcess.h"
#include "servReceive.h"
#include "truefalse.h"

typedef int (*Action) (int, BASE_SECURITY &, MESSAGE *, BOOLEAN, NewBackgroundHandler *&);

    /* TCP/IP request jump table */

static Action jumpTable[NUM_REQUESTS] =
{
  &handleSend,		/* REQ_BEGIN	*/
  (Action) NULL,	/* REQ_NAME	*/
  (Action) NULL,	/* REQ_END      */
  &handleSend,		/* REQ_FORCE	*/
  &handleNewAgent,	/* REQ_SUBMIT   */
  &handleSend,		/* REQ_MESSAGE	*/
  &handleSend,		/* REQ_MEETING  */
  &handleSend,		/* REQ_EVENT	*/
  (Action) NULL,	/* REQ_GET	*/
  &handleSend,		/* REQ_INFO	*/
  (Action) NULL,	/* REQ_STATUS   */
  &handleSend,		/* REQ_IP       */
  (Action) NULL,	/* REQ_AUTH     */
  (Action) NULL,	/* REQ_SECURITY */
  (Action) NULL		/* REQ_ERROR    */
};

/* handleClientRequest 

   Purpose: Handle the incoming client request 

     Input: sockfd       = socket descriptor
		           (int)

	    isUnixSocket = e_TRUE if the request is arriving on the Unix socket
			   (BOOLEAN)

    Output: The first byte of the transmission must be one of

	    REQ_BEGIN
            REQ_NAME
            REQ_END
            REQ_SCRIPT
            REQ_STATE
            REQ_MESSAGE
            REQ_MEETING
            REQ_EVENT
            REQ_GET
	    REQ_STATUS
	    REQ_IP

            The procedure reads the transmission from the socket, checks
            the first byte and takes the appropriate action.  The procedure
            returns -1 on error and 0 otherwise.
*/

static void rejectDueToMachine (int sockfd, UINT_32 peerIp)
{
    char temp[16];
    struct in_addr address;

    address.s_addr = peerIp;
    sprintf (temp, "%s", inet_ntoa (address));

    char *argv[] = {
	"socketd: ERROR: attempt to connect from host \"",
	 temp,
	 "\" which is not an AllowedMachine",
	 ((char *) NULL) 
    };

    serverData -> errorLog.error_app_cont (argv);
    respondWithError (sockfd, ServerInterface::e_SERV_ERR_SOURCE);
}

int handleClientRequest (int sockfd, BOOLEAN isUnixSocket)
{
    int rc = 0;
    UINT_32 port;
    UINT_32 peerIp;
    BASE_SECURITY sender;
    MESSAGE *message = NULL;
    NewBackgroundHandler *newHandler = NULL;

	/* remove the SIGHUP and the SIGCHLD handler handler */ 

    if (SignalHandlers::install_signal (SIGHUP, (SignalHandlers::funcType) SIG_DFL, 0) < 0) {
	serverData -> errorLog.error_sys_quit
	    ("servReceive: unable to install default SIGHUP");
    }

    if (SignalHandlers::install_signal (SIGCHLD, (SignalHandlers::funcType) SIG_DFL, 0) < 0) {
	serverData -> errorLog.error_sys_quit
	    ("servReceive: unable to install default SIGCHLD");
    }

	/* get the IP address of the peer */

    static UINT_32 localhostIp = tcpip_getLocalhostIP();

    if (isUnixSocket) {

	peerIp = localhostIp;

    } else if (tcpip_getPeername (sockfd, peerIp, port) != e_TCPIP_OK) {

	char *argv[] = {
	    "servReceive: ERROR: unable to determine identity of connected client",
	    ((char *) NULL) 
	};

	serverData -> errorLog.error_sys_cont (argv);
	respondWithError (sockfd, ServerInterface::e_SERV_ERR_SYSTEM);
	return (-1);

    } else if ((peerIp == localhostIp) || (peerIp == serverData -> host.getIp())) {

	char *argv[] = {
	    "servReceive: ERROR: an apparently local agent is connecting over the TCP/IP socket",
	    ((char *) NULL) 
	};

	serverData -> errorLog.error_sys_cont (argv);
	respondWithError (sockfd, ServerInterface::e_SERV_ERR_SYSTEM);
	return (-1);
    }
 
	/* SECURITY CHECK -- take the appropriate action if the request */
	/* did not come from an AllowedMachine                          */

    BOOLEAN disallowedMachine = e_FALSE;

    if (serverData -> machineTable.lookup (peerIp, NULL) == NULL) {

	    /* if the server accepts a request from any machine as long */
	    /* as it is signed, then just record that we do not have an */
	    /* AllowedMachine, since we will not know if the message is */
	    /* signed until later; otherwise deny the request           */

	if (serverData -> allowSigned) {

	    disallowedMachine = e_TRUE;

	} else {

	    rejectDueToMachine (sockfd, peerIp);
	    return (-1);
	}
    }

	/* receive the incoming message */

    rc = secureReceive (sockfd, messagesToServer, serverData -> security, sender, message);

    if (rc != MESSAGE_OK) {

	if ((rc == MESSAGE_ENCRYPT) || (rc == MESSAGE_DECRYPT)) {

	    char *argv[] = {
		"servReceive: ERROR: invalid or interrupted message: decryption error:",
		PGP_error().value(),
	        ((char *) NULL)
	    };

	    serverData -> errorLog.error_app_cont (argv);

	} else {

	    char *argv[] = {
	        "servReceive: ERROR: invalid or interrupted message",
	        ((char *) NULL)
	    };

	    serverData -> errorLog.error_app_cont (argv);
	}

	respondWithError (sockfd, ServerInterface::e_SERV_ERR_UNKNOWN);
	return (-1);
    }

	/* SECURITY CHECK -- only certain requests are allowed on the */
	/* foreground connection                                      */

    UINT_8 flag = message -> getFlag ();

    if (jumpTable[flag] == NULL) {

	char *argv[] = {
	    "servReceive: ERROR: invalid request \"",
	    requestFlagToString (message -> getFlag()),
	    "\" on foreground connection",
	    ((char *) NULL)
	};

	serverData -> errorLog.error_app_cont (argv);
	respondWithError (sockfd, ServerInterface::e_SERV_ERR_INVALID);
	delete (message);
	return (-1);
    } 

	/* SYSTEM AND SECURITY CHECK -- it should be impossible for the */
	/* sending agent itself to be marked as authenticated           */

    if (sender.isAgentAuthenticated()) {
	
	char *argv[] = {
	    "servReceive: ERROR: sending agent is incorrectly marked as authenticated for a foreground message",
	    ((char *) NULL),
	};

	serverData -> errorLog.error_app_cont (argv);
	respondWithError (sockfd, ServerInterface::e_SERV_ERR_SYSTEM);
	delete (message);
	return (-1);
    } 

	/* SECURITY CHECK -- if (1) the sending machine is authenticated   */
	/* and (2) the sending machine is a TrustedMachine and (3) the     */
	/* sending machine claims that it able to authenticate the agent's */
	/* owner, then mark the agent's owner as authenticated             */

    int count  = 0;
    DynamicString machineKeyname = DynamicString (sender.getMachineKeyname().value());

    if (!(machineKeyname.isEmpty())) {

	for (char *sp = machineKeyname.value(); *sp != '\0'; sp++) { 

	    if ((*sp == ' ') && (++count == 4)) {

		*sp = '\0';

		if (serverData -> trustedTable.lookup (0, machineKeyname.value()) != NULL) {

		    if (sender.isMachineAuthenticated() && sender.isOwnerClaimed()) {
		        sender.setOwnerAuthenticated (e_TRUE);
		    }

		    sender.setMachineTrusted (e_TRUE);
		    break;
		}
	    }
	}
    }

	/* SECURITY CHECK -- if (1) the owner of the sending agent has been */
	/* authenticated but (2) the owner is not one of the AllowedUser's, */
	/* mark the agent as anonymous -- this test is needed since the set */
	/* of AllowedUser's and the set of keys in the PGP database are not */
	/* necessarily the same                                             */

    if (sender.isOwnerAuthenticated()) {

	DynamicString ownerKeyname = sender.getOwnerKeyname();

	if (serverData -> userTable.lookup (0, ownerKeyname.value()) == NULL) {
	    sender.setOwnerAuthenticated (e_FALSE);
	} 
    }

	/* SECURITY CHECK -- if (1) the request did -not- come from an   */
	/* AllowedMachine and (2) was not signed by an AllowedUser, then */
	/* we have an error                                              */

    if (disallowedMachine && !sender.isOwnerAuthenticated()) {
	rejectDueToMachine (sockfd, peerIp);
	return (-1);
    }

	/* SECURITY CHECKS -- command-specific security checks */

    switch (message -> getFlag()) {

	case REQ_BEGIN:
	{
		/* reject the request if the agent is on a remote machine */
		/* and if we do not allow remote agents to register with  */
		/* the local server                                       */
                        
	    if (!(serverData -> remoteBegin) && (peerIp != localhostIp)) {
		respondWithError (sockfd, ServerInterface::e_SERV_ERR_REMOTE_BEGIN);
		delete (message);
		return (-1);
	    }

		/* reject the request if agent is anonymous and is on a   */
		/* remote machine (since we never allow remote anonymous  */
		/* agents to register with the local server)              */

	    if ((peerIp != localhostIp) && (serverData -> encryption)) {
		if (!sender.isOwnerAuthenticated()) {
		    respondWithError (sockfd, ServerInterface::e_SERV_ERR_ANON_REMOTE_BEGIN);
		    delete (message);
		    return (-1);
		}
	    }

		/* make sure that a remote agent is not lying about its */
		/* actual location                                       */

	    if (peerIp != localhostIp) {

	        char *actualName = message -> getString (0);
	        UINT_32 actualIp = message -> getIp (1);

		if ((localhostIp == actualIp) ||
		    (serverData -> host.getIp() == actualIp) ||
		    (serverData -> host.getName() == actualName)) 
		{
		    respondWithError (sockfd, ServerInterface::e_SERV_ERR_LIAR);
		    delete (actualName);
		    delete (message);
		    return (-1);
		}

		message -> addString (0, actualName, e_TRUE);
		message -> addIp (1, actualIp);
	    } 

	    break;
	}

	case REQ_INFO:
	{
	    if (!(serverData -> remoteInfo) && (peerIp != localhostIp)) {
		respondWithError (sockfd, ServerInterface::e_SERV_ERR_REMOTE_INFO);
		delete (message);
		return (-1);
	    }

	    if ((peerIp != localhostIp) && (serverData -> encryption)) {
		if (!sender.isOwnerAuthenticated()) {
		    respondWithError (sockfd, ServerInterface::e_SERV_ERR_ANON_REMOTE_INFO);
		    delete (message);
		    return (-1);
		}
	    }

	    break;
	}

	case REQ_SUBMIT:
	{
		/* make sure that a remote agent is not lying about its id */

	    if (peerIp != localhostIp) {

	 	AgentId *agentId = message -> getId (3);

		if ((agentId -> getServerIp() == serverData -> host.getIp()) ||
		    (agentId -> getServerName() == serverData -> host.getName()))
		{
		    respondWithError (sockfd, ServerInterface::e_SERV_ERR_LIAR);
		    delete (agentId);
		    delete (message);
		    return (-1);
		}

		message -> addId (3, agentId, e_TRUE);
	    }

	    break;
	}

	case REQ_EVENT  :
	case REQ_MESSAGE:
	case REQ_MEETING:
	{
	    if (!(serverData -> remoteComm) && (peerIp != localhostIp)) {
		respondWithError (sockfd, ServerInterface::e_SERV_ERR_REMOTE_COMM);
		delete (message);
		return (-1);
	    }

	    break;
	}
    }

	/* take action */
 
    rc = (*jumpTable[message -> getFlag()]) (sockfd, sender, message, e_FALSE, newHandler); 
    delete (message);

	/* become a background handler if necessary */

    if (newHandler != NULL) {

	while (newHandler != NULL) {

	    NewBackgroundHandler *nextHandler;

		// close the old socket

	    if (sockfd != newHandler -> sockfd) {
		close (sockfd);
		sockfd = newHandler -> sockfd;
	    }

		// become a background connection

	    rc = backgroundConnection (sender, *newHandler, nextHandler);

		// cleanup and go around again

	    delete (newHandler);
	    newHandler = nextHandler;
	}

	close (sockfd);
	exit (0);
    }

	/* return only if we do not become a background handler */

    return (rc);
}

    /* background request jump table */

static Action backgroundJumpTable[NUM_REQUESTS] =
{
  (Action) NULL,	/* REQ_BEGIN	*/
  &handleSend,		/* REQ_NAME	*/
  &handleSend,		/* REQ_END	*/
  &handleSend,		/* REQ_FORCE	*/
  &handleNewAgent,	/* REQ_SUBMIT   */
  &handleSend,		/* REQ_MESSAGE	*/
  &handleSend,		/* REQ_MEETING  */
  &handleSend,		/* REQ_EVENT	*/
  (Action) NULL,	/* REQ_GET	*/
  &handleSend,		/* REQ_INFO     */
  &handleSend,		/* REQ_STATUS   */
  (Action) NULL,	/* REQ_IP       */
  (Action) NULL,	/* REQ_AUTH     */
  (Action) NULL,	/* REQ_SECURITY */
  (Action) NULL
};

/* handleBackgroundClientRequest 

   Purpose: Handle a client request that is arriving on the background
	    connection

     Input: sockfd = socket descriptor
		     (int)

	    message = the incoming request
		      (class MESSAGE *)

	    sender = security information for the sender
		     (const class BASE_SECURITY &)

    Output: The first byte of the transmission must be one of

	    REQ_BEGIN
            REQ_NAME
            REQ_END
            REQ_SCRIPT
            REQ_STATE
            REQ_MESSAGE
            REQ_MEETING
            REQ_EVENT
            REQ_GET
	    REQ_STATUS
	    REQ_IP

            The procedure reads the transmission from the socket, checks
            the first byte and takes the appropriate action.  The procedure
            returns -1 on error, 0 on a success that should not cause the
	    background handler to terminate, and 1 on a success that should
	    cause the background handler to terminate (i.e. an AGENT_END).

	    If the request was REQ_SUBMIT, REQ_JUMP or REQ_FORK, the lower-
	    level handler routines will do a fork so that there is a new 
	    background handler for the new agent.  In this case 
	    handleBackgroundClientRequest will essentially return twice, once 
	    with newHandler set to NULL and once with newHandler set to a 
	    dynamically allocated NewBackgroundHandler structure that contains 
	    the information needed to finish setting up the new background 
	    handler.  The reason for this somewhat complex setup is that we
	    do not want to increase the stack depth every time that we
	    submit an agent to the same machine; if we do continually increase
	    the stack depth, we might eventually blow the stack.
*/

int handleBackgroundClientRequest (int sockfd, MESSAGE *message, BASE_SECURITY &sender, NewBackgroundHandler *&newHandler)
{
    int rc;

	/* SECURITY CHECK -- only certain requests are allowed on the */
	/* background connection                                      */

    UINT_8 flag = message -> getFlag ();

    if (backgroundJumpTable[flag] == NULL) {

	char *argv[] = {
	    "servReceive: ERROR: invalid request \"",
	    requestFlagToString (message -> getFlag()),
	    "\" on background connection",
	    ((char *) NULL)
	};

	serverData -> errorLog.error_app_cont (argv);
	respondWithError (sockfd, ServerInterface::e_SERV_ERR_INVALID);
	return (-1);
    } 

	/* SYSTEM AND SECURITY CHECK -- it should be impossible for the */
	/* sending agent itself to be marked as unauthenticated         */

    if (!sender.isAgentAuthenticated()) {
	
	char *argv[] = {
	    "servReceive: ERROR: sending agent is incorrectly marked as unauthenticated for a background message",
	    ((char *) NULL),
	};

	serverData -> errorLog.error_app_cont (argv);
	respondWithError (sockfd, ServerInterface::e_SERV_ERR_SYSTEM);
	return (-1);
    }

	/* take action */ 

    rc = (*backgroundJumpTable[flag]) (sockfd, sender, message, e_TRUE, newHandler);
    return (rc);
}
