/* Agent Tcl
   Bob Gray
   9 February 1995

   socketd.cc

   This file implements procedure "socketd" which monitors the TCP/IP port.

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "platExclusion.h"
#include "platInterrupt.h"
#include "platSystem.h"
#include "mesgTcpip.h"
#include "agentd.h"
#include "genError.h"
#include "genFile.h"		   // FileUtility
#include "genServerInterface.h"    // ServerErrorcodes
#include "genTransmit.h"
#include "servConf.h"
#include "servProcess.h"
#include "servReceive.h"
#include "socketd.h"
#include "truefalse.h"

/* sigchld_handler, incrementChildCount and decrementChildCount

   Purpose: These three procedures keep track of socketd's total number of 
	    children.  socketd calls incrementChildCount directly right before 
	    a fork.  socketd calls decrementChildCount directly if the fork 
	    fails.  sigchld_handler is called automatically when a SIGCHLD 
	    interrupt is sent to socketd.  

     Notes: We do not block the SIGCHLD interrupt in incrementChildCount and
	    decrementChildCount since they are only called when SIGCHLD is
 	    already blocked.  SIGCHLD is blocked automatically in 
	    sigchld_handler.
*/

const int c_REASON_LENGTH = 256;
 
static void sigchld_handler (int, SignalHandlers::ClientData)
{
    int oldErrno;
    WAIT_STATUS_TYPE status;  

	/* save the old errno */

    oldErrno = errno;

	/* reap the zombie (log an error message if it was a abnoraml exit) */

    while (waitpid(0, (int *) &status, WNOHANG) > 0) {

	serverData -> socketdChildren -= 1;

	if (WIFEXITED(status) && (WEXITSTATUS(status) == 0)) {
	    continue;
	}

        char *failureCore = "";
	char failureReason[c_REASON_LENGTH];

	if (WIFEXITED(status)) {

	    int exitCode = WEXITSTATUS(status);
	    sprintf (failureReason, "; handler exited with exit code %d", exitCode);

	} else if (WIFSIGNALED(status)) {

	    int signo = WTERMSIG(status);
	    const char *signoString = SignalHandlers::SignoToString (signo);
	    sprintf (failureReason, "; handler was killed with %s signal (signo %d)", 
	        signoString, signo);

#ifdef WCOREDUMP
	    if (WCOREDUMP(status)) {
	        failureCore = "; core dump performed";
	    }
#endif

	} else if (WIFSTOPPED(status)) {

	    int signo = WSTOPSIG(status);
	    const char *signoString = SignalHandlers::SignoToString (signo);
	    sprintf (failureReason, "; handler was stopped with %s signal (signo %d)", 
	        signoString, signo);

	} else {

	    sprintf (failureReason, "; handler stopped for undetermined reason");
	}

	char *argv[] = {
	    "socketd: ERROR: handler failed",
	    failureReason,
	    failureCore,
	    ((char *) NULL)
	};

	serverData -> errorLog.error_app_cont (argv);
    }

	/* restore the old errno */

    errno = oldErrno;
    return;
}

inline static void incrementChildCount (void)
{
    serverData -> socketdChildren += 1;
}

inline static void decrementChildCount (void)
{
    serverData -> socketdChildren -= 1;
}

/* socketd

   Purpose: This procedure monitors the TCP/IP and Unix domain sockets for
	    incoming requests.
*/

static void acceptConnection (MonitorLock &monitor, int sockfd, BOOLEAN isUnixSocket)
{
    int newSockfd;
    pid_t childpid;

	/* get the connection */

    if (tcpip_blockingAccept (sockfd, newSockfd) != e_TCPIP_OK) {
	return;
    }

	/* reject the connection if we have reached maximum load */

    if (serverData -> maxAgents + serverData -> extraProcesses <= serverData -> socketdChildren) {
	MESSAGE message (messagesFromServer.messages[RESP_ERROR]);
	message.addByte (0, ServerInterface::e_SERV_ERR_LOAD);
	(void) message_send (newSockfd, message);
	close (newSockfd);
	return;
    }

	/* fork the child */

    incrementChildCount ();

    if ((childpid = fork()) < 0) {
 
	    /* FORK FAILED -- so send an error message to the client */

	MESSAGE message (messagesFromServer.messages[RESP_ERROR]);
	message.addByte (0, ServerInterface::e_SERV_ERR_LOAD);
	(void) message_send (newSockfd, message);
	close (newSockfd);

	    /* decrement the child count */

	decrementChildCount();

	    /* and log an error message */
	
	serverData -> errorLog.error_sys_cont
	    ("socketd: ERROR: unable to fork child process");

    } else if (childpid == 0) {

	    /* CHILD PROCESS */

	    /* open the lock file -- must do this here in order to avoid the */
	    /* weirdness that occurs when locks are passed across a fork     */

	do {
	    serverData -> lockfd = FileUtility::open (serverData -> lockFilename.value(), O_RDWR | O_TRUNC | O_CREAT, 0600);
	} while ((serverData -> lockfd < 0) && ((errno == EINTR) || (errno == EAGAIN)));

	if (serverData -> lockfd < 0) {
	    serverData -> errorLog.error_sys_quit 
	 	("socketd: ERROR: unable to open lock file");
    	}

	    /* close unnecessary files */ 

	close (serverData -> unixSockfd);
	close (serverData -> sockfd);

	    /* handle the request */

	monitor.release ();
	(void) handleClientRequest (newSockfd, isUnixSocket);
	close (newSockfd);
	exit (0);

    } else {

	    /* PARENT PROCESS */

	if (isSighupPending()) {	// make sure we do not lose a SIGHUP
	   kill (childpid, SIGPROF);	// that arrives during the fork
	}
 
	close (newSockfd);
    }
}

void socketd (void)
{
    int rc;
    int maxSockfd;
    struct timeval timeout;
    fd_mask readSet[MASK_SIZE];

	/* install the SIGCHLD, SIGUSR1, SIGUSR2 and SIGALRM interrupts */

    if (SignalHandlers::install_signal_intr (SIGCHLD, sigchld_handler, 0) < 0) {
	serverData -> errorLog.error_sys_quit 
	  	("socketd: unable to install SIGCHLD handler");
    }
 
    if (SignalHandlers::install_signal_intr (SIGUSR1, sigusr1_handler, 0) < 0) {
	serverData -> errorLog.error_sys_quit
		("socketd: unable to install SIGUSR1 handler");
    }

    if (SignalHandlers::install_signal_intr (SIGUSR2, sigusr2_handler, 0) < 0) {
	serverData -> errorLog.error_sys_quit
		("socketd: unable to install SIGUSR2 handler");
    }
 
    if (SignalHandlers::install_signal_intr (SIGALRM, sigalrm_handler, 0) < 0) {
	serverData -> errorLog.error_sys_quit
		("socketd: unable to install SIGALRM handler");
    }

	/* identify the maximum socket number */

    if (serverData -> sockfd > serverData -> unixSockfd) {
	maxSockfd = serverData -> sockfd; 
    } else {
	maxSockfd = serverData -> unixSockfd;
    }

	/* wait for connections from a client process */

    while (1) {

	    /* NOTE: Do not use SIGIO here! We can not use SIGIO since for   */
	    /* certain operating systems such as Linux SIGIO appears to be   */
	    /* generated at the "wrong" point when dealing with sockets      */
	    /* that are being used to accept connections -- i..e, the SIGIO  */
	    /* is generated BEFORE the socket is in a state where "select"   */
	    /* will return READ-READY.  Thus, if we just do a nonblocking    */
	    /* select in response to a SIGIO, like we do everywhere else,    */
	    /* we will randomly miss connection requests.                    */
	    /*                                                               */
	    /* This SIGIO problem is yet more proof that computers are evil. */
	    /* If daily contact with evil bothers you, you should switch     */
	    /* professions.                                                  */

	FDMASK_ZERO (readSet);
	FDMASK_SET (serverData -> sockfd, readSet);
	FDMASK_SET (serverData -> unixSockfd, readSet);

	timeout.tv_sec  = 200000;
	timeout.tv_usec = 0;

	rc = select (maxSockfd + 1, (SELECT_MASK *) &readSet[0], NULL, NULL, &timeout);

	MonitorLock monitor;
	Guard guard (monitor);

        if (rc > 0) {

	    if (FDMASK_ISSET (serverData -> sockfd, readSet)) {
		acceptConnection (monitor, serverData -> sockfd, e_FALSE);
	    }

	    if (FDMASK_ISSET (serverData -> unixSockfd, readSet)) {
		acceptConnection (monitor, serverData -> unixSockfd, e_TRUE);
	    }
	}

	    /* see if SIGHUP has fired */

	if (hasSighupFired()) {

		/* log a message */

	    char *argv[] = {
		"socketd: INFO: received the SIGHUP: reloading configuration",
		((char *) NULL)
	    };

	    serverData -> errorLog.error_app_cont (argv);

		/* reload the configuration and resolve the IP addresses */

#ifdef FIX_LATER
		/* signals are blocked during a potentially lengthy */
		/* reload process                                   */
#endif

	    if (serverData -> load (*commandLine, ServerData::e_RELOADING_IN_SOCKETD) < 0) {

	        char *argv[] = {
		    "socketd: ERROR: unable to reload configuration",
		    ((char *) NULL)
	        };

		serverData -> errorLog.error_app_quit (argv);
		abort_with_message ("should never get here!");   // should never get here
	    }

	    serverData -> resolveIp (ServerData::e_RELOADING_IN_SOCKETD);
	}
    } 
}
