/* Agent Tcl
   Bob Gray
   16 November 1996

   suppDString.cc

   This file implements a simple dynamic buffer class.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "platDelete.h"
#include "platExclusion.h"
#include "suppDBuffer.h"   

// global empty buffer representation that is shared by all dynamic buffers

static DynamicBufferRef *nullBufferRef = 0; 

// get a null buffer ref

class NullBufferInitializer: public OnceTrap
{
public:

    void initialize (void) {
	nullBufferRef = new DynamicBufferRef (0, (char *) NULL, BufferOwnership::e_COPY);
	nullBufferRef -> GrabWithoutLock();
    }
};

DynamicBufferRef *DynamicBuffer::getNullBufferRef (void)
{
    static MonitorOnce::OnceControl onceControl = MonitorOnce::e_NEVER;

    if (onceControl != MonitorOnce::e_DONE) {
	NullBufferInitializer initializer;
	MonitorOnce::doOnce (onceControl, initializer);
    }

    return (nullBufferRef);
}

// constructors -- deep copy

DynamicBufferRef::DynamicBufferRef (UINT_32 length, const char *buffer)
{
    if (length == 0) {
	m_size = 0;
	m_buffer = NULL;
    } else {
	m_size = length;
	m_buffer = new char [length];
        memcpy (m_buffer, buffer, length);
    }

    m_static    = e_FALSE;
    m_available = 0;  
}

DynamicBufferRef::DynamicBufferRef (UINT_32 length, char *buffer, BufferOwnership::Ownership ownership)
{
    if ((ownership == BufferOwnership::e_TAKE) || (ownership == BufferOwnership::e_USE)) {
        m_size = length;
	m_buffer = buffer;
    } else if (length == 0) {
	m_size = 0;
	m_buffer = NULL;
    } else {
	m_size = length;
	m_buffer = new char [length];
        memcpy (m_buffer, buffer, length);
    }

    m_static    = (ownership == BufferOwnership::e_USE) ? e_TRUE : e_FALSE;
    m_available = 0;  
}

// copy constructor -- deep copy

DynamicBufferRef::DynamicBufferRef (const DynamicBufferRef& buffer)
{
    m_size = buffer.m_size;
    m_available = buffer.m_available;
    m_buffer = new char [m_size]; 
    memcpy (m_buffer, buffer.m_buffer, m_size);
    m_static = e_FALSE;
}

// destructor

DynamicBufferRef::~DynamicBufferRef ()
{
    if (m_static == e_FALSE) {
	DELETE_ARRAY_OBJECT(m_buffer);
    }
}

// expand the buffer to hold a certain size

void DynamicBufferRef::expand (UINT_32 size)
{
    if (size > m_available) {
	char *newBuffer = new char [m_size + size];
	memcpy (newBuffer, m_buffer, m_size);
	DELETE_ARRAY_OBJECT(m_buffer);
	m_buffer = newBuffer;
	m_available += size;
	m_size = m_size + size;
    } 
}

// append to the buffer

void DynamicBufferRef::append (UINT_32 length, const char *buffer)
{
	/* nothing to do if there is no new data */

    if (length == 0) {
	return;
    }

	/* otherwise add the new data */

    if (length > m_available) {

	UINT_32 newSize = (m_size << 1) + length;
	char *newBuffer = new char [newSize];
	memcpy (newBuffer, m_buffer, m_size);
	memcpy (newBuffer + m_size - m_available, buffer, length);
	DELETE_ARRAY_OBJECT(m_buffer);
	m_buffer = newBuffer;
	m_available += m_size;
	m_size = newSize;

    } else {

	memcpy (m_buffer + m_size - m_available, buffer, length);
	m_available -= length;
    }
}
