/* Agent Tcl
   Bob Gray
   16 November 1996

   suppDBuffer.h

   This file defines a dynamic, copy-on-write buffer class.
*/

#ifndef _SUPP_DBUFFER_H
#define _SUPP_DBUFFER_H

#ifndef NO_PRAGMAS
#pragma interface
#endif

#ifndef _PLAT_PORTING_H
#include "platPorting.h"		// NULL
#endif
#ifndef _PLAT_SIZES_H
#include "platSizes.h"			// UINT_32
#endif
#ifndef _SUPP_DOWNER_H
#include "suppDOwner.h"			// BufferOwnership
#endif
#ifndef _SUPP_REFERENCE_COUNT_H
#include "suppReferenceCount.h"		// ReferenceCount
#endif
#ifndef _TRUEFALSE_H
#include "truefalse.h"			// BOOLEAN
#endif

class DynamicBufferRef: public ReferenceCount
{
	// constructors -- deep copy

    DynamicBufferRef 
	(UINT_32 length, const char *buffer);

    DynamicBufferRef 
	(UINT_32 length, char *buffer, BufferOwnership::Ownership ownership);

	// copy constructor -- deep copy

    DynamicBufferRef 
	(const DynamicBufferRef& buffer);

	// append to the buffer

    void append 
	(UINT_32 length, const char *buffer);

	// expand the buffer so that it is big enough for "size" additional bytes */

    void expand 
	(UINT_32 size);

	// destructor

   ~DynamicBufferRef ();

    char *m_buffer;	   // buffer 
    char *m_ptr;           // starting location within the buffer
    UINT_32 m_size;        // size of the buffer
    UINT_32 m_available;   // available space inside the buffer
    BOOLEAN m_static;      // buffer is static

    friend class DynamicBuffer;
    friend class NullBufferInitializer;
};

class DynamicBuffer
{
    private:

	DynamicBufferRef *m_buffer;	// reference-counted buffer

	    // null buffer reference

	DynamicBufferRef *getNullBufferRef (void);

    protected:

	    // clone the buffer -- this involves a deep copy

	void clone (void) {		
	    DynamicBufferRef *temp = new DynamicBufferRef (*m_buffer);
	    temp -> GrabWithoutLock();	  
	    m_buffer -> Release();
	    m_buffer = temp;
	}	    

	    // copy on write

	void copyOnWrite (void) {
	    if ((m_buffer -> checkReferenceCount() > 1) || (m_buffer -> m_static)) {
		clone ();
	    }
	}

    public:

	    // constructor -- make a NULL buffer

	DynamicBuffer () {
	    m_buffer = getNullBufferRef();
	    m_buffer -> Grab();
	}

	    // constructors -- deep copy 

	DynamicBuffer (UINT_32 length, const char *buffer) {

	    if (length == 0) {
		m_buffer = getNullBufferRef();
	    } else {
		m_buffer = new DynamicBufferRef (length, buffer);
	    }

	    m_buffer -> Grab();
	}

	DynamicBuffer (UINT_32 length, char *buffer, BufferOwnership::Ownership ownership) {

	    if (length == 0) {
		m_buffer = getNullBufferRef();
	    } else {
		m_buffer = new DynamicBufferRef (length, buffer, ownership);
	    }

	    m_buffer -> Grab();
	}

	    // constructor -- shallow copy due to later copy-on-write

	DynamicBuffer (const DynamicBuffer &buffer) {
	    m_buffer = buffer.m_buffer;
	    m_buffer -> Grab();
	}

	    // assignment operator -- shallow copy due to later copy-on-write

	DynamicBuffer& operator= (const DynamicBuffer& buffer) {

	    if ((this == &buffer) || (m_buffer == buffer.m_buffer)) {
		return (*this);
	    }

	    buffer.m_buffer -> Grab();
	    m_buffer -> Release();
	    m_buffer = buffer.m_buffer;
	    return (*this);
	}

	    // destructor

	~DynamicBuffer () {
	    m_buffer -> Release();
	}

	    // empty out the buffer

	DynamicBuffer& empty (void) {
	    DynamicBufferRef *temp = getNullBufferRef();
	    temp -> Grab();
	    m_buffer -> Release();
	    m_buffer = temp;
	    return (*this);
	}

	    // append to the buffer
	
	DynamicBuffer& append (UINT_32 length, const char *buffer) {

	    if (length != 0) {
		copyOnWrite ();
		m_buffer -> append (length, buffer);
	    }

	    return (*this);
	}

	    // expand the buffer so that it is big enough for "size" additional bytes */

	void expand (UINT_32 size) {
	    m_buffer -> expand (size);
	}

	    // return the length of the buffer

	UINT_32 size (void) const {
	    return (m_buffer -> m_size - m_buffer -> m_available);
	}

	    // is the buffer empty?
	
	BOOLEAN isEmpty (void) const {
	    return ((size() > 0) ? e_FALSE : e_TRUE);
	}

	    // return the buffer itself -- this function can be misused!

	char *value (void) const {
	    return (m_buffer -> m_buffer);
	}
};

#endif
