/* Agent Tcl
   Bob Gray
   16 November 1996

   suppDString.cc

   This file implements a simple dynamic string class.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "platDelete.h"
#include "platExclusion.h"
#include "suppDString.h"

// global empty buffer representation that is shared by all dynamic buffers

static DynamicStringRef *nullStringRef = 0; 

// get a null string ref

class NullStringInitializer: public OnceTrap
{
public:

    void initialize (void) {
	nullStringRef = new DynamicStringRef ("");
	nullStringRef -> GrabWithoutLock();
    }
};

DynamicStringRef *DynamicString::getNullStringRef (void)
{
    static MonitorOnce::OnceControl onceControl = MonitorOnce::e_NEVER;

    if (onceControl != MonitorOnce::e_DONE) {
	NullStringInitializer initializer;
	MonitorOnce::doOnce (onceControl, initializer);
    }

    return (nullStringRef);
}

// constructors -- deep copy

DynamicStringRef::DynamicStringRef (const char *string)
{
    assert (string != NULL);
    m_size = strlen(string) + 1;
    m_buffer = new char [m_size];
    strcpy (m_buffer, string);
    m_available = 0;  
    m_static = e_FALSE;
}

DynamicStringRef::DynamicStringRef (char *string, BufferOwnership::Ownership ownership)
{
    assert ((string != NULL) && (*string != '\0'));

    if ((ownership == BufferOwnership::e_TAKE) || (ownership == BufferOwnership::e_USE)) {
	m_size = strlen(string) + 1;
	m_buffer = string;
    } else {
	m_size = strlen(string) + 1;
	m_buffer = new char [m_size];
	strcpy (m_buffer, string);
    }

    if (ownership == BufferOwnership::e_USE) {
	m_static = e_TRUE;
    } else {
	m_static = e_FALSE;
    }

    m_available = 0;
}

// copy constructor -- deep copy

DynamicStringRef::DynamicStringRef (const DynamicStringRef& string)
{
    m_size = string.m_size;
    m_available = string.m_available;
    m_buffer = new char [string.m_size]; 
    memcpy (m_buffer, string.m_buffer, string.m_size);
    m_static = e_FALSE; 
}

// destructor

DynamicStringRef::~DynamicStringRef ()
{
    if (m_static == e_FALSE) {
	DELETE_ARRAY_OBJECT(m_buffer);
    }
}

// append to the string

void DynamicStringRef::append (const char *string)
{
    DynamicStringRef::append (string, strlen(string));
}

void DynamicStringRef::append (const DynamicString& string)
{
    DynamicStringRef::append (string.value(), string.length());
}

void DynamicStringRef::append (const char *string, UINT_32 length) 
{
    if (length == 0) {
	return;
    }

    if (length > m_available) {

	UINT_32 newSize = (m_size << 1) + length;
	char *newBuffer = new char [newSize];
	char *appendLoc = newBuffer + (m_size - m_available - 1);
	memcpy (newBuffer, m_buffer, m_size);
	memcpy (appendLoc, string, length);
	*(appendLoc + length) = '\0';
	DELETE_ARRAY_OBJECT(m_buffer);
	m_buffer = newBuffer;
	m_available += m_size;
	m_size = newSize;

    } else {

	char *appendLoc = m_buffer + (m_size - m_available - 1);
	strncpy (appendLoc, string, length);
	*(appendLoc + length) = '\0';
	m_available -= length;
    }
}

// comparison operators

int operator== (const DynamicString& s, const DynamicString& t) 
{
    if (!strcmp(s.m_string -> m_buffer, t.m_string -> m_buffer)) {
	return (1);
    }

    return (0);
}

int operator== (const char *s, const DynamicString& t)
{
    if (s == NULL) {

	if (t.isEmpty()) {
	    return (1);
	}

	return (0);
    }

    if (!strcmp(s, t.m_string -> m_buffer)) {
	return (1);
    }

    return (0);
}

// lowercase

BOOLEAN DynamicStringRef::hasLowercaseLetters (void)
{
    for (char *sp = m_buffer; *sp != '\0'; ++sp) {
	if ((*sp >= 'A') && (*sp <= 'Z')) {
	    return (e_TRUE);
	}
    }

    return (e_FALSE);
}

void DynamicStringRef::convertToLowercase (void)
{
    for (char *sp = m_buffer; *sp != '\0'; ++sp) {
	if ((*sp >= 'A') && (*sp <= 'Z')) {
	    *sp = *sp - 'A' + 'a';
	}
    }
}
