/* Agent Tcl
   Bob Gray
   16 November 1996

   suppDString.h

   This file defines a dynamic, copy-on-write string class.
*/

#ifndef _SUPP_DSTRING_H
#define _SUPP_DSTRING_H

#ifndef NO_PRAGMAS
#pragma interface
#endif

#ifndef _PLAT_PORTING_H
#include "platPorting.h"	   // NULL
#endif
#ifndef _PLAT_DELETE_H
#include "platDelete.h"		// DELETE_OBJECT and DELETE_ARRAY_OBJECT
#endif
#ifndef _PLAT_SIZES_H
#include "platSizes.h"		   // UINT_32
#endif
#ifndef _SUPP_DOWNER_H
#include "suppDOwner.h"		   // BufferOwnership
#endif
#ifndef _SUPP_REFERENCE_COUNT
#include "suppReferenceCount.h"    // ReferenceCount
#endif
#ifndef _TRUEFALSE_H
#include "truefalse.h"		   // BOOLEAN
#endif

    /* forward declarations */

class DynamicString;

    /* dynamic strings */

class DynamicStringRef: public ReferenceCount
{
    friend class DynamicString;
    friend class NullStringInitializer;
    friend int operator== (const DynamicString&, const DynamicString&);
    friend int operator== (const char *, const DynamicString&);

	// constructors -- deep copy

    DynamicStringRef 
	(const char *string);

    DynamicStringRef 
	(char *string, BufferOwnership::Ownership ownership);

	// copy constructor -- deep copy

    DynamicStringRef 
	(const DynamicStringRef& string);

	// append to the string

    void append 
	(const char *string);

    void append
	(const DynamicString& string);

    void append
	(const char *string, UINT_32 length);

	// convert to lowercase or check for lowercase letters

    void convertToLowercase
	(void);

    BOOLEAN hasLowercaseLetters
	(void);

	// destructor

   ~DynamicStringRef ();

	// member data

    char *m_buffer;	   // string buffer
    UINT_32 m_size;        // size of the buffer
    UINT_32 m_available;   // available space inside the buffer
    BOOLEAN m_static;      // e_TRUE if buffer is static
};

class DynamicString
{
	// operators	

    friend int operator== (const DynamicString&, const DynamicString&);
    friend int operator== (const char *, const DynamicString&);

    friend int operator!= (const DynamicString& s, const DynamicString& t) {
	return (!(s == t));
    }

    friend int operator!= (const char *s, const DynamicString& t) {
	return (!(s == t));
    }

    friend int operator!= (const DynamicString& s, const char * t) {
	return (!(t == s));
    }

    friend int operator== (const DynamicString& s, const char *t) {
	return (t == s);
    }

private:

    DynamicStringRef *m_string;

	// null string reference

    DynamicStringRef *getNullStringRef (void);

protected:

            // clone the string -- this involves a deep copy

        void clone (void) {		
	    DynamicStringRef *temp = new DynamicStringRef (*m_string);
	    temp -> GrabWithoutLock();
	    m_string -> Release();
	    m_string = temp;
	}	    

	    // copy on write

	void copyOnWrite (void) {
	    if ((m_string -> checkReferenceCount() > 1) ||
		(m_string -> m_static)) {
		clone ();
	    }
	}

    public:

	    // constructor -- make a NULL string

	DynamicString (void) {
	    m_string = getNullStringRef();
	    m_string -> Grab();
	}

	    // constructors -- deep copy 

	DynamicString (const char *string) {

	    if ((string == NULL) || (*string == 0)) {
		m_string = getNullStringRef();
		m_string -> Grab();
	    } else {
		m_string = new DynamicStringRef (string);
		m_string -> GrabWithoutLock();
	    }
	}

	DynamicString (char *string, BufferOwnership::Ownership ownership) {

	    if ((string == NULL) || (*string == 0)) {

		m_string = getNullStringRef();
		m_string -> Grab();

		if (ownership == BufferOwnership::e_TAKE) {
		    DELETE_ARRAY_OBJECT(string);
		}
	
	    } else {  

		m_string = new DynamicStringRef (string, ownership);
		m_string -> GrabWithoutLock();
 	    }
	}

	    // copy constructor -- shallow copy due to later copy-on-write

	DynamicString (const DynamicString &string) {
	    m_string = string.m_string;
	    m_string -> Grab();
	}

	    // assignment operator -- shallow copy due to later copy-on-write

	DynamicString& operator= (const DynamicString& string) {

	    if ((this == &string) || (m_string == string.m_string)) {
		return (*this);
	    }

	    string.m_string -> Grab();
	    m_string -> Release();
	    m_string = string.m_string;
	    return (*this);
	}

	    // destructor

	~DynamicString () {
	    m_string -> Release();
	}

	    // empty out the string

	DynamicString& empty (void) {
	    DynamicStringRef *temp = getNullStringRef();
	    temp -> Grab();
	    m_string -> Release();
	    m_string = temp;
	    return (*this);
	}

	    // append to the string
	
	DynamicString& append (const char *string) {

	    if ((string != NULL) && (*string != '\0')) {
		copyOnWrite ();
		m_string -> append (string);
	    }

	    return (*this);
	}

	DynamicString& append (const DynamicString& string) {

	    if (!string.isEmpty()) {
		copyOnWrite ();
		m_string -> append (string);
	    }

	    return (*this);
	}

	DynamicString& append (const char *string, UINT_32 length) {

	    if (length != 0) {
		copyOnWrite ();
		m_string -> append (string, length);
	    }

	    return (*this);
	}

	    // convert to lowercase

	DynamicString& convertToLowercase (void) {

	    if (m_string -> hasLowercaseLetters()) {
		copyOnWrite ();
		m_string -> convertToLowercase ();
	    }

	    return (*this);
	}

	    // return the length of the string

	UINT_32 length (void) const {
	    return (m_string->m_size - m_string->m_available - 1);
	}

	    // is the string empty?
	
	BOOLEAN isEmpty (void) const {
	    return ((length() > 0) ? e_FALSE : e_TRUE);
	}

	    // return the string itself -- this function can be misused!

	char *value (void) const {
	    return (m_string -> m_buffer);
	}
};

#endif
