/* Bob Gray
   Agent Tcl
   8 May 1998

   suppFileTable.cc

   This file implements a class that keeps track of all open file descriptors
   (assuming that the files are opened via our functions).
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "platDelete.h"
#include "platExclusion.h"
#include "suppFileTable.h"

    /* an entry in the file descriptor table */

struct FileTableNode {

    int                  m_fd;          // file descriptor
    FileTable::FileTypes m_type;	// file type
    FileTableNode *m_next;		// next entry in the table row

    FileTableNode (int fd, FileTable::FileTypes type): 
	m_fd (fd),
	m_type (type),
	m_next ((FileTableNode *) NULL)
    {
	// empty
    }
};

    /* number of file descriptors in each row of the table */

const int c_ENTRIES_PER_ROW 
	= 16;

/* ----------------------------------------------------------------------------
 * STATIC DATA
 * ----------------------------------------------------------------------------
 */

    /* number of rows in the table */

static int s_tableRows 
	= 4;

    /* table of file descriptors -- This is essentially a hash table */
    /* where each file descriptor fd is mapped into the table row    */
    /* with index (fd / c_ENTRIES_PER_ROW).                          */

static FileTableNode **s_table 
	= NULL;

#ifdef MULTIPLE_THREADS

    /* lock for mutual exclusion */

static MutexLock *s_lock
	= NULL;

#endif

/* ----------------------------------------------------------------------------
 * FileTable: static data
 * ----------------------------------------------------------------------------
 */

BOOLEAN FileTable::m_record
	= e_FALSE;

/* ----------------------------------------------------------------------------
 * FileTable: methods
 * ----------------------------------------------------------------------------
 */

#ifdef FIX_LATER
    /*
     * We are using relatively heavyweight synchronization.  It might be
     * worthwhile to make the synchronization faster, although in the current 
     * multi-threaded server, we are admittedly opening and duping very few 
     * file descriptors, so it is far from critical (since each file
     * descriptor is used for many agents).
     */
#endif

/* FIleTable::lockTable

   Purpose: Lock the table in preparation for a fork.
*/

void FileTable::lockTable (void)
{
#ifdef MULTIPLE_THREADS
    s_lock->acquire();
#endif
}

/* FIleTable::unlockTable

   Purpose: Unlock the table after a fork
*/

void FileTable::unlockTable (void)
{
#ifdef MULTIPLE_THREADS
    s_lock->release();
#endif
}
 
/* FileTable::add

   Purpose: Add a (now open) file descriptor to the table

     Input: fd   = file descriptor
		   (int)

            type = type of file descriptor
                   (FileTable::FileTypes)

    Output: None
*/

class TableInitializer: public OnceTrap
{
public:

    void initialize (void) {

	s_table = new FileTableNode * [s_tableRows];

	for (int i = 0; i < s_tableRows; ++i) {
	    s_table[i] = (FileTableNode *) NULL;
	}

#ifdef MULTIPLE_THREADS
	s_lock = new MutexLock;
#endif
    }
};

void FileTable::add (int fd, FileTable::FileTypes type)
{
	// nothing to do if the process is not recording open file descriptors

    if (!m_record) {
	return;
    }

	// initialize the static data structures if necessary

    static MonitorOnce::OnceControl onceControl = MonitorOnce::e_NEVER;

    if (onceControl != MonitorOnce::e_DONE) {
	TableInitializer tableInitializer;
	MonitorOnce::doOnce (onceControl, tableInitializer);
    }

	// mutual exclusion

#ifdef MULTIPLE_THREADS
    MutexGuard guard (*s_lock);
#endif

	// calculate the row in which the new file descriptor belongs

    int row = fd / c_ENTRIES_PER_ROW;
   
	// expand the table if we do not have enough rows
 
    if (row >= s_tableRows) {

	int newRows = s_tableRows;

	while (newRows <= row) {
	    newRows = newRows << 1;
	}

	FileTableNode **newTable = new FileTableNode * [newRows];

	for (int i = 0; i < s_tableRows; ++i) {
	    newTable[i] = s_table[i];
	}

	for (int j = s_tableRows; j < newRows; ++j) {
	    newTable[j] = NULL;
	}

	DELETE_ARRAY_OBJECT(s_table);
	s_table     = newTable;
	s_tableRows = newRows;
    }

	// error if the file descriptor is already in the row

    FileTableNode *node = s_table[row];

    while (node != NULL) {

	if (node -> m_fd == fd) {
	    printf ("duplicate fd: %d\n", fd);
	    abort_with_message ("FileTable::add: attempting to add descriptor that was already added");
	}

	node = node -> m_next;
    }

	// add the file descriptor

    node = new FileTableNode (fd, type);
    node -> m_next = s_table[row];
    s_table[row] = node;
}

/* FileTable::dup

   Purpose: Duplicate an existing file descriptor (e.g., copy the properties
            of a file descriptor that is already in the table to a new fd that
	    we add to the table)

     Input: existingFd = existing file descriptor
		         (int)

            newFd      = new file descriptor
                        (FileTable::FileTypes)

    Output: None
*/

void FileTable::dup (int existingFd, int newFd)
{
    int row;
    FileTableNode *node;
    FileTable::FileTypes type;

	// nothing to do if the process is not recording open file descriptors

    if (!m_record) {
	return;
    }

	// acquire the lock

#ifdef MULTIPLE_THREADS
    s_lock -> acquire();
#endif
    
	// lookup up the old file descriptor

    row  = existingFd / c_ENTRIES_PER_ROW;
    node = NULL;

    if (row < s_tableRows) {

	node = s_table[row];

	while (node != NULL) {
	
	    if (node -> m_fd == existingFd) {
		break;
	    }

	    node = node -> m_next;
	}
    }

	// if we did not find the old file descriptor, we have an error

    if (node == NULL) {
	abort_with_message ("FileTable::dup: attempting to dup descriptor that is not in the table");
    }

    type = node -> m_type;
	
	// release the lock -- add will re-acquire it

#ifdef MULTIPLE_THREADS
    s_lock -> acquire();
#endif

	// otherwise we add the new file descriptor with the same type
	// as the existing file descriptor

    FileTable::add (newFd, type);
}

/* FileTable::remove

   Purpose: Remove a (now closed) file descriptor from the table

     Input: fd   = file descriptor
		   (int)

    Output: None
*/

void FileTable::remove (int fd)
{
	// mutual exclusion

#ifdef MULTIPLE_THREADS
    MutexGuard guard (*s_lock);
#endif
    
	// identify the table row and proceed only if it is a valid row

    int row = fd / c_ENTRIES_PER_ROW;

    if (row < s_tableRows) {

	    // find the file descriptor in the row

	FileTableNode *node = s_table[row];
	FileTableNode *prevNode = NULL;

	while (node != NULL) {
	
	    if (node -> m_fd == fd) {
		break;
	    }

	    prevNode = node;
	    node     = node -> m_next;
	}

	    // if we found the file descriptor, remove it

	if (node != NULL) {

	    if (prevNode == NULL) {
		s_table[row] = node -> m_next;
            } else {
		prevNode -> m_next = node -> m_next; 
	    }

	    DELETE_OBJECT(node);
	}
    }
}

/* FileTable::closeAllExcept

   Purpose: Close all open file descriptors (except for one distinguished
            file descriptor)

     Input: fd = the distinguished file descriptor
	         (int)

    Output: The method closes all open file descriptors (except for the
            distinguished file descriptor).
*/

void FileTable::closeAllExcept (int fd)
{
	// mutual exclusion

#ifdef MULTIPLE_THREADS
    MutexGuard guard (*s_lock);
#endif

	// close the file descriptors

    for (int i = 0; i < s_tableRows; ++i) {

	FileTableNode *node = s_table[i];

	while (node != NULL) {	

	    if (node -> m_fd != fd) {
	 	close (node -> m_fd);
	    }

	    node = node -> m_next;
	}
    }
}

/* FileTable::closeAllExcept

   Purpose: Close all open file descriptors (except for some number of
	    distinguished file descriptors)

     Input: fdCount = number of distinguished file descriptors
		      (int)

	    fds     = the distinguished file descriptors
		      (int *)

    Output: The method closes all open file descriptors (except for the
            distinguished file descriptor).
*/

void FileTable::closeAllExcept (int fdCount, int *fds)
{
	// mutual exclusion

#ifdef MULTIPLE_THREADS
    MutexGuard guard (*s_lock);
#endif

	// close the file descriptors

    for (int i = 0; i < s_tableRows; ++i) {

	FileTableNode *node = s_table[i];

	while (node != NULL) {	

	    BOOLEAN found = e_FALSE;

	    for (int j = 0; j < fdCount; ++j) {
		if (node -> m_fd == fds[j]) {
		    found = e_TRUE;
		    break;
		}
	    }

	    if (!found) {
	 	close (node -> m_fd);
	    }

	    node = node -> m_next;
	}
    }
}
/* FileTable::printDebuggingInformation

   Purpose: Dump the file table contents to standard output (for debugging
	    purposes)

     Input: None

    Output: None
*/

void FileTable::printDebuggingInformation (void)
{
	// mutual exclusion

#ifdef MULTIPLE_THREADS
    MutexGuard guard (*s_lock);
#endif

	// print the file-descriptor table

    printf ("File descriptor table:\n");
    printf ("\n");
    printf ("Fd   Type\n");
    printf ("==== =============\n");

    for (int i = 0; i < s_tableRows; ++i) {

	FileTableNode *node = s_table[i];

	while (node != NULL) {

	    char *typeString;

	    switch (node -> m_type) {
		case e_FILE	     : typeString = "file"; break;
		case e_TCPIP_SOCKET  : typeString = "TCP/IP socket"; break;
	   	case e_UNIX_SOCKET   : typeString = "UNIX socket"; break;
	  	case e_PIPE	     : typeString = "pipe"; break;
		case e_STANDARD_IN   : typeString = "stdin"; break;
		case e_STANDARD_OUT  : typeString = "stdout"; break;
		case e_STANDARD_ERROR: typeString = "stderr"; break;
		default		     : typeString = "unknown";
	    }

	    printf ("%5d %s\n", node -> m_fd, typeString);
	    node = node -> m_next;
	}
    }
}
