/* Bob Gray
   Agent Tcl
   8 May 1998

   suppFileTable.h

   This file defines a class that keeps track of all open file descriptors
   (assuming that the files are opened via our functions). 

   In the current multi-threaded server, the server forks and execs the agent
   interpreters.  This class allows the child process to close all open 
   (but unnecessary) file descriptors before execing the agent interpreter.

   NOTE that there is actually a "race condition" in this class since client 
   code that actually opens a file does not report the file descriptor until 
   after the open has succeeded.  If the server forks a child process just 
   after some thread has successfully opened a file but before that thread has 
   recorded the file descriptor, the child process will not close the
   descriptor, and the execed interpreter will contain an unnecessary, open
   descriptor.  HOWEVER for now, we do not care about this race condition.  The
   multi-threaded server can have hundreds of open file descriptors.  Our goal
   is simply to close "most of them", so that the interpreter has plenty of free
   (low-numbered) file descriptors for use.  If we change our mind later, and
   want to fix this race condition, client code would need to mark some flag
   inside this class before attempting the open.
*/

#ifndef _SUPP_FILE_TABLE_H
#define _SUPP_FILE_TABLE_H

#ifndef NO_PRAGMAS
#pragma interface
#endif

#ifndef _TRUEFALSE_H
#include "truefalse.h"		// BOOLEAN
#endif

class FileTable 
{
public:

	/* file descriptor types */

    enum FileTypes {
	e_FILE	         = 0x00,
	e_TCPIP_SOCKET   = 0x01,
	e_UNIX_SOCKET    = 0x02,
	e_PIPE	         = 0x03,
	e_STANDARD_IN    = 0x04,
	e_STANDARD_OUT   = 0x05,
	e_STANDARD_ERROR = 0x06
    };

private:

	/* e_TRUE if the process wants to record all file descriptors */

    static BOOLEAN m_record;

public:

	/* see if file descriptor recording is on -- The idea is that */
	/* client routines such as FileUtility::open will use this    */
	/* method to avoid unecessary calls to "add" and "remove.     */

    static BOOLEAN isFileRecordingOn (void) {
	return (m_record);
    }

	/* turn file descriptor recording on or off */

    static void turnFileRecordingOn (void) {
	m_record = e_TRUE;
    }

    static void turnFileRecordingOff (void) {
	m_record = e_FALSE;
    }

	/* We expose the locking since we need to make sure that the    */
	/* lock is in a consistent state across a fork call (since the  */
	/* purpose of this class is to close all unnecessary file       */
	/* descriptors after a fork).                                   */

#ifdef FIX_LATER
	/* It's questionable whether we should expose the locking in    */
	/* this way.  It perhaps would be better to provide an AtFork   */
	/* class with which client could register routines to be called */
	/* before and after a fork.                                     */
#endif

    static void lockTable (void);
    static void unlockTable (void);

	/* add a (now open) file descriptor to the table */

    static void add (int fd, FileTable::FileTypes type);

	/* add a (now open) file descriptor to the table, */
	/* where this file descriptor has the same type   */
	/* as an existing file descriptor                 */

    static void dup (int existingFd, int newFd);

	/* remove a (now closed) file descriptor from the table */

    static void remove (int fd);

	/* close all open file descriptors (except for one distinguished */
	/* file -- pass in -1 if there is no distinguished file)         */ 

    static void closeAllExcept (int fd);

	/* close all open file descriptors (except for some number of    */
	/* distinguished files -- pass in 0 for the count if there are   */
	/* no distignuished files)                                       */

    static void closeAllExcept (int fdCount, int *fds);

	/* print file descriptor information to stderr (for debugging)   */

    static void printDebuggingInformation (void);
};

#endif
