/* Agent Tcl
   Bob Gray
   5 April 1995

   hash.cc

   This file implements the hash table classes.   

   Copyright (c) 1995, Robert S. Gray Dartmouth College

   See the file "agent.terms" for information on usage and redistribution
   of this file and for a DISCLAIMER OF ALL WARRANTIES.
*/

#ifndef NO_PRAGMAS
#pragma implementation
#endif

#include "platPorting.h"
#include "platDelete.h"
#include "suppStrings.h"
#include "suppHash.h"
#include "truefalse.h"

/* HashSearch::HashSearch

   Purpose: These procedures are the constructor for class HashSearch.
*/

HashSearch::HashSearch (Hash *p_hash)
{
	/* assertions on the parameters */

    assert (p_hash != NULL);

	/* initialize */

    HashSearch::hash   = p_hash;
    HashSearch::bucket = 0;
    HashSearch::node   = p_hash -> table[0];
}

HashSearch::HashSearch (const HashSearch &hashSearch)
{
    hash   = hashSearch.hash;
    bucket = hashSearch.bucket;
    node   = hashSearch.node;
}

/* HashSearch::operator=

   Purpose: This procedure is the assignment operator for class HashSearch.
*/

HashSearch &HashSearch::operator= (const HashSearch& hashSearch)
{
    hash   = hashSearch.hash;
    bucket = hashSearch.bucket;
    node   = hashSearch.node;
    return (*this);
}

/* HashSearch::next

   Purpose: Get the next HashNode in the hash table

     Input: None

    Output: The procedure returns NULL if there are no more entries in the
            hash table.  Otherwise the procedure returns a pointer to the next
            node in the hash table.  This node should NOT be deleted by the
            caller.
*/

HashNode *HashSearch::next (void)
{ 
  HashNode *nextNode;

    /* is there another node in the same bucket */

  if (node != NULL) {
    nextNode = node;
    node = node -> next;
    return (nextNode);
  }

    /* no so move to the next nonempty bucket */

  while (bucket < hash -> max_index) {

      /* next bucket */

    bucket += 1;

      /* is there a node in that bucket */

    if ((node = hash -> table[bucket]) != NULL) {
      nextNode = node;
      node = node -> next;
      return (nextNode);
    }
  }

    /* there are no more nonempty buckets */

  return (NULL);
}

/* HashNode::HashNode

   Purpose: This procedure is the constructor for class HashNode.

     Input: key        = the integer key
            string_key = the string key
            data       = the entry data
            next       = next node in a list
*/

HashNode::HashNode (long p_key, const char *p_string_key, void *p_data, HashNode *p_next)
{
  HashNode::key        = p_key;
  HashNode::data       = p_data;
  HashNode::next       = p_next;
  HashNode::string_key = strcpyWithAlloc (p_string_key);
}

/* HashNode::~HashNode

   Purpose: This procedure is the destructor for class HashNode.
*/

HashNode::~HashNode ()
{
    DELETE_ARRAY_OBJECT(string_key);
} 

/* Hash::Hash

   Purpose: This procedure is the constructor for class Hash.

     Input: buckets = number of buckets in the hash table

            buckets must be a power of two since the hash function is
            just a bit-wise AND.
*/

Hash::Hash (unsigned buckets)
{
	/* check for power of two */

	/* initialize */

    Hash::max_index = buckets - 1;
    Hash::entries   = 0;
    Hash::table     = new HashNode * [buckets];

    for (unsigned i = 0; i <= max_index; i++) {
	table[i] = (HashNode *) NULL;
    }
}

/* Hash::empty

   Purpose: This procedure empties out the hash table.

     Input: None

    Output: The procedure empties out the hash table.

      Note: The procedure does not delete the hash data.  This must be done
	    using the HashSearch class before deleting the hash table
	    itself.
*/

void Hash::empty (void)
{
    register HashNode *current;
    register HashNode *next;

	/* delete the table entries */

    for (unsigned i = 0; i <= max_index; i++) {

	for (current = table[i]; current != NULL; current = next) {
	    next = current -> next;
	    DELETE_OBJECT(current);
        } 

	table[i] = NULL;
    }
}

/* Hash::~Hash

   Purpose: This procedure is the destructor for class Hash.
*/

Hash::~Hash ()
{
    empty ();
    DELETE_ARRAY_OBJECT(table);
}

/* Hash::add

   Purpose; Add an entry to the hash table

     Input: key        = entry integer key
            string_key = entry string key 
            data       = entry data

    Output: If the entry is in the hash table, the procedure returns a
            pointer to HashNode structure and sets "there" to TRUE.
            If the entry is not in the hash table, the procedure adds the
            entry to the table, returns a pointer to the new HashNode
            structure and sets "there" to FALSE.
*/

HashNode *Hash::add (long key, const char *string_key, void *data, int &there)
{
  HashNode *current;

    /* check if the entry is in the table */

  long index = hash (key, string_key);

  for (current = table[index]; current != NULL; current = current -> next) {
    if (current -> key == key) {
      if ((string_key == NULL) || (!strcmp (current -> string_key, string_key))) {
        there = TRUE;
        return current;
      }
    }
  }

    /* add a new entry to the table */

  current      = new HashNode (key, string_key, data, table[index]);
  table[index] = current;
  entries     += 1;
  there        = FALSE;
    
     /* return the new entry */
 
  return current;
}

/* Hash::lookup

   Purpose: Lookup an entry in the hash table

     Input: key        = entry integer key
            string_key = entry string key
 
    Output: The procedure returns NULL if the entry is not in the hash table.
            Otherwise the procedure returns a pointer to the HashNode
            structure.
*/

HashNode *Hash::lookup (long key, const char *string_key) const
{
  register HashNode *current;

    /* check if the entry is in the table */

  long index = hash(key, string_key);

  for (current = table[index]; current != NULL; current = current -> next)
    if (current -> key == key)
      if ((string_key == NULL) || (!strcmp (current -> string_key, string_key)))
        return current;

   return NULL;
}

/* Hash::remove

   Purpose: Remove an entry in the hash table

     Input: key        = entry integer key
            string key = entry string key
 
    Output: The procedure returns FALSE if the entry is not in the hash
            table.  Otherwise the procedure removes the entry and returns
            TRUE.
*/

int Hash::remove (long key, const char *string_key)
{
  register HashNode *current;
  register HashNode *previous;

     /* check if the entry is in the table */

 long index = hash(key, string_key);

  for (previous = NULL, current = table[index]; current != NULL; previous = current, current = current -> next)
    if (current -> key == key)
      if ((string_key == NULL) || (!strcmp(current -> string_key, string_key))) {
        if (previous == NULL) {
          table[index] = current -> next;
        } else {
           previous -> next = current -> next;
        }

        DELETE_OBJECT(current);
        entries -= 1;
        return TRUE;
      }

  return FALSE;
} 
